;;;  -*- Package: rhet-user; Mode: Lisp; Syntax: Rhet; Base: 10. -*-

(in-package rhet-user)
;;;	File:		Tempos-Rhet.lisp
;;;	Purpose:	axioms etc. to be read into Rhet
;;;	Last Edit:	4/06/89 11:39:30
;;;
;;;	Copyright (c) 1988, 1989 Johannes A. G. M. Koomen
;;;	All rights reserved.
;;;
;;;	The TEMPOS system provides a temporal reasoning facility for the logic
;;;	programming system RHETORICAL.  It supplies hooks into the TimeLogic
;;;	package, and defines primitives such as TT (True Throughout) etc.

(EVAL-WHEN (LOAD EVAL)
  (LET* ((*READTABLE* UI:*RHET-LISP-READTABLE*)
	 (*PACKAGE* (FIND-PACKAGE "RHET-USER")))
    (apply #'assert-axioms
	   '(

	     ;; Equals is special:  if Rhet thinks two (presumably interval) terms are
	     ;; logically equal, then they are temporally equal too, by defn.

; Not sure how to handle this yet, so punt on it...
;
;  [[Time-Equals ?i*T-Time ?j*T-Time]
;   <Tempos [Add-EQ ?i*T-Time ?j*T-Time] :forward]
;
;  [[Add-EQ ?i*T-Time ?j*T-Time]
;   <Tempos [Time-Equals ?i ?j]
;           [Distinct ?i ?j]
;	   :forward [Time-Equals ?i ?j]]


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	     ;; (Part of) Shoham's taxonomy				;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	     ;; If something is LIQUID, then it's both UPWARD and DOWNWARD

	     [[Time-Upward ?x]
	      <Tempos [Time-Liquid ?x]]

	     [[Time-Downward ?x]
	      <Tempos [Time-Liquid ?x]]

	     ;; If something is UPWARD then it's also CONCATENABLE

	     [[Time-Concatenable ?x]
	      <Tempos [Time-Upward ?x]]


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	     ;; The MT (Maximally True) predicate				;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	     ;; X is maximally true over interval I if X is true over I and every interval
	     ;; J over which X is true is either disjoint from or contained in I

	     ;; We expect the user to assert MT.

	     ;; In order to prevent infinite looping, some axioms refer specifically to
	     ;; asserted recurrence, rather than inferred recurrence.  A forward-chaining
	     ;; axiom allows this to be proven.

	     [[MT-Asserted ?i*T-Time ?x]
	      <Tempos [MT ?i*T-Time ?x] :forward]

	     ;; Enforce the semantics of MT:  two intervals over which something is
	     ;; maximally true must be either equal or disjoint

	     [[Time-Reln ?i*T-Time (:E :Dis) ?j*T-Time]
	      <Tempos [MT ?i*T-Time ?x]
		      [MT ?j*T-Time ?x]
		      [:Prove [DISTINCT ?i*T-Time ?j*T-Time]]
		      :forward [MT ?i*T-Time ?x]]

	     ;; Enforce mutual temporal exclusion:  two intervals over which two
	     ;; mutually exclusive things are maximally true must be disjoint

	     [[Time-Mutex ?x ?y]
	      <Tempos [Time-Mutex ?y ?x]
		      :forward]

	     [[Time-Disjoint ?i*T-Time ?j*T-Time]
	      <Tempos [MT ?i*T-Time ?x]
		      [Time-Mutex ?x ?y]
		      [MT ?j*T-Time ?y]
		      :forward [MT ?i*T-Time ?x]]


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	     ;; The TT (True Throughout) predicate				;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	     ;; If TT is explicitly asserted then assert the existance of the 
	     ;; corresponding MT 

	     [[AND [MT ?k*T-Time ?x]
		   [Time-SubInt ?i*T-Time ?k]]
	      <Tempos [TT ?i*T-Time ?x]
		      [Time-Skolem ?k*T-Time]
		      :forward [TT ?i*T-Time ?x]]

	     ;;	1-  X is true over interval I if X is maximally true over I.

	     [[TT ?i*T-Time ?x]
	      <Tempos [MT ?i*T-Time ?x]]

	     ;;	2- X is true over interval I if X is downward hereditary and maximally
	     ;;	true over some interval J containing I

	     [[TT ?i*T-Time ?x]
	      <Tempos [Time-Downward ?x]
		      [Time-Subint! ?i*T-Time ?j*T-Time]
		      [MT ?j*T-Time ?x]]

	     ;;	3- X is true over interval I if X is upward hereditary and true over all
	     ;;	intervals J during I

; The following is incorrect, as all *known* subints of I may not properly cover I
;  [[TT ?i*T-Time ?x]
;   <Tempos [Time-Upward ?x]
;	   [FORALL ?j*T-Time [Time-Subint! ?j*T-Time ?i*T-Time]
;		      [TT ?j*T-Time ?x]]]

	     ;;	4- X is true over interval I if X is concatenable and there are intervals P and Q
	     ;;	such that I is sequentially composed of P and Q and X is true over P and over Q

	     [[TT ?i*T-Time ?x]
	      <Tempos [Time-Concatenable ?x]
		      [Time-Cover! ?p*T-Time ?q*T-Time ?i*T-Time]
		      [TT ?p*T-Time ?x]
		      [TT ?q*T-Time ?x]]


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	     ;; The FT (False Throughout) predicate			;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	     ;; X is false over interval I iff every interval J over which X is true is
	     ;; disjoint from I

	     ;; User asserts FT.

	     ;; However, enforce the semantics of FT:

	     [[Time-Disjoint ?i*T-Time ?j*T-Time]
	      <Tempos [MT ?i*T-Time ?x]
		      [FT ?j*T-Time ?x]
		      :forward]

	     [[NOTEQ? ?X ?Y] <Tempos [MT ?I*T-TIME ?X] [FT ?J*T-TIME ?Y] [TIME-INTERSECTS ?I ?J]
				     [UNIFY ?X [?PRED &REST ?ARGS-X]]
				     [UNIFY ?Y [?PRED &REST ?ARGS-Y]]	;same "predicate"
				     :FORWARD [MT ?I*T-TIME ?X] [FT ?J*T-TIME ?Y]]
				

	     ;; Falsehood is by definition Downward, i.e., if X is false over I then it is
	     ;; false over each subinterval of I

	     [[FT ?i*T-Time ?x]
	      <Tempos [Time-Subint! ?i*T-Time ?j*T-Time]
		      [FT ?j*T-Time ?x]]


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	     ;; Reasoning about Sequences					;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	     ;; Sequences are objects (events, properties, whatever) that have two
	     ;; constituents. The predicate [MT ?k [SEQ ?x ?y]] indicates that ?x and ?y
	     ;; occurred sequentially over ?k.
	     ;; Of course, either ?x or ?y may itself be a sequence!

	     ;; Prove the occurrence of a sequence by proving the occurrence of its
	     ;; constituents over appropriate subintervals

;  [[MT ?k*T-Time [SEQ ?x ?y]]
;   <Tempos [Time-Meets ?i*T-Time ?j*T-Time]
;	   [MT ?i*T-Time ?x]
;	   [MT ?j*T-Time ?y]
;	   [POST [Time-Cover! ?i*T-Time ?j*T-Time ?k*T-Time]]
;	   [MT-Verify ?k*T-Time [SEQ ?x ?y]]]

	     ;; The occurrence of X over I can be inferred from the occurrence of a
	     ;; sequence of X and Y over K started by I.  Similarly for Y over J.
	     ;; Note that forcing the verification of the companion (i.e., Y in case we're
	     ;; looking for X, and vice versa) forces the appropriate relative time
	     ;; constraints between I and J to be checked.  Use a COND in lieu of CUT
	     ;; since the latter doesn't work right (yet).

	     [[MT ?i*T-Time ?x]
	      <Tempos [NOT [EQ? ?x [SEQ ?v ?w]]]	; no runaway recursion on SEQ!
		      [MT ?k*T-Time [SEQ ?x ?y]]
		      [POST [Time-Starts ?i*T-Time ?k*T-Time]]
		      [COND ([MT-Verify ?i*T-Time ?x]
			     [POST [Time-Cover! ?i*T-Time ?j*T-Time ?k*T-Time]]
			     [MT-Verify ?j*T-Time ?y])]]

	     [[MT ?j*T-Time ?y]
	      <Tempos [NOT [EQ? ?y [SEQ ?v ?w]]]	; no runaway recursion on SEQ!
		      [MT ?k*T-Time [SEQ ?x ?y]]
		      [POST [Time-Finishes ?j*T-Time ?k*T-Time]]
		      [COND ([MT-Verify ?j*T-Time ?y]
			     [POST [Time-Cover! ?i*T-Time ?j*T-Time ?k*T-Time]]
			     [MT-Verify ?i*T-Time ?x])]]


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	     ;; Reasoning about Recurrence					;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	     ;; Recurrence of an (event, property, whatever) is indicated with the RT
	     ;; predicate: [RT ?i ?x] means that ?x repeatedly occurs over interval ?i.
	     ;; (For now, `repeatedly' means `infinitely often.'  Need to introduce the
	     ;; notion of duration to do any better than that...)

	     ;; User asserts RT facts, from which the existence of individual occurrences
	     ;; as well as the recurrence over tails.

	     ;; In order to prevent infinite looping, some axioms refer specifically to
	     ;; asserted recurrence, rather than inferred recurrence.  A forward-chaining
	     ;; axiom allows this to be proven.

	     [[RT-Asserted ?i*T-Time ?x]
	      <Tempos [RT ?i*T-Time ?x] :forward]

	     ;; X occurs over I if X recurs over J and I starts or is equal to J

	     [[MT ?i*T-Time ?x]
	      <Tempos [RT ?j*T-Time ?x]
		      [POST [Time-Reln ?i*T-Time (:S) ?j*T-Time]]	; should be (:S :E)
		      [MT-Verify ?i*T-Time ?x]]

	     ;; X recurs over J if there is an occurrence of X over I meeting J and there
	     ;; is an recurrence of X over K containing I and finished by J.

	     [[RT ?j*T-Time ?x]
	      <Tempos [RT-Asserted ?k*T-Time ?x]
		      [MT-Asserted ?i*T-Time ?x]
		      [Time-Reln ?i*T-Time (:S :D) ?k*T-Time]
		      [POST [Time-Meets ?i*T-Time ?j*T-Time]]
		      [POST [Time-Finishes ?j*T-Time ?k*T-Time]]
		      [RT-Verify ?j*T-Time ?x]]

	     ;; A sequence recurs over K if either component occurs over I
	     ;; meeting K if the sequence recurs over M containing I.

	     [[RT ?k*T-Time [SEQ ?x ?y]]
	      <Tempos [RT-Asserted ?m*T-Time [SEQ ?x ?y]]
		      [MT-Asserted ?i*T-Time ?x]
		      [Time-Reln ?i*T-Time (:S :D) ?m*T-Time]
		      [POST [Time-Before ?i*T-Time ?k*T-Time]]
		      [POST [Time-Finishes ?k*T-Time ?m*T-Time]]
		      [RT-Verify ?k*T-Time [SEQ ?x ?y]]]

	     [[RT ?k*T-Time [SEQ ?x ?y]]
	      <Tempos [RT-Asserted ?m*T-Time [SEQ ?x ?y]]
		      [MT-Asserted ?j*T-Time ?y]
		      [Time-Reln ?j*T-Time (:F :D) ?m*T-Time]
		      [POST [Time-Meets ?j*T-Time ?k*T-Time]]
		      [POST [Time-Finishes ?k*T-Time ?m*T-Time]]
		      [RT-Verify ?k*T-Time [SEQ ?x ?y]]]


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
	     ;; Time Verification						;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

	     ;; Many proofs succeed by posting temporal constraints.  In order to put flesh
	     ;; on the claims, a special verification predicate succeeds if the given time
	     ;; arg is bound (in which case the postings presumably were checked already),
	     ;; or if skolemizing the time arg is successful (which causes the posted
	     ;; constraints to be added to the TimeLogic DB, verifying the constraints).
	     ;; Note that in addition to the posted constraints attached to the time var,
	     ;; MT-Verify also ensures that the time-var is consistent with all asserted MT
	     ;; facts, i.e. equal to or disjoint from the other MT intervals.  In order to
	     ;; do this we add MT-Asserted facts to the KB whenever MT is asserted.  It's
	     ;; incomplete :- but all we can hope to do here.  Can't do FORALL using full
	     ;; proofs on MT since we're already doing this inside an MT proof!

	     [[Time-Verify ?i*T-Time]
	      <Tempos [OR [BOUND ?i*T-Time]
			  [Time-Skolem ?i*T-Time]]]

	     [[MT-Verify-Unique ?i*T-Time ?x]
	      <Tempos [FORALL ?j*T-Time
			      [MT-Asserted ?j*T-Time ?x]
			      [POST [Time-Reln ?i*T-Time (:E :Dis) ?j*T-Time]]]]

	     [[MT-Verify-Mutex ?i*T-Time ?x]
	      <Tempos [FORALL (?j*T-Time ?y*T-U)
			      [AND [Time-Mutex ?x ?y]
				   [MT-Asserted ?j*T-Time ?y]]
			      [POST [Time-Disjoint ?i*T-Time ?j*T-Time]]]]

	     [[MT-Verify ?i*T-Time ?x]
	      <Tempos [COND ([MT-Verify-Unique ?i ?x]
			     [COND ([MT-Verify-Mutex ?i*T-Time ?x]
				    [Time-Verify ?i*T-Time])])]]

	     [[RT-Verify ?i*T-Time ?x]
	      <Tempos [COND ([FORALL ?j*T-Time
				     [RT-Asserted ?j*T-Time ?x]
				     [POST [Time-Reln ?i*T-Time (:E :F :Fi :Dis) ?j*T-Time]]]
			     [Time-Verify ?i*T-Time])]]

	     ))))

;;; End of file Tempos-Rhet
