;;;  -*- Package: Tempos; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10.; Lowercase: Yes -*-

;;;	File:		Tempos-Main.lisp
;;;	Purpose:	general interface
;;;	Last Edit:	3/22/89 14:16:41
;;;
;;;	Copyright (c) 1988, 1989  Johannes A. G. M. Koomen
;;;	All rights reserved.
;;;
;;;	The TEMPOS system provides a temporal reasoning facility for the logic
;;;	programming system RHETORICAL.  It supplies hooks into the TimeLogic
;;;	package, and defines primitives such as TT (True Throughout) etc.

(in-package "TEMPOS")

(eval-when (compile load eval)
  (when (find-package "RHET-USER")
    (use-package "TEMPOS" (find-package "RHET-USER"))))

;;; (Reset-Tempos) subsumes (Reset-Rhetorical).
;;; Look at the file "tempos:tempos;tempos-rhet.lisp" to see
;;; what axioms are currently defined.

;;; (Trace-Tempos (&optional verbose-p)) can be called to enable Tempos tracing.
;;; Call (Untrace-Tempos) to disable it again.


;;; DEFINING INTERVALS

(defun define-time (&rest time-terms)
  (let* ((term nil)
	 (done nil)
	 (todo time-terms)
	 (refint
	   (do ((termptr time-terms (cdr termptr)))
	       ((null termptr) nil)
	     (when (and (cdr termptr)
			(null (cddr termptr))
			(eq (car termptr) :reference))
	       (push (cadr termptr) done)
	       (return (or (convert-term-to-interval (cadr termptr) t)
			  (error "*** Not a Rhet function term: ~S"
				 (cadr termptr))))))))
    (loop
      (unless todo
	(apply 'add-utype 't-time done)	; 8/8/90 bwm
	(return (nreverse done)))
      (setf term (pop todo))
      (cond ((and (eq term :reference) (null (cdr todo)))
	     (setf todo nil))
	    ((not (convert-term-to-interval term t refint))
	     (error "*** Not a Rhet function term: ~S" term))
	    (t (push term done)))))
)

;;; Put an appropriate define-time hook on Rhet, so when ground rhet terms are consed we know about them. (this courtesy bwm)
(eval-when (load eval)
  (pushnew 'define-rhet-term-time
	   rhet-terms:*create-individual-hooks*))

(defun define-rhet-term-time (individual-form itype)
  (unless (null *time-itype*)	;we must not be using TEMPOS! 
    (if (and (rhet-terms:typeq itype *time-itype*)
	     (not (convert-term-to-interval individual-form)))
	(define-time individual-form))))

;;; DEBUGGING AIDS

(defvar *tempos-tracing* nil)

(defvar *tempos-trace-result* nil)

(defun trace-tempos (&optional verbose-p)
  (tl:timelogic-prop :trace (if verbose-p :verbose :off))
  (setf *tempos-tracing* (if verbose-p :verbose :on))
)

(defun untrace-tempos ()
  (tl:timelogic-prop :trace :off)
  (setf *tempos-tracing* nil)
)

(defun tempos-return (successp trace-form)
;;; Return value with optional tracing
  (if *tempos-tracing*
      (let ((pretty-form (cdr trace-form)))
	(case (car trace-form)
	  (:assert (format *trace-output* "~&>>>TEMPOS Assert :  ~A~%"
			   pretty-form))
	  (:query  (format *trace-output* "~&>>>TEMPOS Query  :  ~40A    ==> ~S~%"
			   pretty-form successp))
	  (:skolem (format *trace-output* "~&>>>TEMPOS Skolem :  ~40A    ==> ~S~%"
			   (cadr pretty-form) successp))
	  (t       (format *trace-output* "~&>>>TEMPOS ~6A :  ~40A    ==> ~S~%"
			   (car trace-form)
			   pretty-form successp)))
	(setf *tempos-trace-result*
	      (if successp
		  (list pretty-form))))
      successp)
)


(defun rhet-tests (&key
		   (ask t)
		   (tests nil)
		   (tt-axioms-p t)
		   (trace-tempos :verbose)
		   (trace-reasoner :builtin))
  "Perform each test, which is a list whose car is a Y-or-N prompt
   and whose cdr is a list of strings possibly containing Rhet forms"
  (let ((*trace-output* *standard-output*)
	(*error-output* *standard-output*)
	(*print-pretty* t))
    ;; In case we're dribbling...
    (dolist (test tests)
      (when (or (not ask) (y-or-n-p "~3&~A" (car test)))
	(reset-tempos :tt-axioms-p tt-axioms-p)
	(case trace-tempos
	  (nil (untrace-tempos))
	  (:verbose (trace-tempos t))
	  (t (trace-tempos)))
	(setf reasoner:*trace-reasoner* trace-reasoner)
	(dolist (formstr (cdr test))
	  (let ((form (rhet-from-string formstr)))
	    (format t "~3&Test form  :  ~S~%" form)
	    (format t "~2&Test result:  ~S~%"
		    (E-UNIFY:crunch-vars (eval form))))))))
)


;;; INITIALIZATION

(declaim (special *last-rhet-context* *skolem-counter*))

(defun reset-tempos (&rest timelogic-keywords
		     &key tt-axioms-p trace
		     (auto-define :off)
		     (auto-reference :off)
		     &allow-other-keys)
  "Initializes Rhet and the Tempos interface to the TimeLogic system"
  (reset-rhet)
  (tl:timelogic-init)
  (prog1 (apply #'tl:timelogic-reset-props
		:auto-define auto-define
		:auto-reference auto-reference
		:auto-backtrack :off	;let tempos' error handler deal with it.
		:trace (or trace (if (eq *tempos-tracing* :verbose)
				     :verbose
				     :off))
		:allow-other-keys t timelogic-keywords)
	 (setf *skolem-counter* 0)
	 (setf *last-rhet-context* nil)
	 (tsubtype 't-u 't-time)
	 (SETQ *TIME-ITYPE* (RHET-TERMS:MAKE-I-TYPE 'T-TIME))
	 (reset-tempos-axioms)
         (setf *tempos-tracing* (if (eq trace :verbose) :verbose (if trace :on)))
	 (if tt-axioms-p #+logical-pathnames (load "tempos:tempos;tempos-rhet")
             #-logical-pathnames (load "/s5/tempos/tempos-rhet.lisp")))
)

(defun reset-tempos-axioms ()
  (reset-tempos-builtins)
  ;; This looks pretty awkward, and it is, but it's better than opening and
  ;; reading a Rhet file everytime.
  (apply #'assert-axioms
    (mapcar #'rhet-from-string
	    '(
	      ;; Interval covers

	      ;; Interval K is covered by (i.e., sequentially composed of)
	      ;; intervals I and J if I starts K and J finishes K

	      "[[Time-Cover ?i*T-Time ?j*T-Time ?k*T-Time]
		<Tempos [Time-Starts ?i ?k]
			[Time-Finishes ?j ?k]]"

	      ;; Interval K is strictly covered by intervals I and J
	      ;; if I starts K, J finishes K, and I meets J

	      "[[Time-Cover! ?i*T-Time ?j*T-Time ?k*T-Time]
		<Tempos [Time-Meets ?i ?j]
			[Time-Starts ?i ?k]
			[Time-Finishes ?j ?k]]"

	      ; And the reverse, for assertions

	      "[[AND [Time-Starts ?i*T-Time ?k*T-Time]
	             [Time-Finishes ?j*T-Time ?k]]
		<Tempos [Time-Cover ?i ?j ?k]
		:forward]"

	      "[[AND [Time-Meets ?i*T-Time ?j*T-Time]
		     [Time-Starts ?i ?k*T-Time]
		     [Time-Finishes ?j ?k]]
		<Tempos [Time-Cover! ?i ?j ?k]
		:forward]"
	      )))
)

;;; MISCELLANIOUS

(defun rhet-from-string (string &rest args)
  "Create a Rhet form by reading from a string"
  ;; Kludge in lieu of programmable interface!!!
  (let ((*package* (find-package "RHET-USER"))
	(*readtable* ui:*rhet-lisp-readtable*)
	(*print-case* :upcase))
    (read-from-string (format nil "~?" string args)))
)

;;; End of file Tempos-Main
