;;;  -*- Package: Tempos; Mode: Lisp; Syntax: Ansi-common-lisp; Base: 10.; Lowercase: Yes -*-

;;;	File:		Tempos-Builtins.lisp
;;;	Purpose:	Definitions of Tempos builtins
;;;	Last Edit:	2/23/89 03:50:39
;;;
;;;	Copyright (c) 1988, 1989  Johannes A. G. M. Koomen
;;;	All rights reserved.
;;;
;;;	The TEMPOS system provides a temporal reasoning facility for the logic
;;;	programming system RHETORICAL.  It supplies hooks into the TimeLogic
;;;	package, and defines primitives such as TT (True Throughout) etc.
(IN-PACKAGE TEMPOS)

;;; This file provides the Tempos builtins.  It uses the functions ASSERT-TIME,
;;; QUERY-TIME and QUERY-SKOLEM provided by the file Tempos-Base to implement
;;; the following builtins:
;;;
;;;	[Time-Reln ?i*T-Time ?rel*T-Lisp ?j*T-Time]
;;;		Basic interface to TimeLogic's Add-Interval-Constraint
;;;		(assertions) and Get/Test-Interval-Constraint (queries)
;;;
;;;	[Time-Durn ?i*T-Time ?dur*T-Lisp ?j*T-Time]
;;;		Likewise for durations
;;;
;;;	[Time-Skolem ?i*T-Time]
;;;		If i is unbound, binds it to a skolem interval and succeeds,
;;;		otherwise fails.
;;;
;;;	[Time-Before ?i*T-Time ?j*T-Time]
;;;		Convenient alternative to [Time-Reln ?i*T-Time (:B) ?j*T-Time]
;;;
;;;	Likewise for -After, -During, etc. (See bottom of file)
;;;
;;; For all predicates PRED (except Time-Skolem) there is also PRED-P which can
;;; only be used for queries.  They query the TimeLogic database using the
;;; INTERSECT test, i.e., they succeed if the indicated relation is Possible.
;;; Note that this is non-monotonic!  Suppose I have asserted X (:B :M) Y
;;; Then [Time-Before X Y] fails but [Time-Before-P X Y] succeeds.  Now if I
;;; update the relation by asserting X (:M) Y, [Time-Before X Y] still fails but
;;; [Time-Before-P X Y] will also!

;;; INITIALIZATION

(defvar *tempos-builtin-defs* nil)
(EVAL-WHEN (compile LOAD EVAL)
  (SETQ *TEMPOS-BUILTIN-DEFS* NIL))

(defun reset-tempos-builtins ()
  "Initializes the Tempos builtins"
  (mapc #'(lambda (defn) (apply #'rllib:define-builtin defn))
	*tempos-builtin-defs*)
)


;;; The builtin TIME-RELN
;;; This is the basic entry point: any relation can be asserted or queried.
;;; However, the predicates below, such as TIME-BEFORE, are usually preferred.

(defun assert-time-reln (tx reln ty)
  "Assert function for the Rhet predicate TIME-RELN
   Add constraint RELN between TX and TY to the temporal database"
  (assert-time tx reln ty :rel :time-reln)
)

(defun query-time-reln (failure tx reln ty)
  "Queries the TimeLogic database, binding variables as appropriate."
  (query-time failure tx reln ty :rel :subset :time-reln)
)

(defun query-time-reln-p (failure tx reln ty)
  "Queries the TimeLogic database, binding variables as appropriate."
  (query-time failure tx reln ty :rel :intersect :time-reln-p)
)

(eval-when (load eval)
  (pushnew '(query-time-reln "TIME-RELN" (t-time t-lisp t-time) :f-m assert-time-reln)
	    *tempos-builtin-defs*)
  (pushnew '(query-time-reln-p "TIME-RELN-P" (t-time t-lisp t-time) :nf-nm)
	    *tempos-builtin-defs*)
)



;;; The builtin TIME-DURN

(defun assert-time-durn (tx durn ty)
  "Assert function for the Rhet predicate TIME-DURN
   Add constraint DURN between TX and TY to the temporal database"
  (assert-time tx durn ty :dur :time-durn)
)

(defun query-time-durn (failure tx durn ty)
  "Queries the TimeLogic database, binding variables as appropriate."
  (query-time failure tx durn ty :dur :subset :time-durn)
)

(defun query-time-durn-p (failure tx durn ty)
  "Queries the TimeLogic database, binding variables as appropriate."
  (query-time failure tx durn ty :dur :intersect :time-durn-p)
)

(eval-when (load eval)
  (pushnew '(query-time-durn "TIME-DURN" (t-time t-lisp t-time) :f-m assert-time-durn)
	    *tempos-builtin-defs*)
  (pushnew '(query-time-durn-p "TIME-DURN-P" (t-time t-lisp t-time) :nf-nm)
	    *tempos-builtin-defs*)
)



;;; The builtin TIME-SKOLEM

(defun query-time-skolem (failure tx &optional rx)
  "If tx is unbound, binds it to a gensym'd interval and succeeds; otherwise fails.
   If rx given, it becomes reference of tx."
  (query-skolem failure tx rx :time-skolem)
)

(eval-when (load eval)
  (pushnew '(query-time-skolem "TIME-SKOLEM" (t-time &optional t-time) :nf-m)
	    *tempos-builtin-defs*)
)



;;; The other Tempos builtins

;;; Make life easier with a macro for defining them

(defmacro def-tempos-builtin (name constraint &optional (type :rel))
  "Create function definitions for query and assert of predicate NAME,
   and create a form to push the definition of this predicate onto the defn list"

  (let ((pred-sym (intern (format nil "TIME-~A" name)
			  (find-package "KEYWORD")))
	(query-sym (intern (format nil "QUERY-TIME-~A" name)))
	(assert-sym (intern (format nil "ASSERT-TIME-~A" name)))
	(pred-p-sym (intern (format nil "TIME-~A-P" name)
			    (find-package "KEYWORD")))
	(query-p-sym (intern (format nil "QUERY-TIME-~A-P" name))))

    `(eval-when (compile load eval)
       
       (defun ,assert-sym (tx ty)
	 (assert-time tx ,constraint ty ,type ,pred-sym))
       
       (defun ,query-sym (failure tx ty)
	 (query-time failure tx ,constraint ty ,type :subset ,pred-sym))

       (defun ,query-p-sym (failure tx ty)
	 (query-time failure tx ,constraint ty ,type :intersect ,pred-p-sym))

       (eval-when (load eval)
	 (pushnew '(,query-sym ,(symbol-name pred-sym) (t-time t-time) :f-m ,assert-sym)
		  *tempos-builtin-defs*)
	 (pushnew '(,query-p-sym ,(symbol-name pred-p-sym) (t-time t-time) :nf-nm)
		  *tempos-builtin-defs*))))
)


;;; Relational constraints

(def-tempos-builtin "AFTER"		:A)
(def-tempos-builtin "BEFORE"		:B)
(def-tempos-builtin "CONTAINS"		:C)
(def-tempos-builtin "DURING"		:D)
(def-tempos-builtin "EQUALS"		:E)
(def-tempos-builtin "FINISHES"		:F)
(def-tempos-builtin "FINISHED-BY"	:Fi)
(def-tempos-builtin "MEETS"		:M)
(def-tempos-builtin "MET-BY"		:Mi)
(def-tempos-builtin "OVERLAPS"		:O)
(def-tempos-builtin "OVERLAPPED-BY"	:Oi)
(def-tempos-builtin "STARTS"		:S)
(def-tempos-builtin "STARTED-BY"	:Si)
(def-tempos-builtin "WITHIN"		'(:D :E :F :S))
(def-tempos-builtin "WITHIN!"		'(:D :F :S))
(def-tempos-builtin "DISJOINT"		'(:A :B :M :Mi))
(def-tempos-builtin "DISJOINT!"		'(:A :B))
(def-tempos-builtin "INTERSECTS"	'(:C :D :E :F :Fi :O :Oi :S :Si))
(def-tempos-builtin "STARTS-LATER"	'(:A :D :F :Mi :Oi))
(def-tempos-builtin "STARTS-EARLIER"	'(:B :C :Fi :M :O))
(def-tempos-builtin "FINISHES-LATER"	'(:A :C :Mi :Oi :Si))
(def-tempos-builtin "FINISHES-EARLIER"	'(:B :D :M :O :S))

;;; Durational constraints

(def-tempos-builtin "<"			'<	:dur)
(def-tempos-builtin "="			'=	:dur)
(def-tempos-builtin ">"			'>	:dur)
(def-tempos-builtin "<="		'<=	:dur)
(def-tempos-builtin ">="		'>=	:dur)
(def-tempos-builtin "<>"		'tl:<>	:dur)



;;; End of file Tempos-Builtins
