;;;  -*- Mode:Lisp; Package: TEMPOS; Base:10; Syntax: Ansi-common-lisp -*-

;;;	File:		Tempos-Base.lisp
;;;	Purpose:	TimeLogic interface and builtin generators
;;;	Last Edit:	2/23/89 23:28:04
;;;
;;;	Copyright (c) 1988, 1989  Johannes A. G. M. Koomen
;;;	All rights reserved.
;;;
;;;	The TEMPOS system provides a temporal reasoning facility for the logic
;;;	programming system RHETORICAL.  It supplies hooks into the TimeLogic
;;;	package, and defines primitives such as TT (True Throughout) etc.
(IN-PACKAGE TEMPOS)

;;; This file provides the interface to the TimeLogic system, in the form of
;;; three functions:  ASSERT-TIME, QUERY-TIME and QUERY-SKOLEM.  
;;; These functions are used in the file Tempos-Builtins to create a whole
;;; bunch of predicates.

;
;(proclaim '(optimize safety))

(DECLAIM (SPECIAL *TEMPOS-TRACING*))

(DEFVAR *TEMPOS-DATA-PACKAGE*
	(OR (FIND-PACKAGE "Tempos-Data")
	    (MAKE-PACKAGE "Tempos-Data"))
  "Package in which Rhet term accessors are interned 
   to map them into unique intervals")

(DEFVAR *LAST-RHET-CONTEXT* NIL
  "Remember last Rhet context seen")

(DEFVAR *LAST-TEMPOS-CONTEXT* NIL
  "Remember TimeLogic context that goes with *last-rhet-context*")

(DEFVAR *SKOLEM-COUNTER* 0
  "Index of last Skolem generated")

(DEFVAR *SKOLEMIZING* NIL
  "Special bound to T when Skolemizing.
   Used by query-time to assert when appropriate")

(DEFVAR *TIME-ITYPE* NIL
  "Remember the itype of rhet type 't-time so we don't keep looking it up (set by reset-tempos)")

;;; MAINTAINING CURRENT CONTEXT

(DEFUN SET-TIMELOGIC-CONTEXT ()
  "Map current Rhet context to appropriate TimeLogic context, making sure that
   all TimeLogic children are deleted if all Rhet children were (e.g., in case
   after ASSUME)"
  (LET ((TL-CONTEXT
	  (TL:SWITCH-CONTEXT
	    (IF (EQ TERM-SUPPORT:*CURRENT-CONTEXT* *LAST-RHET-CONTEXT*)
		*LAST-TEMPOS-CONTEXT*
		(SETF *LAST-TEMPOS-CONTEXT*
		      (GET-TL-CONTEXT
			(SETF *LAST-RHET-CONTEXT*
			      TERM-SUPPORT:*CURRENT-CONTEXT*)))))))
    (UNLESS (RHET-TERMS:CONTEXT-CHILDREN TERM-SUPPORT:*CURRENT-CONTEXT*)
      (MAPC #'(LAMBDA (CHILD) (TL::TL-DELETE-CONTEXT CHILD NIL T))
	    (TL::CONTEXT-CHILDREN (TL::TL-FIND-CONTEXT TL-CONTEXT))))
    TL-CONTEXT)
)

(DEFUN GET-TL-CONTEXT (RHET-CONTEXT)
  (OR (EQ RHET-CONTEXT RHET-TERMS:*ROOT-CONTEXT*)
      (TL:CREATE-CONTEXT
	(INTERN (FORMAT NIL "TIME-~A" (RHET-TERMS:CONVERT-CONTEXT-TO-NAME RHET-CONTEXT))
		*TEMPOS-DATA-PACKAGE*)
	(GET-TL-CONTEXT
	  (RHET-TERMS:CONTEXT-PARENT RHET-CONTEXT)))))


;;; ASSURING PROPER TYPING

;;; This rewrite done by miller@cs.rochester.edu to allow canonicals to be handled, 
;;; and make the type update more efficient than reparsing.
(DEFUN CONVERT-TERM-TO-INTERVAL (TX &OPTIONAL ASSERT-P REFINT)
  " Terms are normalized to Rhet fn-terms.  Then the
fn-terms are mapped to unique symbols based on the printname of the
fnt, which are interned in the interval package.
If assert-p = T, create the interval if necessary, otherwise return nil"
  (LET*-NON-NULL ((MARKED-FNT (TYPECASE TX
				(RHET-TERMS:FN-TERM
				  TX)
				(RHET-TERMS:RFORM
				  (E-UNIFY:CONVERT-TO-FN-TERM TX))
				(RHET-TERMS:CANONICAL
				  (BLOCK MFNT
				    (LET (FOUND-FNT)
				      (MAPC #'(LAMBDA (SETENTRY)
						(COND
						  ((RHET-TERMS:FN-TERM-P SETENTRY)
						   (UNLESS FOUND-FNT
						     (SETQ FOUND-FNT SETENTRY))	;use this if none are marked.
						   (IF (ASSOC 'INTERVAL (RHET-TERMS:PLIST SETENTRY))
						       (RETURN-FROM MFNT SETENTRY)))))
					    (RHET-TERMS:CSET TX))
				      FOUND-FNT))))))
    (LET ((INTERVAL (CDR (ASSOC 'INTERVAL (RHET-TERMS:PLIST MARKED-FNT)))))
      (COND
	((AND INTERVAL (TL:INTERVAL-DEFINED-P INTERVAL))
	 (IF (AND ASSERT-P REFINT)
	     (TL:DEFINE-INTERVAL INTERVAL REFINT)
	     INTERVAL))
	((NOT ASSERT-P) NIL)
	(T (SETF INTERVAL (INTERN (FORMAT NIL "int-~d" (RHET-TERMS:SERIAL-NUMBER MARKED-FNT)) *TEMPOS-DATA-PACKAGE*))
	   (PUSH (CONS 'INTERVAL INTERVAL) (RHET-TERMS:PLIST MARKED-FNT))
	   (SETF (GET INTERVAL 'FN-TERM) MARKED-FNT)
	   (UNLESS (RHET-TERMS:TYPECHECK (RHET-TERMS:STYPE MARKED-FNT) *TIME-ITYPE*)
	     (E-UNIFY:TYPE-RESTRICT-TERM MARKED-FNT *TIME-ITYPE* TERM-SUPPORT:*CURRENT-CONTEXT*))
	   (TL:DEFINE-INTERVAL INTERVAL (OR REFINT '|Tempos-Root|)))))))


(DEFUN CONVERT-INTERVAL-TO-TERM (INTERVAL)
  (GET INTERVAL 'FN-TERM))


;;; FAILURES
(DEFINE-CONDITION TEMPOS-PROOF-INCONSISTANCY (REASONER:RHET-PROOF-INCONSISTANCY) 
  ((TX :INITFORM NIL :READER TX :INITARG :TX)
   (TREL :INITFORM NIL :READER TREL :INITARG :TREL)
   (TY :INITFORM NIL :READER TY :INITARG :TY)
   (STYPE :INITFORM NIL :READER COND-TYPE :INITARG :STYPE)
   (MESSAGE :INITFORM NIL :READER MESSAGE :INITARG :MESSAGE)
   (ARGS :INITFORM NIL :READER ARGS :INITARG :ARGS))
  (:REPORT (LAMBDA (CONDITION STREAM)
	     (COND
	       ((EQ (TX CONDITION) :TIMELOGIC)
		(SETQ TL::*TL-ERROR-OCCURRED-P* T)
		(FORMAT STREAM "~2&*** TimeLogic ERROR: ~?~%" (MESSAGE CONDITION) (ARGS CONDITION)))
	       (T
		(FORMAT STREAM "~2&*** Tempos failure in [~A ~A ~S ~A] :~%*** ~?~%"
			(IF (EQ (COND-TYPE CONDITION) :REL) :TIME-RELN :TIME-DURN)
			(TX CONDITION)
			(TREL CONDITION)
			(TY CONDITION)
			(MESSAGE CONDITION)
			(ARGS CONDITION)))))))

(DEFVAR *TEMPOS-FAIL-HANDLER* NIL "Set non-nil if we should just fail when there is a problem")

(DEFUN SCM-ADVICE (ARGLIST)
  (COND
    ((EQ QUERY::*CURRENT-CONTRADICTION-MODE* :DEFAULT)
     ;; make sure (un)defined.
     (SETQ *TEMPOS-FAIL-HANDLER* NIL)))
  (IF (EQ (CAR ARGLIST) :DEFAULT)
      (SETQ *TEMPOS-FAIL-HANDLER* T)
      T))

  #+SYMBOLICS  
(EVAL-WHEN (LOAD EVAL)
  ZL:::SCL:
  (ADVISE QUERY:SET-CONTRADICTION-MODE :BEFORE TEMPOS::TEMPOS-SET-CONTRADICTION-MODE NIL
	  (TEMPOS::SCM-ADVICE ARGLIST)))

  #+EXCL
(eval-when (load eval)
  (EXCL:ADVISE QUERY:SET-CONTRADICTION-MODE :BEFORE TEMPOS-SET-CONTRADICTION-MODE NIL
	  (SCM-ADVICE excl:ARGLIST)))

(DEFUN TEMPOS-FAILURE (TX TREL TY TYPE MESSAGE &REST ARGS)
  (LET ((ERROR-CONDITION (MAKE-CONDITION 'TEMPOS-PROOF-INCONSISTANCY
					 :TX TX :TREL TREL :TY TY :STYPE TYPE :MESSAGE MESSAGE :ARGS ARGS))
	(*DEFAULT-CONTEXT* *CURRENT-CONTEXT*))
    (DECLARE (SPECIAL *DEFAULT-CONTEXT*))
    (COND
      ((EQ QUERY::*CURRENT-CONTRADICTION-MODE* :THROW)
       (TERM-SUPPORT:RHET-CONTRADICTION-THROW ERROR-CONDITION))
      ((AND RHET-TERMS:*CURRENT-CONTINUATION* *TEMPOS-FAIL-HANDLER*)
       (RLLIB:INVOKE-CONTINUATION RHET-TERMS:*CURRENT-CONTINUATION*))
      ((EQ QUERY::*CURRENT-CONTRADICTION-MODE* :IGNORE)
       (RETURN-FROM TEMPOS-FAILURE NIL))
      (T
	(RESTART-CASE (SIGNAL ERROR-CONDITION)
	  (IGNORE () :REPORT "(Tempos) Ignore the call that produced this problem. Pretend success."
		  (RETURN-FROM TEMPOS-FAILURE NIL))
	  (FAIL () :TEST (LAMBDA (C) (DECLARE (IGNORE C)) RHET-TERMS:*CURRENT-CONTINUATION*)
		:REPORT "Fail this subgoal"
		(RLLIB:INVOKE-CONTINUATION RHET-TERMS:*CURRENT-CONTINUATION*))
	  (DEBUG () :REPORT "Invoke Rhet debugger"
		 (BREAK "The Rhet debugger :-)")))))))

(DEFUN TL-ERROR-ADVICE (ARGLIST)
  "Gets executed before timelogic's normal tl-error function. Intercepts the error and uses the condition system."
  (LET ((ERROR-CONDITION (MAKE-CONDITION 'TEMPOS-PROOF-INCONSISTANCY
					 :TX :TIMELOGIC :MESSAGE (CAR ARGLIST) :ARGS (CDR ARGLIST)))
	(*DEFAULT-CONTEXT* *CURRENT-CONTEXT*))
    (DECLARE (SPECIAL *DEFAULT-CONTEXT*))
    (COND
      ((EQ QUERY::*CURRENT-CONTRADICTION-MODE* :THROW)
       (TL::TL-BACKTRACK NIL T)
       (TERM-SUPPORT:RHET-CONTRADICTION-THROW ERROR-CONDITION))
      ((AND RHET-TERMS:*CURRENT-CONTINUATION* *TEMPOS-FAIL-HANDLER*)
       (TL::TL-BACKTRACK NIL T)
       (RLLIB:INVOKE-CONTINUATION RHET-TERMS:*CURRENT-CONTINUATION*))
      (T
	(RESTART-CASE (SIGNAL ERROR-CONDITION)
	  (FAIL () :TEST (LAMBDA (C) (DECLARE (IGNORE C)) RHET-TERMS:*CURRENT-CONTINUATION*)
		:REPORT "Fail this subgoal"
		(PROGN (TL::TL-BACKTRACK NIL T)
		       (RLLIB:INVOKE-CONTINUATION RHET-TERMS:*CURRENT-CONTINUATION*)))
	  (DEBUG () :REPORT "Invoke Rhet debugger"
		 (BREAK "The Rhet debugger :-)"))
	  (CONTINUE () :REPORT "(Tempos) Ignore the problem, make things inconsistant, continue adding the assertion."
		    (PROGN (TL::TL-BACKTRACK NIL T "Undo last constraint propagation? ")
			   (FORMAT *ERROR-OUTPUT*
				   "~2&*** Proceeding from TimeLogic ERROR ~A~%"
				   "with value NIL (may not work...)")
			   (UNLESS (OR (NULL TL::*TL-BACKTRACK-POINT*)
				       (EQ (TL::BTPOINT-KIND TL::*TL-BACKTRACK-POINT*) :CHECKPOINT))
			     (WARN "DATABASE IS PROBABLY INCONSISTENT!!!   All bets are off..."))
			   NIL)))))))


#+SYMBOLICS
(EVAL-WHEN (LOAD EVAL)
  ZL:::SCL:
  (ADVISE TL::TL-ERROR :BEFORE TEMPOS::TEMPOS-ERROR-HANDLING NIL
	  (TEMPOS::TL-ERROR-ADVICE ARGLIST)))
#+EXCL
(eval-when (load eval)
  (EXCL:ADVISE TL::TL-ERROR :BEFORE TEMPOS-ERROR-HANDLING NIL
               (TL-ERROR-ADVICE EXCL:ARGLIST)))

#+excl
(eval-when (load eval)
  (excl:advise tl::tl-backtrack :before tempos-error-handling nil
               (when *tempos-tracing*
                 (format *trace-output* "~&>>>TEMPOS Backtrack :  ~S~%" tl::*tl-backtrack-point*))))

#+excl
(eval-when (load eval)
  (excl:advise tl::timelogic-checkpoint :after tempos-error-handling nil
               (when (eq *tempos-tracing* :verbose)
                 (format *trace-output* "~&>>>TEMPOS Checkpoint :  ~S~%" tl::*tl-backtrack-point*))))

;;; TIME ASSERTIONS

(DEFMACRO ASSERT-TIME-RETURN (RESULT)
  `(TEMPOS-RETURN
     ,RESULT
     (IF *TEMPOS-TRACING*
	 (IF (OR (EQ PRED-NAME :TIME-RELN)
		 (EQ PRED-NAME :TIME-DURN))
	     (LIST :ASSERT PRED-NAME TX CONSTRAINT TY)
	     (LIST :ASSERT PRED-NAME TX TY))))
)

(DEFUN ASSERT-TIME (TX CONSTRAINT TY TYPE PRED-NAME)
  "Assert function for the Rhet predicates TIME-RELN and TIME-DURN
   Add CONSTRAINT between TX and TY to the temporal database"
  (SET-TIMELOGIC-CONTEXT)
  (LET (XINT YINT)
    (COND ((OR (NOT (RHET-TERMS:TYPECOMPAT (RHET-TERMS:GET-TYPE TX) *TIME-ITYPE*))
	       (NULL (SETF XINT (CONVERT-TERM-TO-INTERVAL TX T))))	;assert it otherwise, since it *should* be ok.
	   (TEMPOS-FAILURE TX CONSTRAINT TY TYPE
			   "~A cannot be converted to an interval!" TX)
	   (ASSERT-TIME-RETURN NIL))
	  ((OR (NOT (RHET-TERMS:TYPECOMPAT (RHET-TERMS:GET-TYPE TY) *TIME-ITYPE*))
	       (NULL (SETF YINT (CONVERT-TERM-TO-INTERVAL TY T))))	;assert it otherwise, since it *should* be ok.
	   (TEMPOS-FAILURE TX CONSTRAINT TY TYPE
			   "~A cannot be converted to an interval!" TY)
	   (ASSERT-TIME-RETURN NIL))
	  ((NULL (TL:ADD-INTERVAL-CONSTRAINT XINT CONSTRAINT YINT
					     :TYPE TYPE))
	   (TEMPOS-FAILURE TX CONSTRAINT TY TYPE
			   "~S is inconsistent with existing constraint ~S"
			   CONSTRAINT
			   (TL:GET-INTERVAL-CONSTRAINT XINT YINT :TYPE TYPE))
	   (ASSERT-TIME-RETURN NIL))
	  (T
	   ;; ---------------------------------------------------------- 
	   ;; --- NEED TO DECIDE WHAT TO DO ABOUT BACKTRACKING POINT ---
	   ;; ----------------------------------------------------------
	   (ASSERT-TIME-RETURN T)
	   T)))
)


;;; TIME QUERIES


(DEFMACRO QUERY-TIME-RETURN (RESULT)
  `(TEMPOS-RETURN
     ,RESULT
     (IF *TEMPOS-TRACING*
	 (IF (OR (EQ PRED-NAME :TIME-RELN)
		 (EQ PRED-NAME :TIME-DURN)
		 (EQ PRED-NAME :TIME-RELN-P)
		 (EQ PRED-NAME :TIME-DURN-P))
	     (LIST :QUERY PRED-NAME TX TREL TY)
	     (LIST :QUERY PRED-NAME TX TY))))
)

(DEFUN QUERY-TIME-PROGRESS (SUCCESSCNT POSSIBILITIES
			    &OPTIONAL TAIL-P XTAIL YTAIL) 
  (LET ((POSSCNT (IF (NULL TAIL-P)
		     (LIST-LENGTH POSSIBILITIES)
		     (+ (LIST-LENGTH YTAIL)
			(IF (CDR XTAIL)
			    (* (LIST-LENGTH (CDR XTAIL))
			       (LIST-LENGTH (CDR POSSIBILITIES)))
			    0)))))
    (FORMAT *TRACE-OUTPUT* "~&>>>TEMPOS Query  :  ~D succes~:[ses~;s~]"
	    SUCCESSCNT (= SUCCESSCNT 1))
    (FORMAT *TRACE-OUTPUT* ", ~D possibilit~:[ies~;y~] left~%"
	    POSSCNT (= POSSCNT 1)))
)

(DEFUN QUERY-TIME (FAILURE TX TREL TY TYPE TEST PRED-NAME)
  "Returns an arg-dependent generator to query the TimeLogic database,
   binding variables as appropriate."
  (E-UNIFY:RATIONALIZE-ARGUMENT TX)
  (E-UNIFY:RATIONALIZE-ARGUMENT TREL)
  (E-UNIFY:RATIONALIZE-ARGUMENT TY)
  (LET ((XVARP (RHET-TERMS:RVARIABLE-P TX))
	(RVARP (RHET-TERMS:RVARIABLE-P TREL))
	(YVARP (RHET-TERMS:RVARIABLE-P TY)))
    (LET ((XINT (UNLESS XVARP (CONVERT-TERM-TO-INTERVAL TX)))
	  (YINT (UNLESS YVARP (CONVERT-TERM-TO-INTERVAL TY))))

      (COND
       ((AND (NOT XVARP) (NULL XINT))
        (let ((rhet-terms:*current-continuation* failure))
          (TEMPOS-FAILURE TX TREL TY TYPE
                          "~A cannot be converted to an interval!" TX))
        (RLLIB:BUILTIN-EXECUTES-ONCE-ONLY FAILURE NIL 
                                          (QUERY-TIME-RETURN NIL)))

       ((AND (NOT YVARP) (NULL YINT))
        (let ((rhet-terms:*current-continuation* failure))
          (TEMPOS-FAILURE TX TREL TY TYPE
                          "~A cannot be converted to an interval!" TY)
          (RLLIB:BUILTIN-EXECUTES-ONCE-ONLY FAILURE NIL 
                                            (QUERY-TIME-RETURN NIL))))

	((AND (NOT XVARP) (NOT RVARP) (NOT YVARP))
	 (QUERY-TIME-GROUND FAILURE TX TREL TY TYPE TEST PRED-NAME XINT YINT))

	(*SKOLEMIZING*
	 ;; Verifying the posted constraints an a fresh skolem, but obviously
	 ;; there are other vars in the form, so punt on it...
	 (RLLIB:BUILTIN-EXECUTES-ONCE-ONLY FAILURE NIL
	   (QUERY-TIME-RETURN T)))

	((AND (NOT XVARP) RVARP (NOT YVARP))
	 ;; GET CURRENT RELATION, QUIT IF NO GOOD
	 (RLLIB:BUILTIN-EXECUTES-ONCE-ONLY FAILURE NIL
	   (QUERY-TIME-RETURN
	     (LET ((RESULT (TL:GET-INTERVAL-CONSTRAINT
			     XINT YINT
			     :TYPE TYPE
			     :CONTEXT (SET-TIMELOGIC-CONTEXT))))
	       (AND RESULT (E-UNIFY:UNIFY-RVARIABLE TREL RESULT))))))

	((NOT YVARP)
	 (QUERY-TIME-NOT-YVARP
	   FAILURE TX TREL TY TYPE TEST PRED-NAME YINT RVARP)) 

	((NOT XVARP)
	 (QUERY-TIME-NOT-XVARP
	   FAILURE TX TREL TY TYPE TEST PRED-NAME XINT RVARP))

	((NOT RVARP)
	 (QUERY-TIME-NOT-RVARP
	   FAILURE TX TREL TY TYPE TEST PRED-NAME))

	(T
         ;; GIVE UP -- DON'T KNOW WHAT TO DO
         (let ((rhet-terms:*current-continuation* failure))
           (TEMPOS-FAILURE TX TREL TY TYPE "Don't know what to do with this!"))
	 (RLLIB:BUILTIN-EXECUTES-ONCE-ONLY FAILURE NIL 
	   (QUERY-TIME-RETURN NIL))))))
)

(DEFUN QUERY-TIME-GROUND (FAILURE TX TREL TY TYPE TEST PRED-NAME XINT YINT)
  ;; GROUND TERM
  ;; Check the constraint, succeed if ok, fail if not skolemizing.
  (RLLIB:BUILTIN-EXECUTES-ONCE-ONLY FAILURE NIL
    (SET-TIMELOGIC-CONTEXT)
    (QUERY-TIME-RETURN
      (COND ((NOT *SKOLEMIZING*)
	     (TL:TEST-INTERVAL-CONSTRAINT XINT TREL YINT :TYPE TYPE :TEST TEST))
	    ;; One of the ints is a skolem so try to assert constraint and
	    ;; succeed.  A posted temporal constraint holds for a skolem by
	    ;; default.  However, we may need to back out of this commitment.
	    ;; Unfortunately, the prover for posted constraints does not
	    ;; backtrack.  We don't need to worry about it here, though, because
	    ;; Time-Skolem obtained a TimeLogic checkpoint and will backtrack in
	    ;; one fell swoop if necessary. 
	    ((TL:ADD-INTERVAL-CONSTRAINT XINT TREL YINT :TYPE TYPE) T))))
)

(DEFUN QUERY-TIME-NOT-YVARP (FAILURE TX TREL TY TYPE TEST PRED-NAME YINT RVARP)
  ;;TX IS A VAR, AND TREL MIGHT BE
  (LET ((SUCCESSCNT 0)
	(POSSIBILITIES (TL:DEFINED-INTERVALS)))
    #'(LAMBDA (CULPRIT)
	(DECLARE (IGNORE CULPRIT))
	(LET (CANDIDATE)
	  (SET-TIMELOGIC-CONTEXT)
	  (IF (EQ *TEMPOS-TRACING* :VERBOSE)
	      (QUERY-TIME-PROGRESS SUCCESSCNT POSSIBILITIES))
	  (LOOP
	    (UNLESS POSSIBILITIES
	      (IF (EQ *TEMPOS-TRACING* :VERBOSE)
		  (QUERY-TIME-PROGRESS SUCCESSCNT POSSIBILITIES))
	      (RLLIB:INVOKE-CONTINUATION FAILURE (QUERY-TIME-RETURN NIL)))
	    (SETF CANDIDATE (POP POSSIBILITIES))
	    (COND
	      (RVARP
	       (LET ((RESULT (TL:GET-INTERVAL-CONSTRAINT
			       CANDIDATE YINT
			       :TYPE TYPE)))
		 (IF (AND (NOT (EQ RESULT :ALL))
			  (E-UNIFY:UNIFY-RVARIABLE
			    TX (CONVERT-INTERVAL-TO-TERM CANDIDATE))
			  (E-UNIFY:UNIFY-RVARIABLE TREL RESULT))
		     (RETURN))))
	      ((TL:TEST-INTERVAL-CONSTRAINT
		 CANDIDATE TREL YINT
		 :TYPE TYPE
		 :TEST TEST)
	       (IF (E-UNIFY:UNIFY-RVARIABLE
		     TX (CONVERT-INTERVAL-TO-TERM CANDIDATE))
		   (RETURN)))))
	  ;; Broke out of the loop with a succesful candidate
	  (IF (EQ *TEMPOS-TRACING* :VERBOSE)
	      (QUERY-TIME-PROGRESS (INCF SUCCESSCNT) POSSIBILITIES))
	  (QUERY-TIME-RETURN T))))
)

(DEFUN QUERY-TIME-NOT-XVARP (FAILURE TX TREL TY TYPE TEST PRED-NAME XINT RVARP)
  ;;TY IS A VAR, AND TREL MIGHT BE
  (LET ((SUCCESSCNT 0)
	(POSSIBILITIES (TL:DEFINED-INTERVALS)))
    ;; Better be cautious here.  Don't use tl:related-intervals as this returns
    ;; only directly related intervals!
    #'(LAMBDA (CULPRIT)
	(DECLARE (IGNORE CULPRIT))
	(LET (CANDIDATE)
	  (SET-TIMELOGIC-CONTEXT)
	  (IF (EQ *TEMPOS-TRACING* :VERBOSE)
	      (QUERY-TIME-PROGRESS SUCCESSCNT POSSIBILITIES))
	  (LOOP
	    (UNLESS POSSIBILITIES
	      (IF (EQ *TEMPOS-TRACING* :VERBOSE)
		  (QUERY-TIME-PROGRESS SUCCESSCNT POSSIBILITIES))
	      (RLLIB:INVOKE-CONTINUATION FAILURE (QUERY-TIME-RETURN NIL)))
	    (SETF CANDIDATE (POP POSSIBILITIES))
	    (COND
	      (RVARP
	       (LET ((RESULT (TL:GET-INTERVAL-CONSTRAINT
			       XINT CANDIDATE
			       :TYPE TYPE)))
		 (IF (AND (NOT (EQ RESULT :ALL))
			  (E-UNIFY:UNIFY-RVARIABLE
			    TY (CONVERT-INTERVAL-TO-TERM CANDIDATE))
			  (E-UNIFY:UNIFY-RVARIABLE TREL RESULT))
		     (RETURN))))
	      ((TL:TEST-INTERVAL-CONSTRAINT
		 XINT TREL CANDIDATE
		 :TYPE TYPE
		 :TEST TEST)
	       (IF (E-UNIFY:UNIFY-RVARIABLE
		     TY (CONVERT-INTERVAL-TO-TERM CANDIDATE))
		   (RETURN)))))
	  ;; Broke out of the loop with a succesful candidate
	  (IF (EQ *TEMPOS-TRACING* :VERBOSE)
	      (QUERY-TIME-PROGRESS (INCF SUCCESSCNT) POSSIBILITIES))
	  (QUERY-TIME-RETURN T))))
)

(DEFUN QUERY-TIME-NOT-RVARP (FAILURE TX TREL TY TYPE TEST PRED-NAME)
  ;;TX AND TY ARE VARS, TREL IS NOT
  (LET* ((SUCCESSCNT 0)
	 (POSSIBILITIES (TL:DEFINED-INTERVALS))
	 (XTAIL POSSIBILITIES)
	 (YTAIL (CDR POSSIBILITIES)))
    #'(LAMBDA (CULPRIT)
	(DECLARE (IGNORE CULPRIT))
	(LET (XHEAD YHEAD)
	  (SET-TIMELOGIC-CONTEXT)
	  (IF (EQ *TEMPOS-TRACING* :VERBOSE)
	      (QUERY-TIME-PROGRESS SUCCESSCNT POSSIBILITIES T XTAIL YTAIL))
	  (SETF XHEAD (CAR XTAIL))
	  (LOOP
	    (WHEN (NULL YTAIL)
	      (POP XTAIL)
	      (UNLESS XTAIL
		(IF (EQ *TEMPOS-TRACING* :VERBOSE)
		    (QUERY-TIME-PROGRESS SUCCESSCNT POSSIBILITIES T XTAIL YTAIL))
		(RLLIB:INVOKE-CONTINUATION FAILURE (QUERY-TIME-RETURN NIL)))
	      (SETF XHEAD (CAR XTAIL))
	      (SETF YTAIL POSSIBILITIES))
	    (SETF YHEAD (POP YTAIL))
	    (WHEN (AND (NOT (EQ YHEAD XHEAD))
		       (TL:TEST-INTERVAL-CONSTRAINT XHEAD TREL YHEAD
			 :TYPE TYPE :TEST TEST)
		       (E-UNIFY:UNIFY-RVARIABLE
			 TX (CONVERT-INTERVAL-TO-TERM XHEAD))
		       (E-UNIFY:UNIFY-RVARIABLE
			 TY (CONVERT-INTERVAL-TO-TERM YHEAD)))
	      (RETURN)))
	  ;; Broke out of the loop with a succesful candidate
	  (IF (EQ *TEMPOS-TRACING* :VERBOSE)
	      (QUERY-TIME-PROGRESS (INCF SUCCESSCNT) POSSIBILITIES T XTAIL YTAIL))
	  (QUERY-TIME-RETURN T))))
)



;;; Generating time skolems

(DEFUN MAKE-SKOLEM-TERM (REFINT)
  ;; Returns a Rhet fn-term
  (LET (NAME INTERVAL FN-TERM)
    (LOOP
      (INCF *SKOLEM-COUNTER*)
      (SETF NAME (FORMAT NIL "TIME-SKOLEM-~4,'0D" *SKOLEM-COUNTER*))
      (SETF INTERVAL (FIND-SYMBOL NAME *TEMPOS-DATA-PACKAGE*))
      (UNLESS (AND INTERVAL (TL:INTERVAL-DEFINED-P INTERVAL))
	(SETF INTERVAL (INTERN NAME *TEMPOS-DATA-PACKAGE*))
        (if refint
            (define-time (setq fn-term (cons-rhet-form interval)) :reference refint)
          (define-time (setq fn-term (cons-rhet-form interval))))
        (setq fn-term (e-unify::convert-to-fn-term fn-term))
        (setq interval (convert-term-to-interval fn-term))
        ;;	(PUSH (CONS 'INTERVAL INTERVAL) (RHET-TERMS:PLIST FN-TERM))
        ;; (SETF (GET INTERVAL 'FN-TERM) FN-TERM)
	(SETF (GET INTERVAL 'SKOLEM-P) T)
        ;; (TL:DEFINE-INTERVAL INTERVAL (OR REFINT '|Tempos-Root|))
	(RETURN FN-TERM))))
)


(DEFUN QUERY-SKOLEM (FAILURE TX RX PRED-NAME)
  "Creates a generator to generate a Skolem interval provided tx is unbound.
   Existing (i.e., posted) constraints on tx are asserted, with backtracking
   done here because Rhet doesn't backtrack under e-unify:unify-rvariable"
  (E-UNIFY:RATIONALIZE-ARGUMENT TX)
  (IF RX (E-UNIFY:RATIONALIZE-ARGUMENT RX))
  (COND ((RHET-TERMS:RVARIABLE-P TX)
	 ;; Variable, so bind it to new skolem
	 (LET ((BTPOINT (TL:TIMELOGIC-CHECKPOINT))
	       (TRIED NIL)
	       (SUCCESS NIL))
	   #'(LAMBDA (CULPRIT)
	       (DECLARE (IGNORE CULPRIT))
	       (SET-TIMELOGIC-CONTEXT)
	       (COND ((NOT TRIED)
		      (LET ((*SKOLEMIZING* T)
			    (FN-TERM
			      (MAKE-SKOLEM-TERM
				(IF RX (CONVERT-TERM-TO-INTERVAL RX)))))
			(SETF TRIED T)
			(SETF SUCCESS (E-UNIFY:UNIFY-RVARIABLE TX FN-TERM))))
		     (T
		      ;; Coming in again, so Rhet is backtracking, so we
		      ;; backtrack TimeLogic and fail
		      (SETF SUCCESS NIL)))
	       (LET ((TRACE-FORM
		       (IF *TEMPOS-TRACING* (LIST :SKOLEM PRED-NAME TX)))) 
		 (COND ((NOT SUCCESS)
			(WHEN BTPOINT
			  (TL:TIMELOGIC-BACKTRACK BTPOINT)
			  (SETF BTPOINT NIL))
			(RLLIB:INVOKE-CONTINUATION FAILURE
			  (TEMPOS-RETURN NIL TRACE-FORM)))
		       (T (TEMPOS-RETURN T TRACE-FORM)))))))
	(T
	 ;; Ground term, so fail by definition
	 (RLLIB:BUILTIN-EXECUTES-ONCE-ONLY FAILURE NIL 
	   (TEMPOS-RETURN
	     NIL
	     (IF *TEMPOS-TRACING* (LIST :SKOLEM PRED-NAME TX))))))
)

(EVAL-WHEN (LOAD EVAL)
  (SETQ TL:*TLR-ASSERT-HOOK* 'ASSERT-RLINK-HOOK))

(DEFVAR *CURRENT-EQUALITIES* NIL
  "Keep track of the equalities assert-rlink-hook has open so we don't recurse infinitely")

(DEFUN ASSERT-RLINK-HOOK (TX RLINK TY)
  ;; Called by TimeLogic while trying to assert rlink between tx and ty
  ;; Called with encoded args for efficiency reasons
  (COND
    ((TL::TLR-SAME-RLINK-P RLINK (TL::TLR-CONST :E))    ;; don't care about this one...
     (LET ((TERM1 (CONVERT-INTERVAL-TO-TERM (TL::INTERVAL-NAME TX)))
	   (TERM2 (CONVERT-INTERVAL-TO-TERM (TL::INTERVAL-NAME TY))))
       (COND
	 ((ADDING-EQ-P TERM1 TERM2 *CURRENT-CONTEXT*)
	  T)
	 ((LET ((*CURRENT-EQUALITIES* (CONS (LIST TERM1 TERM2 TERM-SUPPORT:*CURRENT-CONTEXT*)
					    (IF (BOUNDP '*CURRENT-EQUALITIES*)
						*CURRENT-EQUALITIES*))))
	    (DECLARE (SPECIAL *CURRENT-EQUALITIES*))

	    (ASSERT:ADD-EQ TERM1 TERM2 :RCONTEXT TERM-SUPPORT:*CURRENT-CONTEXT*))
	  ;; Rhet said it was ok so go ahead and do it
	  T)
	 (T
	  ;; Rhet choked on it, so tell TimeLogic to forget it!
	  NIL))))
    (T T)))

(EVAL-WHEN (LOAD EVAL)
  (PUSHNEW 'ASSERT-EQ-HOOK E-UNIFY:*ADD-EQ-BEFORE-HOOKS*))

(DEFUN ADDING-EQ-P (TX TY CONTEXT)
  (SOME #'(LAMBDA (TRIPLE)
	    (LET ((TERM1 (FIRST TRIPLE))
		  (TERM2 (SECOND TRIPLE))
		  (TERMCONTEXT (THIRD TRIPLE)))
	      (AND (OR (RHET-TERMS:RHET-EQUALP TERM1 TX)
		       (RHET-TERMS:RHET-EQUALP TERM1 TY)
		       ;; note in the case of the first term, we are invoked with a canonical as tx, but a fn-term
		       ;; is in the *current-equalities*. See if the fnt is in the canonical set.
		       (IF (RHET-TERMS:CANONICAL-P TX)
			   (MEMBER TERM1 (RHET-TERMS:HN-FIND TX CONTEXT) :TEST #'RHET-TERMS:RHET-EQUALP))
		       (IF (RHET-TERMS:CANONICAL-P TY)
			   (MEMBER TERM1 (RHET-TERMS:HN-FIND TY CONTEXT) :TEST #'RHET-TERMS:RHET-EQUALP)))
		   (OR (RHET-TERMS:RHET-EQUALP TERM2 TY)
		       (RHET-TERMS:RHET-EQUALP TERM2 TX)
		       (IF (RHET-TERMS:CANONICAL-P TX)
			   (MEMBER TERM2 (RHET-TERMS:HN-FIND TX CONTEXT) :TEST #'RHET-TERMS:RHET-EQUALP))
		       (IF (RHET-TERMS:CANONICAL-P TY)
			   (MEMBER TERM2 (RHET-TERMS:HN-FIND TY CONTEXT) :TEST #'RHET-TERMS:RHET-EQUALP)))
		   (RHET-TERMS:RHET-EQUALP TERMCONTEXT CONTEXT))))
	*CURRENT-EQUALITIES*))

(DEFVAR *ASSERT-EQ-COMMIT-POINT* NIL
  "Holds the first TimeLogic backtracking point in a series of calls to
   ASSERT-EQ-HOOK.  Reset by ASSERT-EQ-COMMIT-HOOK."
)

(DEFUN ASSERT-EQ-HOOK (TX TY CONTEXT)
  "Called by Rhet while trying to assert equality between tx and ty
Return T if Tempos/TimeLogic thinks it's ok"
  (IF (AND *TIME-ITYPE* (RHET-TERMS:TYPECHECK (RHET-TERMS:GET-TYPE TX) *TIME-ITYPE*))	 ; only if we care
      (LET (XINT YINT BTPOINT (TERM-SUPPORT:*CURRENT-CONTEXT* CONTEXT))
	(DECLARE (SPECIAL TERM-SUPPORT:*CURRENT-CONTEXT*))				 ; so we get the context right in case of
											 ; mutual recursion.
	;; If the 'ignore' argument could be different from
	;; term-support:*current-context* then you need to modify
	;; set-timelogic-context (in file tempos-base.lisp) to
	;; take an optional context arg and pass 'ignore' to it.
	(COND ((NULL (SETF XINT (CONVERT-TERM-TO-INTERVAL TX)))
	       (TEMPOS-FAILURE TX :E TY :REL
			       "~A cannot be converted to an interval!" TX)
	       NIL)
	      ((NULL (SETF YINT (CONVERT-TERM-TO-INTERVAL TY)))
	       (TEMPOS-FAILURE TX :E TY :REL
			       "~A cannot be converted to an interval!" TY)
	       NIL)
	      ((ADDING-EQ-P TX TY CONTEXT)
	       T)	;we are recursing from assert-rlink-hook; say it's OK.
	      ((LET ((*CURRENT-EQUALITIES* (CONS (LIST (CONVERT-INTERVAL-TO-TERM XINT)
 						       (CONVERT-INTERVAL-TO-TERM YINT)
						       CONTEXT)
						 (IF (BOUNDP '*CURRENT-EQUALITIES*)
						     *CURRENT-EQUALITIES*))))
		 (DECLARE (SPECIAL *CURRENT-EQUALITIES*))
		 (SETQ BTPOINT (TL:ADD-INTERVAL-CONSTRAINT XINT :E YINT
							 :CONTEXT (GET-TL-CONTEXT CONTEXT))))
	       (UNLESS *ASSERT-EQ-COMMIT-POINT*		; keep the first one
		 (SETQ *ASSERT-EQ-COMMIT-POINT* BTPOINT))
	       T)
	      (T NIL)))
      T)
)

(EVAL-WHEN (LOAD EVAL)
  (PUSHNEW 'ASSERT-EQ-COMMIT-HOOK E-UNIFY:*ADD-EQ-COMMIT-HOOKS*)
  (PUSHNEW 'ASSERT-EQ-UNDO-HOOK E-UNIFY:*ADD-EQ-UNDO-HOOKS*))

(DEFUN ASSERT-EQ-COMMIT-HOOK ()
  "Called by Rhet to commit to a series of (recursive) ADD-EQ's."
  (SETQ *ASSERT-EQ-COMMIT-POINT* NIL)
)

(DEFUN ASSERT-EQ-UNDO-HOOK ()
  "Called by Rhet to renege on series of (recursive) ADD-EQ's.
   This causes TimeLogic to be reset to the state the TL
   database was in when the first ADD-EQ in the series was asserted."
  (WHEN *ASSERT-EQ-COMMIT-POINT*
    (TL:TIMELOGIC-BACKTRACK *ASSERT-EQ-COMMIT-POINT*)
    (SETQ *ASSERT-EQ-COMMIT-POINT* NIL))
)

(EVAL-WHEN (LOAD EVAL)
  (PUSHNEW 'ASSERT-INEQ-HOOK E-UNIFY:*ADD-INEQ-BEFORE-HOOKS*))

(DEFUN ASSERT-INEQ-HOOK (TX TY FOO)
  "Called by Rhet while trying to assert inequality between tx and ty
  Return T if Tempos/TimeLogic thinks it's ok."
  (DECLARE (IGNORE FOO))
  (IF (AND *TIME-ITYPE* (RHET-TERMS:TYPECHECK (RHET-TERMS:GET-TYPE TX) *TIME-ITYPE*))	 ;only if we care
      (LET (XINT YINT)
	(COND ((NULL (SETF XINT (CONVERT-TERM-TO-INTERVAL TX)))
	       (TEMPOS-FAILURE TX '(:A :B :C :D :F :FI :M :MI :O :OI :S :SI) TY :REL
			       "~A cannot be converted to an interval!" TX)
	       NIL)
	      ((NULL (SETF YINT (CONVERT-TERM-TO-INTERVAL TY)))
	       (TEMPOS-FAILURE TX '(:A :B :C :D :F :FI :M :MI :O :OI :S :SI) TY :REL
			       "~A cannot be converted to an interval!" TY)
	       NIL)
	      (T (NOT (NULL (TL:ADD-INTERVAL-CONSTRAINT XINT '(:A :B :C :D :F :FI :M :MI :O :OI :S :SI) YINT))))))
      T)
)

;; interface to Timelogic trace intervals
(eval-when (compile load eval)
  (EXPORT 'TRACE-INTERVAL))
(DEFUN TRACE-INTERVAL (INTERVAL &OPTIONAL WITH-INTERVALS)
  (LET ((*CURRENT-CONTEXT* *DEFAULT-CONTEXT*))
      (DECLARE (SPECIAL *CURRENT-CONTEXT* *DEFAULT-CONTEXT*))
      (FLET ((FORM-TO-INTERVAL (INT)
	       (CONVERT-TERM-TO-INTERVAL (E-UNIFY:CONVERT-TO-FN-TERM INT))))
	(TL::TRACE-INTERVAL 
	  (FORM-TO-INTERVAL INTERVAL)
	  (MAPCAR #'FORM-TO-INTERVAL WITH-INTERVALS)))))




;;;interface to timelogic trace intervals
;;;make things easier to read


#+EXCL
(EVAL-WHEN (LOAD EVAL)
  (EXCL:ADVISE TL::TL-SUBTRACE-1 :BEFORE TRACE-IMPROVEMENT-1 NIL
   (LET ((REL-ARG (GET (SECOND EXCL:ARGLIST) 'FN-TERM)))
     (SETF (SECOND EXCL:ARGLIST) REL-ARG)))

  (EXCL:ADVISE TL::TL-SUBTRACE-2 :BEFORE TRACE-IMPROVEMENT-2 NIL
   (LET ((REL-ARG2 (GET (SECOND EXCL:ARGLIST) 'FN-TERM))
	 (REL-ARG3 (GET (THIRD EXCL:ARGLIST) 'FN-TERM)))
     (SETF (SECOND EXCL:ARGLIST) REL-ARG2)
     (SETF (THIRD EXCL:ARGLIST) REL-ARG3)))

  (EXCL:ADVISE TL::TL-SUBTRACE-3 :BEFORE TRACE-IMPROVEMENT-3 NIL
   (LET ((REL-ARG2 (GET (SECOND EXCL:ARGLIST) 'FN-TERM))
	 (REL-ARG3 (GET (THIRD EXCL:ARGLIST) 'FN-TERM)))
     (SETF (SECOND EXCL:ARGLIST) REL-ARG2)
     (SETF (THIRD EXCL:ARGLIST) REL-ARG3))))
#+SYMBOLICS
(EVAL-WHEN (LOAD EVAL)
  (ZL:::SCL:ADVISE TL::TL-SUBTRACE-1 :BEFORE TEMPOS::TRACE-IMPROVEMENT-1 NIL
   (LET ((REL-ARG (GET (SECOND ZL:::SCL:ARGLIST) 'TEMPOS::FN-TERM)))
     (SETF (SECOND ZL:::SCL:ARGLIST) REL-ARG)))

  (ZL:::SCL:ADVISE TL::TL-SUBTRACE-2 :BEFORE TRACE-IMPROVEMENT-2 NIL
   (LET ((REL-ARG2 (GET (SECOND ZL:::SCL:ARGLIST) 'TEMPOS::FN-TERM))
	 (REL-ARG3 (GET (THIRD ZL:::SCL:ARGLIST) 'TEMPOS::FN-TERM)))
     (SETF (SECOND ZL:::SCL:ARGLIST) REL-ARG2)
     (SETF (THIRD ZL:::SCL:ARGLIST) REL-ARG3)))

  (ZL:::SCL:ADVISE TL::TL-SUBTRACE-3 :BEFORE TRACE-IMPROVEMENT-3 NIL
   (LET ((REL-ARG2 (GET (SECOND ZL:::SCL:ARGLIST) 'TEMPOS::FN-TERM))
	 (REL-ARG3 (GET (THIRD ZL:::SCL:ARGLIST) 'TEMPOS::FN-TERM)))
     (SETF (SECOND ZL:::SCL:ARGLIST) REL-ARG2)
     (SETF (THIRD ZL:::SCL:ARGLIST) REL-ARG3))))
       
;;; End of file Tempos-Base
