;;;  -*- Package: Tempos; Mode: Lisp; Syntax: Common-Lisp; Base: 10. -*-

;;;	File:		Tempos-Axioms.lisp
;;;	Purpose:	axioms to be asserted into Rhet
;;;	Last Edit:	02/18/89 11:05:28
;;;
;;;	Copyright (c) 1988, 1989 Johannes A. G. M. Koomen
;;;	All rights reserved.
;;;
;;;	The TEMPOS system provides a temporal reasoning facility for the logic
;;;	programming system RHETORICAL.  It supplies hooks into the TimeLogic
;;;	package, and defines primitives such as TT (True Throughout) etc.

;;;	These axioms are represented as strings, which are read-from and
;;;	asserted at run-time under Reset-Tempos

(defvar *tempos-basic-axioms* '(

  ;; Sequential intervals

  ;; Interval K is sequentially composed of intervals I and J if I starts K and J finishes K
"
  [[Time-Seqint ?i ?j ?k]
   <Tempos [Time-Starts ?i ?k]
	   [Time-Finishes ?j ?k]]
"
  ; And the reverse, for assertions
"
  [[AND [Time-Starts ?i ?k]
	[Time-Finishes ?j ?k]]
   <Tempos [Time-Seqint ?i ?j ?k] :forward [Time-Seqint ?i ?j ?k]]
"

  ;; Strict sequential intervals

  ;; Interval K is strictly sequentially composed of intervals I and J if I
  ;; starts K, J finishes K, and I meets J
"
  [[Time-Seqint! ?i ?j ?k]
   <Tempos [Time-Seqint ?i ?j ?k]
	   [Time-Meets ?i ?j]]
"
  ;; And the reverse, for assertions
"
  [[AND [Time-Starts ?i ?k]
	[Time-Finishes ?j ?k]
	[Time-Meets ?i ?j]]
   <Tempos [Time-Seqint! ?i ?j ?k] :forward [Time-Seqint! ?i ?j ?k]]
"
  ;; Equals is special:  if Rhet thinks two (presumably interval) terms are
  ;; logically equal, then they are temporally equal too, by defn.

; Not sure how o handle this yet, so punt on it...
;
;  [[Time-Equals ?i*Time ?j*Time]
;   <Tempos [Add-EQ ?i*Time ?j*Time] :forward [Add-EQ ?i*Time ?j*Time]]
;
;  [[Add-EQ ?i ?j]
;   <Tempos [Time-Equals ?i ?j]
;           [Distinct ?i ?j]
;	   :forward [Time-Equals ?i ?j]]

))					; end *Tempos-Basic-Axioms*


(defvar *Tempos-Extended-Axioms* '(

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;; (Part of) Shoham's taxonomy				;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
"
  [[Time-Liquid ?x]
   <Tempos [Time-Upward ?x]
	   [Time-Downward ?x]]
"
"
  [[Time-Concatenable ?x]
   <Tempos [Time-Upward ?x]]
"

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;; The MT (Maximally True) predicate				;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  ;; X is maximally true over interval I if X is true over I and every interval
  ;; J over which X is true is either disjoint from or contained in I

  ;; We expect the user to assert MT.


  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;; The TT (True Throughout) predicate				;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  ;;	1)  X is true over interval I if X is maximally true over I.
"
  [[TT ?i ?x]
   <Tempos [MT ?i ?x]]
"
  ;;	2) X is true over interval I if X is downward hereditary and maximally
  ;;	true over some interval J containing I
"
  [[TT ?i ?x]
   <Tempos [Time-Downward ?x]
	   [Time-Subint! ?i ?j]
	   [TT ?j ?x]]
"
  ;;	3) X is true over interval I if X is upward hereditary and true over all
  ;;	intervals J during I

; The following is incorrect, as all *known* subints of I may not properly cover I
;  [[TT ?i ?x]
;   <Tempos [Time-Upward ?x]
;	   [FORALL ?j [Time-Subint! ?j ?i]
;		      [TT ?j ?x]]]

  ;;	4) X is true over interval I if X is concatenable and there are intervals P and Q
  ;;	such that I is sequentially composed of P and Q and X is true over P and over Q
"
  [[TT ?i ?x]
   <Tempos [Time-Concatenable ?x]
	   [Time-Seqint! ?p ?q ?i]
	   [TT ?p ?x]
	   [TT ?q ?x]]
"

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;; Reasoning about Sequences					;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  ;; Sequences are objects (events, properties, whatever) that have two
  ;; constituents. The predicate [SEQUENCE ?i ?j ?k] indicates that ?k is a
  ;; sequence object with constituents ?i and ?j.  Asserting the occurrence of a
  ;; sequence ?z over interval ?k is done by saying [MT ?k ?z]

  ;; It would have been preferable to use the syntax [MT ?k [SEQ ?x ?y]] which
  ;; would have made it unnecessary to name the sequence, but Rhet won't allow
  ;; destructuring, at least not for the purpose of assertions.  If it did, we
  ;; could have inferred sequence occurrence from the occurrence of the
  ;; constituents, as follows:
  ;;   [[MT ?k [SEQ ?x ?y]]
  ;;	<Tempos [Time-Seqint! ?i ?j ?k]
  ;;		[MT ?i ?x]
  ;;		[MT ?j ?y]]

  ;; Instead, we'll just make do with a FC axiom:  If the occurrence of a
  ;; sequence is asserted, assert the occurrence of its components
"
  [[AND [Time-Seqint! ?i ?j ?k]
	[MT ?i ?x]
	[MT ?j ?y]]
   <Tempos [MT ?k ?z]
	   [SEQUENCE ?x ?y ?z]
	   [Time-Skolem ?i]
	   [Time-Skolem ?j]
	   :forward [MT ?k ?z]]
"

  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
  ;; Reasoning about Recurrence					;;
  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

  ;; Recurrence of an (event, property, whatever) is indicated with the
  ;; RT predicate: [RT ?i ?x] means that ?x repeatedly occurs over interval ?i.
  ;; (For now, `repeatedly' means `infinitely often.'  We need to introduce the
  ;; notion of duration to do any better than that...)

  ;; We expect the user to assert RT statements, from which we can infer the
  ;; existance of individual occurrences.

  ;; If we ever need this, than presumably we haven't been able to prove MT any
  ;; other way.  Expand the RT recursion one step and try again.
"
  [[MT ?i ?x]
   <Tempos [COND ([VAR ?i]			; if it isn't, can't conclude anything anyway
		  [MT-under-RT ?i ?x])]]
"
"
  [[MT-under-RT ?i ?x]
   <Tempos [RT ?k ?x]
	   [Time-Skolem ?khead]
	   [Time-Skolem ?ktail]
	   [ASSERT-FACT [Time-Seqint! ?khead ?ktail ?k]]
	   [ASSERT-FACT [MT ?khead ?x]]
	   [ASSERT-FACT [RT ?ktail ?x]]
	   [Time-Seqint! ?i ?ktail ?k]]
"

  ;; Handle repeated sequences similarly:
"
  [[MT ?l ?v]
   <Tempos [COND ([BOUND ?l] [FAIL])
		 ([MT-under-RT-Seq ?l ?v ?j ?y])
		 ([MT-under-RT-Seq ?i ?x ?l ?v])]]
"
"
  [[MT-under-RT-Seq ?i ?x ?j ?y]
   <Tempos [SEQUENCE ?x ?y ?z]
	   [RT ?k ?z]
	   [Time-Skolem ?khead]
	   [Time-Skolem ?ktail]
	   [ASSERT-FACT [Time-Seqint! ?khead ?ktail ?k]]
	   [ASSERT-FACT [MT ?khead ?z]]
	   [ASSERT-FACT [RT ?ktail ?z]]
	   [Time-Seqint! ?i ?j ?khead]]
"

))					; end of *Tempos-Extended-Axioms*



;;; End of file Tempos-Rhet
