/* ---------------------------------------------------------- 
%   (C)1993 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */

/*
  Clause Indexing Tree Structure

  Value binding/type checking node
    v(Position,			% value identifier
      AtomicNode,
      ListNode,
      FunctNode,
      BoundNode,		% Cases where only bound status is required
      UncondNode)		% Unconditional cases
  Node not specifically depending on one value
    x(Cond,Node,Uncond)
  Leaf node without any further condition
    l(Body)
  Empty node
    e

   AtomicNode:
      a([Atom-Node,...],AtomNode,[Int-Node],IntNode,AtomicNode)
      e

   ListNode: = usual node

   FunctNode:
      f([F/A-Node,...],ObjNode,Node)
      e

   ObjNode
      o([Class-Node],Uncond)
      e

   Cond:
      gp(Name(Pos,...))			% guard builtin pred
      gb(Name(Pos,...),Type)		% guard builtin with output
      gg(gg(F/A,Obj,In),NumOuts)	% guard generic call
      eq(Pos,Pos)			% guard unification
      il(Format,Args,Info)		% inline expansion
*/

make_index_tree([],_V,T,T):- !.
make_index_tree([Cond-Body|CT],V0,T0,T) :- !,
    ( ins_index(Cond,T0,V0,Body,T1) -> true
    ; warning("Clause deleted", []), T1 = T0 ),
    make_index_tree(CT,V0,T1,T).
make_index_tree([otherwise|CT],V0,T0,other(T0,T)) :- !,
    make_index_tree(CT,V0,e,T).
make_index_tree([alternatively|CT],V0,T0,alter(T0,T)) :- !,
    make_index_tree(CT,V0,e,T).

ins_index([],T0,Vars,Body,T) :- ins_uncond(T0,Vars,Body,T).
ins_index([(X;Y)],T0,Vars,Body,T) :- !,
    ins_index(X,T0,Vars,Body,T1), ins_index(Y,T1,Vars,Body,T).
ins_index([Position:Pattern|CondT],T0,Vars,Body,T) :-
    subsumed(Pattern,Position,Vars), !,
    ins_index(CondT,T0,Vars,Body,T).
ins_index([atom(Value):Pattern|CondT],T0,Vars,Body,T) :- !,
    ins_index([Pattern:atom(Value)|CondT],T0,Vars,Body,T).
ins_index([int(Value):Pattern|CondT],T0,Vars,Body,T) :- !,
    ins_index([Pattern:int(Value)|CondT],T0,Vars,Body,T).
ins_index([list(Value):Pattern|CondT],T0,Vars,Body,T) :- !,
    ins_index([Pattern:list(Value)|CondT],T0,Vars,Body,T).
ins_index([functor(Value):Pattern|CondT],T0,Vars,Body,T) :- !,
    ins_index([Pattern:functor(Value)|CondT],T0,Vars,Body,T).
ins_index([Position:Pattern|CondT],T0,Vars0,Body,T) :- !,
    ins_pattern(T0,Pattern,Position,T,Vars0,Vars,T1,T2),
    ins_index(CondT,T1,Vars,Body,T2).
ins_index([gp(C)|CondT],T0,Vars0,Body,T) :- !,
    ins_gb(T0,gp(C),T,T1,T2),
    ins_index(CondT,T1,Vars0,Body,T2).
ins_index([gb(C,OT)|CondT],T0,Vars,Body,T) :- !,
    ins_gb(T0,gb(C,OT),T,T1,T2),
    ins_index(CondT,T1,[gb(C)=OT|Vars],Body,T2).
ins_index([gg(C,K)|CondT],T0,Vars,Body,T) :-
    ins_gb(T0,gg(C,K),T,T1,T2),
    ins_index(CondT,T1,Vars,Body,T2).
ins_index([il(F,A,I)|CondT],T0,Vars,Body,T) :-
    ins_gb(T0,il(F,A,I),T,T1,T2),
    ins_index(CondT,T1,Vars,Body,T2).

/* Leaf Case */
ins_uncond(e,Vars,Body0,l(Body)) :- !,
    ins_body(Body0,Vars,Body).
ins_uncond(_Node,_Vars,_Body,_Out) :- ins_redundant.

ins_body([],_Vars,[]) :- !.
ins_body([Goal|Rest0],Vars,[Goal|Rest]) :- ins_body(Rest0,Vars,Rest).

/* Normal Pattern */

ins_pattern(e,Pat,Pos,NewN,Vars,[Pos=Pat|Vars],T0,T) :-
    ins_value_node(Pat,Pos,e,e,e,e,e,NewN,T0,T).
ins_pattern(v(Pos,A,L,F,B,U),Pat,Pos,NewN,Vars,[Pos=Pat|Vars],T0,T) :- !,
    ins_value_node(Pat,Pos,A,L,F,B,U,NewN,T0,T).
ins_pattern(v(Pos,A,L,F,B,U),Pos,Pat,NewN,Vars,[Pos=Pat|Vars],T0,T) :- !,
    ins_value_node(Pat,Pos,A,L,F,B,U,NewN,T0,T).
ins_pattern(v(Pos0,A,L,F,B,U0),Pat,Pos,v(Pos0,A,L,F,B,U),Vars0,Vars,T0,T) :-
    ins_pattern(U0,Pat,Pos,U,Vars0,Vars,T0,T).
ins_pattern(x(Cond,Node,U0),Pat,Pos,x(Cond,Node,U),Vars0,Vars,T0,T) :-
    ins_pattern(U0,Pat,Pos,U,Vars0,Vars,T0,T).
ins_pattern(l(_Body),_,_,_,_,_,_,_) :- ins_redundant.

ins_value_node(X,Pos,A0,L,F,B,U,v(Pos,A,L,F,B,U),T0,T) :-
    atomic_pattern(X), !,
    ins_atomic_node(A0,X,A,T0,T).
ins_value_node(list,Pos,A,L0,F,B,U,v(Pos,A,L,F,B,U),L0,L) :- !.
ins_value_node(X,Pos,A,L,F0,B,U,v(Pos,A,L,F,B,U),T0,T) :-
    functor_pattern(X), !,
    ins_funct_node(F0,X,F,T0,T).
ins_value_node(bound,Pos,A,L,F,B0,U,v(Pos,A,L,F,B,U),B0,B).
ins_value_node(Pos1,Pos,A,L,F,B0,U,v(Pos,A,L,F,B,U),T0,T) :-
    ins_eq(B0,Pos1,Pos,B,T0,T).

atomic_pattern(atom(_)).
atomic_pattern(int(_)).
atomic_pattern(atom).
atomic_pattern(int).
atomic_pattern(atomic).

functor_pattern(functor).
functor_pattern(functor(_,_)).
functor_pattern(object).
functor_pattern(object(_)).

ins_eq(e,Pos1,Pos2,v(Pos1,e,e,e,x(eq(Pos1,Pos2),T,e),e),e,T).
ins_eq(v(Pos,A,L,F,B0,U),Pos,Pos1,v(Pos,A,L,F,B,U),T0,T) :- !,
    ins_eq(B0,Pos,Pos1,B,T0,T).
ins_eq(v(Pos0,A,L,F,B,U0),Pos1,Pos2,v(Pos0,A,L,F,B,U),T0,T) :-
    ins_eq(U0,Pos1,Pos2,U,T0,T).
ins_eq(x(Cond,Node,U0),Pos1,Pos2,x(Cond,Node,U),T0,T) :-
    ins_eq(U0,Pos1,Pos2,U,T0,T).
ins_eq(l(_Body),_Pos1,_Pos2,_Node,_T0,_T) :- ins_redundant.

/* Atomic and Functor Cases */

ins_atomic_node(e,atomic,	   a([],e,[],e,T),e,T) :- !.
ins_atomic_node(e,atom,		   a([],T,[],e,e),e,T) :- !.
ins_atomic_node(e,int,		   a([],e,[],T,e),e,T) :- !.
ins_atomic_node(e,atom(A),	   a([A-T],e,[],e,e),e,T) :- !.
ins_atomic_node(e,int(N),	   a([],e,[N-T],e,e),e,T) :- !.
ins_atomic_node(a(AC,AN,IC,IN,EC0),atomic, a(AC,AN,IC,IN,EC),EC0,EC) :- !.
ins_atomic_node(a(AC,AN0,IC,IN,EC),atom,   a(AC,AN,IC,IN,EC),AN0,AN) :- !.
ins_atomic_node(a(AC,AN,IC,IN0,EC),int,	   a(AC,AN,IC,IN,EC),IN0,IN) :- !.
ins_atomic_node(a(AC0,AN,IC,IN,EC),atom(A),a(AC,AN,IC,IN,EC),T0,T) :- !,
    ins_merge_case(AC0,A,AC,T0,T).
ins_atomic_node(a(AC,AN,IC0,IN,EC),int(N), a(AC,AN,IC,IN,EC),T0,T) :- !,
    ins_merge_case(IC0,N,IC,T0,T).

ins_funct_node(e,functor,	f([],e,T),e,T) :- !.
ins_funct_node(e,functor(F,A),	f([F/A-T],e,e),e,T) :- !.
ins_funct_node(e,object,		f([],o([],T),e),e,T) :- !.
ins_funct_node(e,object(Class),		f([],o([Class-T],e),e),e,T) :- !.
ins_funct_node(f(FC,FO,FN0),functor,	f(FC,FO,FN),FN0,FN) :- !.
ins_funct_node(f(FC0,FO,FN),functor(F,A),f(FC,FO,FN),T0,T) :-
    ins_merge_case(FC0,F/A,FC,T0,T).
ins_funct_node(f(FC,o(O,FO0),FN),object,f(FC,o(O,FO),FN),FO0,FO) :- !.
ins_funct_node(f(FC,o(O0,U),FN),object(Class),f(FC,o(O,U),FN),FO0,FO) :- !,
    ins_merge_case(O0,Class,O,FO0,FO).

ins_merge_case([],V,[V-T],e,T).
ins_merge_case([V-T0|Rest],V,[V-T|Rest],T0,T) :- !.
ins_merge_case([V0-X|Rest],V,[V0-X|Tail],T0,T) :- !,
    ins_merge_case(Rest,V,Tail,T0,T).

/* Guard Builtin */

ins_gb(e,GB,x(GB,T,e),e,T).
ins_gb(v(Pos,A,L,F,B,U0),GB,v(Pos,A,L,F,B,U),T0,T) :- ins_gb(U0,GB,U,T0,T).
ins_gb(x(GB,T0,U),GB,x(GB,T,U),T0,T) :- !.
ins_gb(x(Cond,Node,U0),GB,x(Cond,Node,U),T0,T) :- ins_gb(U0,GB,U,T0,T).
ins_gb(l(_Body),_GB,_T,_T1,_T2) :- ins_redundant.

/* Type Information Handling */
subsumed(Pattern,Pos,_Vars) :- subsumed_type(Pos,Pattern), !.
subsumed(Pattern,Pos,Vars) :-
    assoc(Vars,Pos,Position), !,
    subsumed_type(Position,Pattern).

subsumed_type(Type,Type) :- !.
subsumed_type(_,any) :- !.
subsumed_type(atom(_),X) :- subsumed_type(atom,X).
subsumed_type(int(_),X) :- subsumed_type(int,X).
subsumed_type(list(_),X) :- subsumed_type(list,X).
subsumed_type(functor(_),X) :- subsumed_type(functor,X).
subsumed_type(functor(_,_),X) :- subsumed_type(functor,X).
subsumed_type(atom,atomic) :- !.
subsumed_type(atom,bound).
subsumed_type(int,atomic) :- !.
subsumed_type(int,bound).
subsumed_type(atomic,bound).
subsumed_type(functor,bound).

/* Warning */
ins_redundant :- warning("Redundant clause", []), fail.
