/* ---------------------------------------------------------- 
%   (C)1993 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */

/*
  struct.h

  Definition of data structures for KL1/C system

  fix for DEC alpha
*/

#include <klic/param.h>

/****************************************
  UNIVERSAL ONE-WORD DATA

  All the one-word data is represented as type "q".
  This pointer type will always be used after type conversion.
  Using "struct gazonk *" type is for reducing confusion.
****************************************/

typedef struct gazonk *q;

/****************************************
  POINTER TAGS

  [Pointer Tags]
  Lower 2 bits of pointers are used as tags in the following way.

     ...00: Variable Reference
     ...01: Pointer to a Cons Cell
     ...10: Atomic (integer, symbolic atom, etc)
     ...11: Pointer to a Functor Structure

  Variables without suspended goals are represented as
    self-referencing variables.
  Variables _with_ suspended goals are represented as a pointer to
    a cell which contains a special atom.

  [Tags for Atomic Data]
     ..0010: Integer (integers have 28 bits including sign)
     ..0110: Symbolic Atom
     ..1010: Reserved
     ..1110: Reserved
****************************************/

#define VALUEMASK	(~3L)
#define ADDRESSMASK	(~3L)

#define PTAGBITS	2L
#define PTAGMASK	0x3
#define	VARREF		0x0
#define CONS		0x1
#define ATOMIC		0x2
#define FUNCTOR		0x3

#define STRUCT		0x1
#define FUNCTNOTCONS	0x2
#define ATOMNOTREF	0x2

#define ptagof(x)	((unsigned long)(x) & PTAGMASK)
#define addressof(x)	((unsigned long)(x) & ADDRESSMASK)

#define isatomic(x)	(ptagof(x) == ATOMIC)
#define iscons(x)	(ptagof(x) == CONS)
#define isfunctor(x)	(ptagof(x) == FUNCTOR)
#define isref(x)	(ptagof(x) == VARREF)

#define isstruct(x)	((unsigned long)(x) & STRUCT)
#define functnotcons(x)	((unsigned long)(x) & FUNCTNOTCONS)
#define atomicnotref(x)	((unsigned long)(x) & ATOMNOTREF)

#define ATAGBITS	4
#define ATAGMASK	0xFL
#define INT		(ATOMIC+0x0)
#define SYM		(ATOMIC+0x4)

#define atagof(x)	((unsigned long)(x) & ATAGMASK)

#define isint(x)	(atagof(x) == INT)
#define issym(x)	(atagof(x) == SYM)

/****************************************
  DATA STRUCTURES

  All the data structures are to be allocated at 4 byte boundaries.
****************************************/

struct functor {
  q functor;			/* principal functor as atomic q object */
  q args[1];			/* arguments */
};

#define functor_of(s)		(functorp(s)->functor)
#define arg(s, k)		(functorp(s)->args[k])

struct cons {
  q cdr, car;
};

#define car_of(x)		(consp(x)->car)
#define cdr_of(x)		(consp(x)->cdr)

#define pcar_of(x)              (makeref(consp(x)+1))
#define pcdr_of(x)              (makeref(consp(x)))

/****************************************
  GOAL MANAGEMENT STRUCTURES
****************************************/

/* Predicate structures are constant structures outside of heap */
/* They are pointed to by atomic tags and treated as atomic in GC */

typedef void * (*module)();

struct predicate {
  module func;			/* pointer to function */
  unsigned short int pred;	/* predicate number */
  unsigned short int arity;	/* number of args */
};

/* Goal records */

/* Suspension records are referenced via pointers with variable tag */
/* They are recognized by its value being a special atom */
/* When a variable with suspended goals is instantiated, */
/*   the first word of this suspension record is used as the cell */

struct goalrec {
  struct goalrec *next;		/* pointer to next suspended goal */
  Const struct predicate *pred;	/* predicate descriptor of the goal */
  q args[6];			/* arguments; number 6 is meaningless */
};

/* Priority queue is currently implemented as a linear list of */
/* priority queue records, in descending order of precedence */

struct prioqrec {
  struct prioqrec *next;
  long prio;			/* signed, to use -1 as sentinel */
  struct goalrec *q;
};

/*
  Stuffs for suspension
*/
struct susprec {
  q marker;
  q next;
  q back; /* back points the last record */
  q reserved;
};

struct shook {
  struct goalrec *goals;        /* pointer to goal record */
  q next; /* next suspension structure and tag */
};

struct mhook {
  struct goalrec *goals;
  q next;
  q back;         /* back pointer */
  struct mhook *pal; /* pointer to the mhook record which is suspended
		     simultaniously */
};

/****************************************
  DATA MANIPULATION MACROS
****************************************/

/* Type conversion */

#define refp(x)		((q *)((unsigned long)(x) - VARREF))
#define consp(x)	((struct cons *)((unsigned long)(x) - CONS))
#define functorp(x)	((struct functor *)((unsigned long)(x) - FUNCTOR))
#define goalp(x)	((struct goalrec *)((unsigned long)(x) - ATOMIC))
#define suspp(x)	((struct susprec *)((unsigned long)(x) - VARREF))
#define predp(x)	((struct predicate *)((unsigned long)(x) - ATOMIC))

/* Variables */

#define makeref(x)	(q) ((unsigned long)(x) + VARREF)
#define derefone(x)	(*refp(x))

/* Atomic Values */

#define intval(x)	((long)(x) >> ATAGBITS)
#define symval(x)	((unsigned long)(x) >> ATAGBITS)

#define makeint(n)	((q)(((long)(n) << ATAGBITS) + INT))
#define makecint(n)	((long)makeint(n))
#define makesym(n)	((q)(((unsigned long)(n) << ATAGBITS) + SYM))

#define eqatom(x, y)	((x) == (y))

/* for non-heap addr */
#define makeatomic(adr) ((q)((unsigned long)(adr) + ATOMIC))

/* for principal functors */
#define makepf(x)	makesym(x)

/* Conses */
#define makecons(x)	((q)((unsigned long)(x) + CONS))

/* Functors */

#define makefunctor(x) 	((q)((unsigned long)(x) + FUNCTOR))

Extern struct funcdesc {
  q name;			/* atom */
  unsigned long arity;		/* number of args */
} functab[1];

/* Structures for linkage */

struct ext_atom {
  char *name;			/* name string */
  q value;			/* its value with symbol tag */
};

struct ext_funct {
  char *name;			/* name string */
  long arity;			/* arity */
  q value;			/* functor id with atomic tag */
};

struct ext_pred {
  char *module;			/* module name string */
  char *name;			/* predicate name */
  long arity;			/* arity */
  q value;			/* pred id with atomic tag */
};

/* Special Atoms */

#define NILATOM		makesym(0)
#define PERIODATOM	makesym(1)

/**********************
  Suspension staffs
***********************/
/* Tags for single/multiple suspension */
#define SSUSP 0
#define MSUSP 1
#define GOBJ  2
#define SUSPTAG 3L
#define SUSPMASK (~SUSPTAG)

#define makesusprec(rec) { susprecalloc(rec); }
#define initsusprec(susp, ref, hook) { \
  (susp)->marker = makeref(ref); \
  (susp)->next = (q)MSUSP; \
  setnexthook((susp), (hook)); \
  settailhook((susp), (hook)); \
  derefone(ref) = makeref(susp); \
}

#define makeshook(rec, pointer) \
  makehook(rec, shookalloc, pointer, SSUSP)
#define makemhook(rec, pointer) \
  makehook(rec, mhookalloc, pointer, MSUSP)
#define makegeneric(rec, pointer) \
  makehook(((struct mhook *)(rec)), mhookalloc, pointer, GOBJ)
#define collectgeneric(s) collectmhook((struct mhook *)(s))

#define makehook(rec, alloc, pointer, tag) \
  {  \
    alloc(rec); \
    /*printhooks((rec), (tag));*/ \
    (rec)->next = (q)(((long)(((q*)(pointer))+1))|(tag)); \
  }

#ifdef DEBUG
#define printhooks(rec, tag) \
    printf("allocated %s : %x\n", (tag == SSUSP? "shook": "mhook"), rec)
#else
#define printhooks(rec, tag)
#endif


#define setnexthook(rec, nextrec) { \
  (rec)->next = (q)((((long)(&((nextrec)->next))) & SUSPMASK) | \
    (((long)((rec)->next)) & SUSPTAG)); }

#define settailhook(rec, tailrec) { (rec)->back = (q)(((q*)(tailrec))+1); }
#define setbackhook(rec, prevrec) { (rec)->back = (q)(((q*)(prevrec))+1); }

#define makemergercommon(rec) { susprecalloc((struct shook *)rec); }


#define hooktagword(rec) (*(((q*)(rec)) + 1))

#define hooktag(rec) (((long)hooktagword(rec)) & SUSPTAG)

#define hookpointer(rec) (((long)hooktagword(rec)) & SUSPMASK)

#define getnexthook(rec) ((q*)hookpointer(rec) - 1)
#define getprevhook(rec) ((q*)(((long)((struct mhook *)(rec)->back)) \
			  & SUSPMASK) - 1)

#define hooktagp(rec) (((long)hooktagwordp(rec)) & SUSPTAG)

#define hookpoinetrp(rec) (((long)hooktagwordp(rec)) & SUSPMASK)

#define changetagp(rec, newtag) \
  change_next(rec, newtag, SUSPTAG)
#define changepointerp(rec, newpointer) \
  change_next(rec, newpointer, SUSPMASK)

#define change_next(rec, new, mask) \
{ ((long)hooktagwordp(rec)) = ((((long)hooktagwordp(rec)) & (mask)) | \
			      ((long)(new)); }

#define shookp(rec) ((struct shook *)(rec))
#define mhookp(rec) ((struct mhook *)(rec))
#define mergerp(rec) ((struct merger *)(rec))

#define isshook(rec) ((long)hooktag(rec) == SSUSP)
#define ismhook(rec) ((long)hooktag(rec) == MSUSP)
#define ismerger(rec) ((long)hooktag(rec) == MERGER)

/* Global Variables */

#define queue		(glbl->queue0)
#define prioq		(glbl->prioq0)
#define current_prio	(glbl->current_prio0)
#define top_prio	(glbl->top_prio0)
#define heapp		(glbl->heapp0)
#define heaptop		(glbl->heaptop0)
#define heaplimit	(glbl->heaplimit0)
#define real_heaplimit	(glbl->real_heaplimit0)
#define heapbottom	(glbl->heapbottom0)
#define heapend		(glbl->heapend0)
#define heapsize	(glbl->heapsize0)
#define totalheapsize	(glbl->totalheapsize0)
#define halfheapsize	(glbl->halfheapsize0)
#define incrementsize	(glbl->incrementsize0)
#define new_space_top	(glbl->new_space_top0)
#define old_space_top	(glbl->old_space_top0)
#define gcstack		(glbl->gcstack0)
#define gcsp		(glbl->gcsp0)
#define gcmax		(glbl->gcmax0)
#define gcstack_size	(glbl->gcstack_size0)
#define reasons		(glbl->reasons0)
#ifdef CSUSPS
#define suspensions	(glbl->suspensions0)
#define	resumes		(glbl->resumes0)
#endif

Extern struct global_variables {
  /* First four items are accessed frequently */
  q *heapp0;
  q *heaplimit0;
  struct goalrec *queue0;	/* queue for current priority */
  unsigned long current_prio0;	/* current priority */
  unsigned long top_prio0;	/* highest priority with ready goal */
  /* The rest are not accessed as often */
  struct prioqrec prioq0;	/* priority queue head */
  q *heaptop0;
  q *heapbottom0;
  q *heapend0;
  q *real_heaplimit0;
  unsigned long heapsize0, incrementsize0, totalheapsize0;
  unsigned long halfheapsize0;
  q *new_space_top0, *old_space_top0;
  q **gcstack0;
  q **gcsp0;
  q **gcmax0;
  unsigned long gcstack_size0;
  q reasons0[MAXSUSPENSION];	/* suspension reasons */
#ifdef CSUSPS
  unsigned long suspensions0, resumes0;
#endif
} globals;

/*
  memory block outside of heap.
*/

Extern struct shook *shookreclist;
Extern struct mhook *mhookreclist;

Extern long higher_priority_goal;

/*
  variable for mutable objects
  */

Extern q suspension_reason;
Extern q rest_of_stream;
