/* ---------------------------------------------------------- 
%   (C)1993 Institute for New Generation Computer Technology 
%	(Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
write_object(Obj,Info) :- write_object(Obj,Info,0).

write_object([],_,_):- !.
write_object([H|T],Info,H0) :-
    write_one(H,Info,H0,Heap), !,
    write_object(T,Info,Heap).
write_object([H|_],_,_) :-
    warning("Failed to write out object: ~w", [H]),
    fail.

/* Labels */
write_one(label(Lab),Info,H,H) :-
    klicformat(" ~l:~n",[Info,Lab]).
write_one(begin_body(_,Lab,_),Info,H,H) :-
    klicformat(" ~l:~n",[Info,Lab]).

/* Register transfer */
write_one(move(X,X),_,H,H) :- !.
write_one(move(To,From),_,H,H) :-
    klicformat("  ~r = ~r;~n",[To,From]).

/* Element access */
write_one(get_elem(To,From,Pos),_,H,H) :-
    klicformat("  ~r = ~e;~n",[To,From,Pos]).

/* Indexing */
write_one(deref(X,Lloop,Lsusp),Info,H,H) :-
    klicformat("  deref_and_jump(~r,~l);~n",[X,Info,Lloop]),
    klicformat("  *reasonp++ =  ~r;~n",[X]),
    klicformat("  goto ~l;~n",[Info,Lsusp]).
write_one(push_reason(X,_),_,H,H) :- !,
    klicformat("  *reasonp++ =  ~r;~n",[X]).
write_one(goto(Label),Info,H,H) :-
    klicformat("  goto ~l;~n",[Info,Label]).
write_one(sw_tag(X),_,H,H) :-
    klicformat("  switch (ptagof(~r)) {~n",[X]).
write_one(sw_atag(X),_,H,H) :-
    klicformat("  switch (atagof(~r)) {~n",[X]).
write_one(sw_sym(X),_,H,H) :-
    klicformat("  switch (symval(~r)) {~n",[X]).
write_one(sw_int(X),_,H,H) :-
    klicformat("  switch ((unsigned long)~r) {~n",[X]).
write_one(sw_funct(X),_,H,H) :-
    klicformat("  switch (symval(functor_of(~r))) {~n",[X]).
write_one(case_label(S),_,H,H) :-
    klicformat(" case ~s:~n",[S]).
write_one(case_sym(Atom),_,H,H) :-
    klicformat(" case symval(~a):~n",[Atom]).
write_one(case_int(Atom),_,H,H) :-
    klicformat(" case (unsigned long) ~a:~n",[Atom]).
write_one(case_funct(Funct),_,H,H) :-
    klicformat(" case ~f:~n",[Funct]).
write_one(case_default,_,H,H) :-
    klicformat(" default:~n",[]).
write_one(end_sw,_,H,H) :-
    klicformat("  };~n",[]).
write_one(eq(X,Y,Lab,FailLab,SuspLab),Info,H,H) :-
    klicformat("  if_equal(~r, ~r, ~l, ~l, ~l);~n",
    	[X,Y,Info,Lab,Info,FailLab,Info,SuspLab]).
write_one(if_not_eq(X,Y,Lab),Info,H,H) :-
    klicformat("  if_not_equal(~r, ~r, ~l);~n", [X,Y,Info,Lab]).
write_one(if_int(X,ElseLab),Info,H,H) :-
    klicformat("  if (isint(~r)) goto ~l;~n",[X,Info,ElseLab]).
write_one(if_not_int(X,ElseLab),Info,H,H) :-
    klicformat("  if (!isint(~r)) goto ~l;~n",[X,Info,ElseLab]).
write_one(if_list(X,ElseLab),Info,H,H) :-
    klicformat("  if (iscons(~r)) goto ~l;~n",[X,Info,ElseLab]).
write_one(if_not_list(X,ElseLab),Info,H,H) :-
    klicformat("  if (!iscons(~r)) goto ~l;~n",[X,Info,ElseLab]).
write_one(if_funct(X,ElseLab),Info,H,H) :-
    klicformat("  if (isfunctor(~r)) goto ~l;~n",[X,Info,ElseLab]).
write_one(if_not_funct(X,ElseLab),Info,H,H) :-
    klicformat("  if (!isfunctor(~r)) goto ~l;~n",[X,Info,ElseLab]).
write_one(if_not_gobj(X,ElseLab),Info,H,H) :-
    klicformat("  if (!isgobj(~r)) goto ~l;~n",[X,Info,ElseLab]).
write_one(if_ref(X,ElseLab),Info,H,H) :-
    klicformat("  if (isref(~r)) goto ~l;~n",[X,Info,ElseLab]).
write_one(if_not_ref(X,ElseLab),Info,H,H) :-
    klicformat("  if (!isref(~r)) goto ~l;~n",[X,Info,ElseLab]).
write_one(if_funct_not(X,F,ElseLab),Info,H,H) :-
    klicformat("  if (functor_of(~r) != makesym(~f)) goto ~l;~n",
	[X,F,Info,ElseLab]).
write_one(if_not(X,Value,ElseLab),Info,H,H) :-
    klicformat("  if (~r != ~k) goto ~l;~n",[X,Value,Info,ElseLab]).

/* Inline expansion */

write_one(inline(Format,Args,ElseLab),Info,H,H) :-
    klicformat("  ",[]),
    write_inline(Format,Args,Info,ElseLab).

/* Constant loading */
write_one(load_atom(X,A),_,H,H) :-
    klicformat("  ~r = ~a;~n",[X,A]).
write_one(load_const(X,cons(Offset)),_,H,H) :-
    klicformat("  ~r = makecons(constants+~d);~n",[X,Offset]).
write_one(load_const(X,funct(Offset)),_,H,H) :-
    klicformat("  ~r = makefunctor(constants+~d);~n",[X,Offset]).

/* Variable loading */
write_one(load_newvar(V),_,H0,H) :-
    klicformat("  allocp[~d] = ~r = makeref(&allocp[~d]);~n",[H0,V,H0]),
    H is H0+1.

/* Allocation */
write_one(make_cons(X),_,H,H) :-
    klicformat("  ~r = makecons(&allocp[~d]);~n",[X,H]).
write_one(make_functor(X),_,H,H) :-
    klicformat("  ~r = makefunctor(&allocp[~d]);~n",[X,H]).
write_one(alloc_functor_id(F,A),_,H,H) :-
    klicformat("  allocp[~d] = makesym(~f);~n",[H,F/A]).
write_one(alloc_atomic(A,K),_,H,H) :-
    Off is H+K,
    klicformat("  allocp[~d] = ~a;~n",[Off,A]).
write_one(alloc_value(X,K),_,H,H) :-
    Off is H+K,
    klicformat("  allocp[~d] = ~r;~n",[Off,X]).
write_one(alloc_var(X,K),_,H,H) :-
    Off is H+K,
    klicformat("  allocp[~d] = ~r = makeref(&allocp[~d]);~n",[Off,X,Off]).
write_one(alloc_elem(X,K,KK),_,H,H) :-
    Off is H+KK,
    klicformat("  allocp[~d] = ~e;~n",[Off,X,K]).
write_one(alloc_const(cons(C),K),_,H,H) :-
    Off is H+K,
    klicformat("  allocp[~d] = makecons(constants+~d);~n",[Off,C]).
write_one(alloc_const(funct(C),K),_,H,H) :-
    Off is H+K,
    klicformat("  allocp[~d] = makefunctor(constants+~d);~n",[Off,C]).
write_one(make_space(N),_,H0,H) :-
    H is H0+N.

/* Unificaiton */
write_one(unify_atom(X,A,Q),_,H,H) :-
    klicformat("  unify_value(~r, ~a, ~Q);~n",[X,A,Q]).
write_one(unify(X,Y,Q),_,H,H) :-
    klicformat("  unify(~k, ~k, ~Q);~n",[X,Y,Q]).
write_one(unify_value(X,Y,Q),_,H,H) :-
    klicformat("  unify_value(~k, ~k, ~Q);~n",[X,Y,Q]).

/* Execution control */
write_one(execute(Name,Arity,Susp,Q),_,H0,0) :-
    ( Q=qp -> true; klicformat("  qp = ~Q;~n",[Q]) ),
    ( H0 =:= 0 -> true
    ;  klicformat("  allocp += ~d;~n",[H0]) ),
    ( Susp = 0 ->
      klicformat("  execute(~q_~d_0);~n",[Name,Arity])
    ; klicformat("  execute(~q_~d_clear_reason);~n",[Name,Arity]) ),
    klicformat("  goto ~q_~d_ext_interrupt;~n",[Name,Arity]).
write_one(proceed(Q),_,H0,0) :- !,
    ( Q=qp -> true; klicformat("  qp = ~Q;~n",[Q]) ),
    ( H0 =:= 0, !; klicformat("  allocp += ~d;~n",[H0]) ),
    klicformat("  proceed();~n",[]).
write_one(set_pred(Name,Arity),info(M/_/_),H,H) :-
    H1 is H+1,
    klicformat("  allocp[~d] = (q)(&~p);~n",[H1,M/Name/Arity]).
write_one(set_ext_pred(Module,Name,Arity),_,H,H) :-
    H1 is H+1,
    klicformat("  allocp[~d] = (q)(&~p);~n",[H1,Module/Name/Arity]).
write_one(push_goal(Q0,H),_,H,H) :-
    klicformat("  allocp[~d] = (q)~Q;~n",[H,Q0]).
write_one(enq_at_lower_prio(Q0,Q),_,H,H) :-
    klicformat("  enqueue_at_lower_priority(~Q,~Q);~n",[Q0,Q]).
write_one(enq_at_prio_no_check(P,Q0,Q),_,H,H) :-
    klicformat("  enqueue_at_priority_no_check(~k,~Q,~Q);~n",[P,Q0,Q]).
write_one(enq_at_prio(P,Q0,Q),_,H,H) :-
    klicformat("  enqueue_at_priority(~k,~Q,~Q);~n",[P,Q0,Q]).
write_one(set_value(P,X),_,H,H) :-
    PP is P+H+2,
    klicformat("  allocp[~d] = ~r;~n",[PP,X]).
write_one(set_atomic(P,A),_,H,H) :-
    PP is P+H+2,
    klicformat("  allocp[~d] = ~a;~n",[PP,A]).
write_one(set_newvar(P,X),_,H,H) :-
    PP is P+H+2,
    klicformat("  allocp[~d] = ~r = makeref(&allocp[~d]);~n",[PP,X,PP]).
write_one(set_elem(P,X,K),_,H,H) :-
    PP is P+H+2,
    klicformat("  allocp[~d] = ~e;~n",[PP,X,K]).
write_one(set_const(P,cons(C)),_,H,H) :-
    PP is P+H+2,
    klicformat("  allocp[~d] = makecons(constants+~d);~n",[PP,C]).
write_one(set_const(P,funct(C)),_,H,H) :-
    PP is P+H+2,
    klicformat("  allocp[~d] = makefunctor(constants+~d);~n",[PP,C]).

/* Builtin */
write_one(bblt([],NF,Args,OArgs),_,H,H) :- !,
    append(Args,OArgs,AArgs),
    klicformat("  ~w_no_check(~,);~n",[NF,AArgs]).
write_one(bblt([1],NF,Args,OArgs),_,H,H) :- length(Args,1), !,
    append(Args,OArgs,AArgs),
    klicformat("  ~w(~,);~n",[NF,AArgs]).
write_one(bblt([1],NF,Args,OArgs),_,H,H) :- length(Args,2), !,
    append(Args,OArgs,AArgs),
    klicformat("  ~w_no_check_y(~,);~n",[NF,AArgs]).
write_one(bblt([2],NF,Args,OArgs),_,H,H) :- !,
    append(Args,OArgs,AArgs),
    klicformat("  ~w_no_check_x(~,);~n",[NF,AArgs]).
write_one(bblt(_,NF,Args,OArgs),_,H,H) :- !,
    append(Args,OArgs,AArgs),
    klicformat("  ~w(~,);~n",[NF,AArgs]).

/* Guard builtin */
write_one(gblt(F,_,Args,Value,Lint),Info,H,H) :-
    klicformat("  ~w(~,,~k,~l);~n",[F,Args,Value,Info,Lint]).
write_one(gblt_pred(F,_,Args,Lint),Info,H,H) :-
    klicformat("  ~w(~,,~l);~n",[F,Args,Info,Lint]).

write_one(otherwise,Info,H,H) :-
    klicformat("  otherwise(~l);~n",[Info,interrupt]).
write_one(alternatively,_,H,H) :-
    klicformat("  alternatively;~n",[]).

write_one(equiv(R1,R0),_,H,H) :-
    klicformat("  ~r = ~r;~n",[R0,R1]).

/* Generic Objects */
write_one(new_generic(Class,Arity,Obj),_,H,H) :-
    klicformat("  new_generic(~q_g_new, ~d, ~r);~n", [Class,Arity,Obj]).
write_one(call_generic(Obj,FA,Q),_,H,H) :-
    klicformat("  call_generic(~k, ~f, ~Q);~n",[Obj,FA,Q]).
write_one(guard_generic(Obj,FA,Nin,Lint),Info,H,H) :-
    klicformat("  guard_generic(~k, ~f, ~w, ~l);~n",
	[Obj,FA,Nin,Info,Lint]).
write_one(store_generic_arg(X,K),_,H,H) :-
    klicformat("  generic_arg[~d] = ~k;~n",[K,X]).
write_one(load_generic_arg(R,K),_,H,H) :-
    klicformat("  ~r = generic_arg[~d];~n",[R,K]).

/* Formatting routine */
klicformat(Stream,Format,Args) :-
    telling(OldStream), tell(Stream),
    klicformat(Format,Args), tell(OldStream).

klicformat([],_Args).
klicformat([0'~,C|T],Args) :- !,
    format_special(C,Args,ArgsT),
    klicformat(T,ArgsT).
klicformat([C|T],Args) :- put(C), klicformat(T,Args).

format_special(0'a,[A|T],T) :- !, write_atom(A).
format_special(0'c,[N|T],T) :- !, put(N).
format_special(0'd,[N|T],T) :- !, write(N).
format_special(0'e,[X,E|T],T) :- !, write_elem_access(X,E).
format_special(0'f,[F/A|T],T) :- !, write_funct_name(F,A).
format_special(0'k,[K|T],T) :- !, write_param(K).
format_special(0'l,[Info,L|T],T) :- !, write_lab(Info,L).
format_special(0'n,T,T) :- !, nl.
format_special(0'p,[M/N/A|T],T) :- !, write_pred_name(M,N,A).
format_special(0'q,[A|T],T) :- !,
    name(A,AN), make_atom_name_string(AN, AS), write_string(AS).
format_special(0'r,[R|T],T) :- !, write_reg(R).
format_special(0's,[S|T],T) :- !, write_string(S).
format_special(0'w,[X|T],T) :- !, write(X).
format_special(0',,[L|T],T) :- !, write_params(L).
format_special(0'Q,[Q|T],T) :- !, write_qp(Q).
format_special(0'~,T,T) :- !, put(0'~).
format_special(X,T,T) :- report_error("unknown format char ~c", [X]).

write_params([]):- !.
write_params([H]):- !, write_param(H).
write_params([H|T]):- write_param(H), put(0',), write_params(T).

write_param(int(Int)):- !, klicformat("makeint(~dL)",[Int]).
write_param(atom(Atom)):- !, write_atom(Atom).
write_param(r(E)):- !, write_reg(E).
write_param(arg(Reg,E)):- !, write_elem_access(Reg,E).

write_string([]).
write_string([H|T]) :- put(H), write_string(T).

/* Element access */
write_elem_access(X, car) :- !, klicformat("car_of(~r)", [X]).
write_elem_access(X, cdr) :- !, klicformat("cdr_of(~r)", [X]).
write_elem_access(X, Pos) :- !, klicformat("arg(~r, ~d)", [X,Pos]).

write_lab(info(_/N/A),interrupt):- !,
    klicformat("~q_~d_interrupt",[N,A]).
write_lab(info(_/N/A),Lab) :- klicformat("~q_~d_~w", [N,A,Lab]).

write_reg(a(N)) :- put(0'a), write(N).
write_reg(x(N)) :- put(0'x), write(N).
write_reg(r(R)) :- write_reg(R).

write_qp(qp) :- klicformat("qp",[]).
write_qp(N) :- integer(N), !,
    klicformat("(struct goalrec*)&allocp[~d]", [N]).

/* Inline expansion */
write_inline([],_Args,_Info,_ElseLab) :- nl.
write_inline([0'%,S|F],Args,Info,ElseLab) :-
    inline_special(S,Args,Info,ElseLab), !,
    write_inline(F,Args,Info,ElseLab).
write_inline([C|F],Args,Info,ElseLab) :-
    put(C),
    write_inline(F,Args,Info,ElseLab).

inline_special(0'%,_Args,_Info,_ElseLab) :- !, put(0'%).
inline_special(0'f,_Args,Info,ElseLab) :- !,
    klicformat("~l",[Info,ElseLab]).
inline_special(C,Args,_Info,_ElseLab) :- 0'0=<C, C=<0'9,
    N is C-0'0,
    nth(N,Args,R),
    klicformat("~r",[R]).
