/* ---------------------------------------------------------- 
%   (C)1993 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include <sys/stat.h>
#include "klic.h"

extern void *malloc();

char *dbdir = 0;
char *initdbdir = LIBDIR;

struct atomrec {
  struct atomrec *next;
  int id;
  char *name;
} *atomroot;
int nextatom;

struct functrec {
  struct functrec *next;
  int id;
  struct atomrec *principal_functor;
  int arity;
} *functroot;
int nextfunct;

int changes;

static struct atomrec *enter_atom(name)
     char *name;
{
  struct atomrec *a;
  int len = strlen(name);
  for (a=atomroot; a!=0; a=a->next) {
    if (strcmp(name,a->name) == 0) return a;
  }
  a = (struct atomrec *)malloc(sizeof(struct atomrec));
  a->name = strcpy((char *)malloc(len+1),name);
  a->next = atomroot;
  a->id = nextatom++;
  atomroot = a;
  changes = 1;
  return a;
}

static void enter_functor(name)
     char *name;
{
  struct functrec *f;
  char *p;
  struct atomrec *functor;
  int arity;

  for (p=name+strlen(name)-1; *p!='_'; p--)
    ;
  *p = 0;
  arity = atoi(p+1);
  functor = enter_atom(name);
  for (f=functroot; f!=0; f=f->next) {
    if (f->principal_functor == functor &&
	f->arity == arity)
      return;
  }
  f = (struct functrec *)malloc(sizeof(struct functrec));
  f->principal_functor = functor;
  f->arity = arity;
  f->next = functroot;
  f->id = nextfunct++;
  functroot = f;
  changes = 1;
}

static void enter_one_line(buf, pathname)
     char *buf;
     char *pathname;
{
  if (strncmp(buf,"atom_", 5) == 0) {
    (void) enter_atom(buf+5);
  } else if (strncmp(buf,"functor_",8)==0) {
    enter_functor(buf+8);
  } else if (strncmp(buf,"predicate_",10)==0) {
    /* currently predicate names are not handled */
  } else if (strncmp(buf,"module_",7)==0) {
    /* currently module names are not handled */
  } else {
    fprintf(stderr, "Unrecognized line in file %s:\n%s\n",
	    pathname, buf);
    exit(-1);
  }
}

static void read_db_file(file, pathname)
     FILE *file;
     char *pathname;
{
  char buf[BUFSIZE];
  while (fgets(buf,BUFSIZE,file) != 0) {
    *strchr(buf,'\n') = 0;
    enter_one_line(buf, pathname);
  }
  fclose(file);
}

static void read_ext_file(pathname)
     char *pathname;
{
  FILE *file;

  file = fopen(pathname, "r");
  if (file==0) {
    fprintf(stderr, "Cannot open file %s\n", pathname);
    exit(-1);
  }
  read_db_file(file, pathname);
}

static char *make_path(name)
     char *name;
{
  char *path;
  if (dbdir != 0) {
    path = (char *)malloc(strlen(dbdir)+strlen(name)+2);
    (void)sprintf(path, "%s/%s", dbdir, name);
  } else {
    path = name;
  }
  return path;
}

static FILE *open_cwd_out(name)
     char *name;
{
  FILE *file;
  if ((file=fopen(name, "w")) == 0) {
    fprintf(stderr, "Cannot open file %s\n", name);
    exit(-1);
  }
  return file;
}

static FILE *open_db_out(name)
     char *name;
{
  FILE *file;
  if ((file=fopen(make_path(name), "w")) == 0) {
    fprintf(stderr, "Cannot open file %s\n", make_path(name));
    exit(-1);
  }
  return file;
}

static void reverse_atomrecs()
{
  struct atomrec
    *next = atomroot,
    *x = 0;
  while (next != 0) {
    struct atomrec *last = x;
    x = next;
    next = x->next;
    x->next = last;
  }
  atomroot = x;
}

static void reverse_functrecs()
{
  struct functrec
    *next = functroot,
    *x = 0;
  while (next != 0) {
    struct functrec *last = x;
    x = next;
    next = x->next;
    x->next = last;
  }
  functroot = x;
}

static int hexvalue(c, name)
     int c;
     char *name;
{
  if ('0' <= c && c <= '9') return c - '0';
  if ('A' <= c && c <= 'F') return c - 'A' + 10;
  if ('a' <= c && c <= 'f') return c - 'a' + 10;
  fprintf(stderr, "Invalid atom name: %s\n", name);
  exit(-1);
}

static void real_atom_name(name,q)
     char *name, *q;
{
  char *p = name;
  while (*p != 0) {
    if (*p == '_') {
      int c = *++p;
      if (c == '_') {
	*q = '_';
      } else {
	int n = (hexvalue(c,name) << 4);
	c = *++p;
	*q = n+hexvalue(c,name);
      }
    } else {
      *q = *p;
    }
    p++; q++;
  }
  *q = 0;
}

static void print_c_string(file, str)
     FILE *file;
     char *str;
{
  int c;
  putc('\"', file);
  while ((c=*str) != 0) {
    if (c=='\\' || c=='"') putc('\\', file);
    putc(c, file);
    str++;
  }
  putc('\"', file);
}

static write_db_files()
{
  FILE *atomh, *atomc, *functh, *functc, *klicdb;
  struct atomrec *a;
  struct functrec *f;
  char buf[BUFSIZE];

  reverse_atomrecs();
  atomh = open_cwd_out("atom.h");
  atomc = open_db_out("atom.c");
  klicdb = open_db_out("klic.db");
  fprintf(atomh, "#include <klic/atomstuffs.h>\n");
  fprintf(atomc, "char *init_atomname[] = {\n");
  for (a=atomroot; a!=0; a=a->next) {
    char realname[BUFSIZE];
    real_atom_name(a->name,realname);
    fprintf(atomh, "#define atom_%s %dL+ATOMNUMBERBASE\n",
	    a->name, a->id);
    putc('\t', atomc);
    print_c_string(atomc, realname);
    putc(',', atomc);
    putc('\n', atomc);
    fprintf(klicdb, "atom_%s\n", a->name);
  }
  fprintf(atomc, "};\n");
  fprintf(atomc, "unsigned long initial_atoms = %d;\n", nextatom);
  fprintf(atomc, "char** atomname = init_atomname;\n");
  fclose(atomh);
  fclose(atomc);

  reverse_functrecs();
  functh = open_cwd_out("funct.h");
  functc = open_db_out("funct.c");
  fprintf(functh, "#include <klic/functorstuffs.h>\n\n");
  fprintf(functc, "#include <klic/atomstuffs.h>\n\n");
  fprintf(functc, "unsigned long init_functors[] = {\n");
  for (f=functroot; f!=0; f=f->next) {
    fprintf(functh, "#define functor_%s_%d\t%dL+FUNCTORNUMBERBASE\n",
	    f->principal_functor->name, f->arity,
	    f->id);
    fprintf(functc, "\t%dL+ATOMNUMBERBASE,\n",
	    f->principal_functor->id);
    fprintf(klicdb, "functor_%s_%d\n",
	    f->principal_functor->name, f->arity);
  }
  fprintf(functc, "};\n\nunsigned long init_arities[] = {\n");
  for (f=functroot; f!=0; f=f->next) {
    fprintf(functc, "\t%dL,\n", f->arity);
  }
  fprintf(functc, "};\n");
  fprintf(functc, "unsigned long initial_functors = %d;\n", nextfunct);
  fprintf(functc, "unsigned long *functors = init_functors;\n");
  fprintf(functc, "unsigned long *arities = init_arities;\n");
  fclose(functh);
  fclose(functc);
}

usage_error(command)
     char *command;
{
  fprintf(stderr, "Usage: %s [-x dbdir] [-X initdbdir] file ...\n", command);
  exit(-1);
}

main(argc,argv)
     int argc;
     char **argv;
{
  extern char *optarg;
  extern int optind, opterr;
  char *dbpath;
  FILE *klicdb;
  int n;
  int c;
  if (argc < 2) {
    usage_error(argv[0]);
  }
  while ((c=getopt(argc,argv,"X:x:")) != -1) {
    switch (c) {
    case 'X': initdbdir = optarg; break;
    case 'x': dbdir = optarg; break;
    case '?': usage_error(argv[0]);
    }
  }
  atomroot = 0; nextatom = 0;
  functroot = 0; nextfunct = 0;
  klicdb = fopen(make_path(DBFILENAME), "r");
  if (klicdb != 0) {
    read_db_file(klicdb);
    changes = 0;
  } else {
    char *dbdirsave = dbdir;
    dbdir = initdbdir;
    klicdb = fopen(make_path("klicdb.init"), "r");
    if (klicdb == 0) {
      fprintf(stderr, "Can't open initial database file: %s\n",
	      make_path("klicdb.init"));
      exit(-1);
    }
    read_db_file(klicdb);
    changes = 1;
    dbdir = dbdirsave;
  }
  for (n=optind; n<argc; n++) {
    read_ext_file(argv[n]);
  }
  if (changes ||
      access("atom.h", F_OK) != 0 ||
      access(make_path("atom.c"), F_OK) != 0 ||
      access("funct.h", F_OK) != 0 ||
      access(make_path("funct.c"), F_OK) != 0) {
    write_db_files();
  }
  return 0;
}
