% Copyright (C) 1991
% Institute for New Generation Computer Technology, Japan
% ALL RIGHTS RESERVED

%
% File name : io.spec
%
% (01) 90.12.20 by T.S. from V3 builtin
% (02) 91. 1.19 by T.S. change io I/F to message manager
% (03) 91. 1.23 by T.S. add print_env/2
% (04) 91. 1.23 by T.S. add pp/1
% (05) 91. 1.24 by T.S. add statistics/2,var/1,nonvar/1
% (06) 91. 1.24 by T.S. change installation of print
% (07) 91. 1.25 by T.S. remove prolog predicates
% (08) 91. 1.26 by T.S. remain see(user) alternative
% (09) 91. 1.30 by T.S. change module_manager call from ds to ms
% (10) 91. 1.30 by T.S. change refute call for portray to syntax_utility
% (11) 91. 2. 4 by T.S. remove use spec_ms/system
% (12) 91. 2.13 by S.K. define package
% (13) 91. 2.16 by T.S. env name check in interpretive
% (14) 91. 2.25 by S.K. change suppressor from ` to $
% (15) 91. 2.26 by T.S. change interpretive call
% (16) 91. 2.26 by S.K. remove get_parent_module


:- define io.

%
% IO environment
%   specifies esp_call as an environment
%   uses io/pretty_print,spec_ms/system
%

:-define io/builtin.
:-environment esp_call.
:-use io/pretty_print.


% slot definition
:-attribute input_stream, input_stream_id.
:-attribute output_stream, output_stream_id.
:-attribute stream_pool, gensym_table.
:-attribute window.

%
% i/o
%

% print term
:-view print/1.
print(X):-
    get_environment_name(Env),
    portray(X, [], _, Env),!.

:-view print/2.
print(X, VI):-
    get_environment_name(Env),
    portray(X, VI, _, Env),!.

:-view print/3.
print(X, VI, NVI):-
    get_environment_name(Env),
    portray(X, VI, NVI, Env),!.

get_environment_name(Env):-
    get_self_object(Self),
    :get_parent(#spec_ms##object_manager, Self, Parent),
    Module = Parent,
%    get_parent_module(Parent, Module),
    :refute(Module, environment_name, {Env}),!.
get_environment_name(spec).

portray(X, VI, NVI, Env):-
    transform(X, Y, Env),
    putt(Y, VI, NVI).

transform(X, Y, Env):-
    atom(Env),
    :get(#spec_ms##module_manager, class(`(Env/portray)), Class),
    :new(Class, Portray),
    :get(#spec_ms##module_manager, class(`(spec_ms/syntax_utility)), UTL),
    :refute_call(UTL, Portray, Env, transform, {X, Y}),!.
transform(X, X, _).

putt(X, VI, NVI):-
    get_output_stream(Stream),
    :unquote_atom(Stream),
    :putt(Stream, X, VI, NVI),
    :quote_atom(Stream),!.

:-view write/1.
write(X):-
    putt(X, [], _).

:-view writeq/1.
writeq(X):-
    get_output_stream(Stream),
    :quote_atom(Stream),
    :putt(Stream, X, []).

% print string
:-view printf/2.
printf(X,Y):-
    get_output_stream(Stream),
    :putf(Stream, X, Y).

:-view printl/1.
printl(X):-
    get_output_stream(Stream),
    :putl(Stream, X).

:-view pp/1.
pp(X):-
    pp(X, [], _), fail.
pp(_).

:-view pp/2.
pp(X, CVI):-
    pp(X, CVI, _), fail.
pp(_, _).

:-view pp/3.
pp(X, CVI, NVI):-
%    :tst_to_pst(#cil_builtin, X,Y),
    get_output_stream(Stream),
    % io/pretty_print
    myPrint(X, Stream, CVI, NVI),
    :putc(Stream, key#lf).

:-view indention_size/1.
indention_size(Size):-
    % io/pretty_print
    set_indention_size(Size).


% print control
:-view nl/0.
nl:-
    get_output_stream(Stream),
    :putc(Stream, key#lf).

:-view tab/1.
tab(N):-
    get_output_stream(Stream),
    tab(N, Stream).

tab(0,_):-!.
tab(N,Stream):-
    integer(N),
    :putf(Stream, "\.@b", [N]).

% read term
:-view read/1.
read(X):-
    get_input_stream(Stream),
    :gett(Stream, X).

:-view read/2.
read(X, VI):-!,
    get_input_stream(Stream),
    :gett(Stream, X, [], VI).

:-view getl/1.
getl(X):-
    get_input_stream(Stream),
    :getl(Stream, X).


% character i/o
:-view get/1.
get(X):-
    get_input_stream(Stream),
    :getc(Stream, X).

:-view put/1.
put(X):-
    get_output_stream(Stream),
    :putc(Stream, X).

%
% stream operation
%

% file input
:-view see/1.
see(user):-!,
    see_user.
see(standard_input):-!,
    :self(#process, Me),
    :standard_input(Me, Input),
    !input_stream := Input,
    !input_stream_id := standard_input.
see(Filename):-
    :get_atom_string(#symbolizer, Filename, Filestring),
    get_stream_pool(Stream_pool),
    (
        get_at(Stream_pool, File, Filename, input),!
    ;    
        :create(#standard_input_file, File, Filestring),
        add_at(Stream_pool, File, Filename, input)
    ),
    !input_stream := File,
    !input_stream_id := Filename.

see_user:-
    get_window(Window),!,
    !input_stream := Window,
    !input_stream_id := user.
see_user:-
    error(no_window),
    fail.

:-view seeing/1.
seeing(F):-
    get_input_stream_id(F).

:-view seen/0.
seen:-
    get_input_stream_id(Input_stream_id),
    (
        Input_stream_id == user,!
    ;
        Input_stream_id == standard_input,!,
        get_input_stream(Stream),
        :close_input(Stream)
    ;
        get_input_stream(Stream),
        :close_input(Stream),
        get_stream_pool(Stream_pool),
        :remove_at(Stream_pool,_,Input_stream_id)
    ),
    (get_window(Window),!,
        !input_stream := Window,
        !input_stream_id := user
    ;error(no_window),fail).

% file output
:-view tell/1.
tell(user):-!,
    tell_user.
tell(standard_output):-!,
    :self(#process, Me),
    :standard_output(Me, Output),
    !output_stream := Output,
    !output_stream_id := standard_output.
tell(standard_message):-!,
    :self(#process, Me),
    :standard_message(Me, Output),
    !output_stream := Output,
    !output_stream_id := standard_message.
tell(Filename):-
    :get_atom_string(#symbolizer, Filename, Filestring),
    get_stream_pool(Stream_pool),
    (
        get_at(Stream_pool, File, Filename, output),!
    ;    
        :create(#standard_output_file, File, Filestring),
        add_at(Stream_pool, File, Filename, output)
    ),
    !output_stream := File,
    !output_stream_id := Filename.

:-view telling/1.
telling(F):-
    get_output_stream_id(F).

:-view told/0.
told:-
    get_output_stream_id(Output_stream_id),
    (
        Output_stream_id == user,!
    ;
        Output_stream_id == standard_output,!,
        get_output_stream(Stream),
        :close_output(Stream)
    ;
        Output_stream_id == standard_message,!,
        get_output_stream(Stream),
        :close_output(Stream)
    ;
        get_output_stream(Stream),
        :close_output(Stream),
        get_stream_pool(Stream_pool),
        :remove_at(Stream_pool,_,Output_stream_id)
    ),
    (get_window(Window),!,
        !output_stream := Window,
        !output_stream_id := user
    ;error(no_window),fail).

tell_user:-
    get_window(Window),!,
    !output_stream := Window,
    !output_stream_id := user.
tell_user:-
    error(no_window),
    fail.

:-view end_of_file/0.
end_of_file:-
    get_input_stream_id(Stream_id),
    Stream_id \== user,
    get_input_stream(Stream),
    :end_of_input(Stream),!.

:-view close/0.
close:-
    see(user),
    tell(user),
    get_stream_pool(Stream_pool),
    :get_contents(Stream_pool, Stream_list),
    close(Stream_list),
    :clear(Stream_pool).

close([]):-!.
close([Frozen|Streams]):-
    close_stream(Frozen),!,
    close(Streams).

close_stream(Frozen):-
    melt(Frozen, Melt),
    Melt = {File,Mode},
    (
        Mode==input,!, :close_input(File)
    ;
        Mode==output, :close_output(File)
    ).


get_at(Pool, Object, Id, Mode):-
    :get_at(Pool, Heap, Id),
    melt(Heap, {Object,Mode}).

add_at(Pool, Object, Id, Mode):-
    freeze({Object,Mode},Heap),
    :add_at(Pool, Heap, Id).


% subroutines
get_input_stream(Stream):-
    Stream = !input_stream,
    object(Stream,_,_),!.
get_input_stream(Stream):-
    get_window(Window),!,
    Stream = Window,
    !input_stream := Stream,
    !input_stream_id := user.
get_input_stream(_):-
    error(no_window),
    fail.

get_input_stream_id(Id):-
    Id = !input_stream_id,
    Id \== 0,!.
get_input_stream_id(Id):-
    get_window(Window),!,
    !input_stream := Window,
    Id = user,
    !input_stream_id := Id.
get_input_stream_id(_):-
    error(no_window),
    fail.

get_output_stream(Stream):-
    Stream = !output_stream,
    object(Stream,_,_),!.
get_output_stream(Stream):-
    get_window(Window),!,
    Stream = Window,
    !output_stream := Stream,
    !output_stream_id := user.
get_output_stream(_):-
    error(no_window),
    fail.

get_output_stream_id(Id):-
    Id = !output_stream_id,
    Id \== 0,!.
get_output_stream_id(Id):-
    get_window(Window),!,
    !output_stream := Window,
    Id = user,
    !output_stream_id := Id.
get_output_stream_id(_):-
    error(no_window),
    fail.

get_stream_pool(Stream_pool):-
    Stream_pool = !stream_pool,
    object(Stream_pool,_,_),!.
get_stream_pool(Stream_pool):-
    :create(#hash_index, Stream_pool, 64),
    !stream_pool := Stream_pool.

error(Key):-
    :error_message(#spec_ms##message_manager, Key, [], io).

get_window(Window):-
    :get_window(#spec_ms##message_manager, Window),
    !window := Window.


% syntax sugar
%:-define io/syntax.

% error message
:-define io/message.
:-view message/2.
message(no_window, {"No window in this process","             "}):-!.

%
% pretty printer
%
:-define io/pretty_print.
:-environment esp_call.


% pretty print term on the given object
% X : term to print out
% Obj : object to print out

:-view myPrint/4.
myPrint(X, Obj, CVI, NVI):-
    myPrint(X, 0, 100, _, _, Obj, CVI, NVI).

% myPrint/8
% Obj : object to print out
myPrint(X, _, L0, L1, _, Obj, CVI, NVI):-unbound(X),!,    % unbound
    :putt(Obj, X, CVI, NVI),
    L1 is L0 + 1.
myPrint({F,X, Y}, N, L0, L2, Vars, Obj, CVI, NVI):-        % (_,_)
    F == (','),!,
    myPrint(X, N, L0, L1, Vars, Vars, Obj, CVI, TVI),
    :putf(Obj, ",\n"),  tab(N, Obj),
    myPrint(Y, N, L1, L2, Vars, Obj, TVI, NVI).
myPrint(X, N, L0, L1, Vars, Obj, CVI, NVI):-
    myPrint(X, N, L0, L1, Vars, Vars, Obj, CVI, NVI).

% myPrint/9
myPrint(X, _, L0, L1, _, _, Obj, CVI, NVI):-
    % unbound,atomic
    (unbound(X) ; atomic(X) ), !,
    :putt(Obj, X, CVI, NVI),
    L1 is L0 + 1.
myPrint(X, _, L0, L1, _, _, Obj, CVI, NVI):-
    % PSI data
    ( string(X,_,_) ; heap_vector(X,_) ; object(X,_,_) ),!,
    :putt(Obj, X, CVI, NVI),
    L1 is L0 + 1.
myPrint(X, _, L0, L1, _, _, Obj, CVI, NVI):-
    % {}
    stack_vector(X,0),!,
%        (L = 0; first(X,F), unbound(F)),!,
    :putt(Obj, X, CVI, NVI),
    L1 is L0 + 1.
myPrint(X, N, L0, L1, B, Vars, Obj, CVI, NVI):-
    loop_check_and_print(X, N, L0, L1, B, Vars, Obj, CVI, NVI).

loop_check_and_print(X, N, L0, L1, B, Vars, Obj, CVI, NVI):-
    unbound(B), !,
    B = lp(X, L0, _),
    print_form(X, N, L0, L1, Vars, Obj, CVI, NVI).
loop_check_and_print(A, _, L0, L1, lp(B, L, _), _, Obj, CVI, CVI):-
    A == B,  !,
    %!! LOOP or SAME_DATA is appeared
    :putf(Obj, string#" >LINE: "),
    :putt(Obj, L),  L1 is L0 + 1.
loop_check_and_print(X, N, L0, L1, lp(_, _, B), Vars, Obj, CVI, NVI):-
    loop_check_and_print(X, N, L0, L1, B, Vars, Obj, CVI, NVI).

print_form({F,A, B}, N, L0, L1, Vars, Obj, CVI, NVI):-     % (_,_)
    F == (','),!,
    :putc(Obj, #"("),
    idt_par((A, B), N, L0, L1, Vars, Obj, CVI, NVI),
    :putc(Obj, #")").
% partial term
print_form({F,X, S}, N, L0, L1, Vars, Obj, CVI, NVI):-
    F == x, !,
    ( unbound(X), !, 
        idt_x(S, N, L0, L1, Vars, Obj, CVI, NVI)
    ; true, 
        myPrint(X, N, L0, L1, Vars, Vars, Obj, CVI, NVI) ).
% normal term
print_form(X, N, L0, L1, Vars, Obj, CVI, NVI):-
    stack_vector(X, I),  I >= 2,
    first(X, F), atomic(F),!,
    :putt(Obj, F),
    :putc(Obj, #"("),
    idt_func(1, I - 1, X, N, L0, L1, Vars, Obj, CVI, NVI),
    :putc(Obj, #")").
% list
print_form([X|Y], N, L0, L2, Vars, Obj, CVI, NVI):-!,
    :putc(Obj, #"["),
    myPrint(X, N, L0, L1, Vars, Vars, Obj, CVI, NVI1),
    idt_list(Y, N, L1, L2, Vars, Obj, NVI1, NVI),
    :putc(Obj, #"]").
% vector
print_form(X, N, L0, L1, Vars, Obj, CVI, NVI):-
    stack_vector(X, I),
    :putc(Obj, #"{"),
    idt_func(0, I - 1, X, N, L0, L1, Vars, Obj, CVI, NVI),
    :putc(Obj, #"}").

idt_func(I, I, X, N, L0, L1, Vars, Obj, CVI, NVI):-!,
    vector_element(X, I, Y),
    myPrint(Y, N, L0, L1, Vars, Vars, Obj, CVI, NVI).
idt_func(I, K, X, N, L0, L2, Vars, Obj, CVI, NVI):-
    vector_element(X, I, Y),
    myPrint(Y, N, L0, L1, Vars, Vars, Obj, CVI, NVI1),
    :putc(Obj, #","),
    idt_func(I + 1, K, X, N, L1, L2, Vars, Obj, NVI1, NVI).

idt_list(X, _, L, L, _, Obj, CVI, NVI):-unbound(X),!,
    :putc(Obj, #"|"), :putt(Obj, X, CVI, NVI).
idt_list([], _, L, L, _, _, CVI, CVI):-!.
idt_list([X|Y], N, L0, L2, Vars, Obj, CVI, NVI):-!,
    :putc(Obj, #","),
    myPrint(X, N, L0, L1, Vars, Vars, Obj, CVI, NVI1),!,
    idt_list(Y, N, L1, L2, Vars, Obj, NVI1, NVI).
idt_list(X, N, L0, L1, Vars, Obj, CVI, NVI):-
    :putc(Obj, #"|"),
    myPrint(X, N, L0, L1, Vars, Vars, Obj, CVI, NVI).

idt_par(X, _, L, L, _, Obj, CVI, NVI):-
    unbound(X),!,
    :putt(Obj,X,CVI,NVI).
idt_par((X, Y), N, L0, L2, Vars, Obj, CVI, NVI):-!,
    myPrint(X, N, L0, L1, Vars, Vars, Obj, CVI, NVI1),
    :putc(Obj, #","),
    idt_par(Y, N, L1, L2, Vars, Obj, NVI1, NVI).
idt_par(X, N, L0, L1, Vars, Obj, CVI, NVI):-!,
    myPrint(X, N, L0, L1, Vars, Vars, Obj, CVI, NVI).

idt_x(X, _, L, L, _, _, CVI, CVI):-unbound(X),!.
idt_x(t((Name, Value), L, R), N, L0, L3, Vars, Obj, CVI, NVI):-
    :putc(Obj, key#lf),  writeNo(Obj, L0),  tab(N, Obj),
    :putt(Obj, Name, CVI, NVI1),
    :putc(Obj, #"/"),
    myPrint(Value, N+1, L0+1, L1, Vars, Vars, Obj, NVI1, NVI2),
    idt_x(L, N, L1, L2, Vars, Obj, NVI2, NVI3),
    idt_x(R, N, L2, L3, Vars, Obj, NVI3, NVI).

writeNo(Obj, L0):-
    ( L0 < 1000, !,
            :putf(Obj, "\.2b\q: ",[L0])
    ; L0 < 10000, !,
            :putf(Obj, "\.1b\q: ",[L0])
    ; :putt(Obj, L0), :putf(Obj, ": ")  ).


tab(Tab, Obj):-
    get_indention_size(Size),
    N is Tab * Size,
    :putf(Obj, "\@b", [N]).

% get indention size
:-view get_indention_size/1.
:-attribute indention_size.
get_indention_size(Size):-
    is_indention_set,!,
    Size = !indention_size.
get_indention_size(Size):-
    % default indention size is 1
    Size = 1,
    set_indention_size(Size).

% set indention size
:-view set_indention_size/1.
set_indention_size(Size):-
    !indention_size := Size,
    indention_set_on.

% check indention flag
is_indention_set:-
    !indention_flag == on.

% raise indention flag
:-attribute indention_flag.
indention_set_on:-
    !indention_flag := on.