% (C) 1992 Institute for New Generation Computer Technology
% (Read COPYRIGHT for detailed information)

%
% file name :   constraint.spec
%
% (01) 91. 1.26 by S.K. create constraint module
% (02) 91. 1.26 by S.K. add operators
% (03) 91. 1.30 by T.S. change suppressor from ` to $
% (04) 91. 2. 5 by T.S. add syntax for constr suppress
% (05) 91. 2. 6 by T.S. bug of constr suppress
% (06) 91. 2.13 by S.K. define package
% (07) 91. 2.25 by S.K. change suppressor from $ to `
%

%
% constraint environment
%   specifies meta_esp_call as an environment
%   uses pst/builtin
%
:-define constraint.

%
% constraint builtin
%
:- define constraint/builtin.

:- environment meta_esp_call.
:- use pst/builtin.

%   equal
:- view equal/5,t_equal/5.

%   assign
:- view assign/3,assign/4.

%   dif
:- view dif/2.

%   boolean
:- view not/2,and_cil/3,or_cil/3,exor/3,nand/3,nor/3,imply/3,iff/3.

%   arithmetic
:- view add/6,multiply/6,mod/6,exp/5,t_exp/5,evaluate/5.

%   constraint
:- view constr/1,constr/2,constr/3,constr/4.

%   for meta predicate
:-view andx/2,orx/2,dif_gr/2,t_evaluate/5.

%   for compiling optimize
%   :-view equal1/5,t_equal1/5.
%   :-view less/5.

%
% constraint prediacte
%
%% :-public constr/4.

    %CONSTR(X,<EXPECTATION>,<RESULT>,<CONTINUE>)

constr(E):-
    bind_hook(Y,(Y = true)),
    constr(E,true,Y,Y).
constr(X,Y):-
    bind_hook(Y,(Y = true)),
    constr(X,true,Y,Y).
constr(X,E,R):-
    constr(X,E,R,R).

constr(_,_,_,R):-
    bound(R),!.
constr(X,P,Q,_):-
    unbound(X),!,
    X = P,
    X = Q.
constr(true,P,Q,_):-!,
    P = true,
    Q = true.
constr(false,P,Q,_):-!,
    P = false,
    Q = false.
constr(not(A),P,Q,R):-!,
    not(V,P),
    not(U,Q),
    constr(A,V,U,R).
constr((A,B),P,Q,R):-!,
    and_cil(X,Y,P),
    and_cil(Z,U,Q),
    constr(A,X,Z,R),
    constr(B,Y,U,R).
constr(and(A,B),P,Q,R):-!,  % Sequential 'and'
    and_cil(X,Y,P),
    and_cil(Z,U,Q),
    constr(A,X,Z,R),
    bind_hook(Z,constr(B,Y,U,R)).
constr((A ; B),P,Q,R):-!,
    or_cil(X,Y,P),
    or_cil(Z,U,Q),
    constr(A,X,Z,R),
    constr(B,Y,U,R).
constr(or(A,B),P,Q,R):-!, %Sequential 'or'
    or_cil(X,Y,P),
    or_cil(Z,U,Q),
    constr(A,X,Z,R),
    bind_hook(Z,constr(B,Y,U,R)).
constr('->'(A,B),P,Q,R):-!,
    imply(X,Y,P),
    imply(Z,U,Q),
    constr(A,X,Z,R),
    constr(B,Y,U,R).
constr(imply(A,B),P,Q,R):-!,    %Sequential 'imply'
    imply(X,Y,P),
    imply(Z,U,Q),
    constr(A,X,Z,R),
    bind_hook(Z,constr(B,Y,U,R)).
constr(if(A,B,C),P,Q,R):-!, % 'if' statement
    constr(A,_,H,R),
    constr(or(and(H,B),and(not(H),C)),P,Q,R).
constr(<->(A,B),P,Q,R):-!,
    iff(X,Y,P),
    iff(Z,U,Q),
    constr(A,X,Z,R),
    constr(B,Y,U,R).
constr(exor(A,B),P,Q,R):-!,
    exor(X,Y,P),
    exor(Z,U,Q),
    constr(A,X,Z,R),
    constr(B,Y,U,R).
constr(nor(A,B),P,Q,R):-!,
    nor(X,Y,P),
    nor(Z,U,Q),
    constr(A,X,Z,R),
    constr(B,Y,U,R).
constr(nand(A,B),P,Q,R):-!,
    nand(X,Y,P),
    nand(Z,U,Q),
    constr(A,X,Z,R),
    constr(B,Y,U,R).
constr(assign(X,Y),P,Q,R):-!,   % for parameres passing
    assign(X,Y,Q,R),
    bind_hook(Q,unify(P,Q)).
constr(`(X = Y),P,Q,R):-!,
    equal(X,Y,P,Q,R).
constr(`(X \== Y),P,Q,R):-!,
    constr(not(`(X = Y)),P,Q,R).
constr(`(X =:= Y),P,Q,R):-!,
    and_cil(P1,P2,P),
    and_cil(Q1,Q2,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,T,P2,Q2,R).
constr(`(X =\= Y),P,Q,R):-!,
    constr(not(`(X =:= Y)),P,Q,R).
constr(`(X < Y),P,Q,R):-!,
    and_cil(P1,P2,V),
    and_cil(V,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    bind_hook(T,bind_hook(U,less(T,U,P3,Q3,R))).
constr(`(X > Y),P,Q,R):-!,
    constr(`(Y < X),P,Q,R).
%constr(X,P,Q,R):-
%    defcon(X,Y),
%    constr(Y,P,Q,R). %user defined

%% :-public less/5.
less(T,U,P,Q,_):-
    T < U,!,
    P = true,
    Q = true.
less(_,_,P,Q,_):-
    P = false,
    Q = false.

t_multiply(X,Y,Z,P,Q,R):-
    bind_hook(X,bind_hook(Y,pv(F,t_evaluate(`(X * Y),Z,P,Q,R)))),
    bind_hook(Z,bind_hook(Y,pv(F,t_evaluate(`(Z / Y),X,P,Q,R)))),
    bind_hook(Z,bind_hook(X,pv(F,t_evaluate(`(Z / X),Y,P,Q,R)))).

t_mod(X,Y,Z,P,Q,R):-
    bind_hook(X,bind_hook(Y,t_evaluate(`(X mod Y),Z,P,Q,R))).

% EVALUATE ARITHMETIC OPERATOR
% evaluate/5
evaluate(E,V,P,Q,R):-
    evaluate(E,T),
    equal(V,T,P,Q,R).
% temporary evaluate/5
t_evaluate(E,V,P,Q,R):-
    evaluate(E,T),
    t_equal(T,V,P,Q,R).

% :- public evaluate/2.
evaluate(`(X + Y),Z):-!,
    Z is X + Y.
evaluate(`(X - Y),Z):-!,
    Z is X - Y.
evaluate(`(X / Y),Z):-!,
    Z is X / Y.
evaluate(`(X * Y),Z):-!,
    Z is X * Y.
evaluate(`(X mod Y),Z):-!,
    Z is X mod Y.


%%%%%% ARITHMETIC EXPRESSION 

% exp/5
exp(X,Y,P,Q,R):-
    unbound(X),!,
    t_equal(X,Y,P,Q,R).
exp(`(X + Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    add(T,U,Z,P3,Q3,R).
exp(`(X - Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    add(Z,U,T,P3,Q3,R).
exp(`(X * Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    multiply(T,U,Z,P3,Q3,R).
exp(`(X / Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    multiply(U,Z,T,P3,Q3,R).
exp(`(X mod Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    mod(T,U,Z,P3,Q3,R).
exp(X,Y,P,Q,R):-
    t_equal(X,Y,P,Q,R).

    % exp/5 of temporary equality
t_exp(X,Y,P,Q,R):-
    unbound(X),!,
    t_equal(X,Y,P,Q,R).
t_exp(`(X + Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_add(T,U,Z,P3,Q3,R).
t_exp(`(X - Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_add(Z,U,T,P3,Q3,R).
t_exp(`(X * Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_multiply(T,U,Z,P3,Q3,R).
t_exp(`(X / Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_multiply(U,Z,T,P3,Q3,R).
t_exp(`(X mod Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_mod(T,U,Z,P3,Q3,R).
t_exp(X,Y,P,Q,R):-
    t_equal(X,Y,P,Q,R).

%
% utilities
%

add_variables(X,L,L):-
    eq_mem(X,L),!.
add_variables(X,L,[X|L]).

get_variables(X,L,LL):-
    unbound(X),!,
    add_variables(X,L,LL).
get_variables([X|R],L,LL):-!,
    get_variables(X,L,LT),!,
    get_variables(R,LT,LL).
get_variables(X,L,LL):-
    pst(X),!,
    get_pst_variables(X,L,LL).
get_variables(X,L,LL):-
    stack_vector(X,N),!,
    get_variables_vector(N,X,L,LL).
get_variables(_,L,L).

get_pst_variables(x(P,T),L,LL):-
    unbound(P),!,
    get_tree_variables(T,L,LL).
get_pst_variables(x(P,_),L,LL):-!,
    get_pst_variables(P,L,LL).

get_tree_variables(T,L,L):-
    unbound(T),!.
get_tree_variables(t((_,V),S,B),L,LL):-
    get_tree_variables(S,L,TL1),
    get_variables(V,TL1,TL2),!,
    get_tree_variables(B,TL2,LL).

pst(X):-
    stack_vector(X,3),
    first(X,F),
    F == x,!.


get_variables_vector(0,_,L,L):-!.
get_variables_vector(N,X,L,LL):-
    M is N - 1,
    vector_element(X,M,E),
    get_variables(E,L,LT),!,
    get_variables_vector(M,X,LT,LL).

eq_mem(X,[Y|_]):-
    X =:= Y,!.
eq_mem(X,[_|R]):-
    eq_mem(X,R).

%
% dif/2
%
dif(X,Y):- suspend((X,Y),[],dif_gr(X,Y)).

dif_gr(X,Y):- same_gr(X,Y), !,
        fail.
dif_gr(_,_).

%    dif_a(X,A) :- bind_hook(X,not_equal(A,X)).

suspend(X,Y,Z):-
    unbound(X),!,
    bind_hook(X,suspend_tr(X,Y,Z)).
suspend(X,Y,Z):-
    suspend_tr(X,Y,Z).

suspend([],P):-!,
    solve(P).
suspend([(list,L)|Y],P):-
    list(L),!,
    L = [A|R],
    suspend(A,[(list,R)|Y],P).
suspend([(list,R)|Y],P):-!,
    suspend(R,Y,P).
suspend([(0,X)|Y],P):-!,
    first(X,A),
    suspend(A,Y,P).
suspend([(I,X)|Y],P):-
    J is I - 1,
    vector_element(X,I,A),
    suspend(A,[(J,X)|Y],P).

%    suspend_tr(T,Y,P):-atomic(T),!,
%    suspend(Y,P).
suspend_tr([A|R],Y,P):-!,
    suspend(A,[(list,R)|Y],P).
suspend_tr(T,Y,P):-
    stack_vector(T,N),!,
    ( N > 0,!,
      suspend_tr(N - 1,T,Y,P)
    ; suspend(Y,P)).
suspend_tr(_,Y,P):-   % atomic,heap,string,etc
    suspend(Y,P).

suspend_tr(0,T,Y,P):-!,
    first(T,A),
    suspend(A,Y,P).
suspend_tr(N,T,Y,P):-
    M is N - 1,
    vector_element(T,N,A),
    suspend(A,[(M,T)|Y],P).


%
% equal/5
%
equal(_,_,_,_,W):-
    bound(W),!.
equal(X,Y,V,R,W):-
    bound(V),!,
    equal1(X,Y,V,R,W).
equal(X,Y,Z,R,S):-
    unbound(Y),!,
    freeze(Y,Z,equal(X,Y,Z,R,S)).
equal(X,Y,Z,R,S):-
    unbound(X),!,
    freeze(X,Z,equal(X,Y,Z,R,S)).
equal([A|X],[B|Y],Z,R,S):-!,
    and_cil(C,D,Z),
    and_cil(P,Q,R),
    equal(A,B,C,P,S),!,
    equal(X,Y,D,Q,S).
equal(X,Y,Z,R,S):-
    stack_vector(X,N),
    stack_vector(Y,N),!,
    equal(N,X,Y,Z,R,S). 
equal(X,Y,Z,R,S):-     % atomic,string,heap,etc
    X = Y,!,
    equal(0,X,Y,Z,R,S).
equal(_,_,false,false,_).

equal(_,_,_,_,_,S):-
    bound(S),!.
equal(0,_,_,T,T,_):-!,
    T = true.
equal(I,X,Y,Z,R,S):-!,
    J is I - 1,
    and_cil(C,D,Z),
    and_cil(P,Q,R),
    vector_element(X,J,A),
    vector_element(Y,J,B),
    equal(A,B,C,P,S),
    equal(J,X,Y,D,Q,S).

equal1(X,Y,true,U,_):-!,
    unify_cil(X,Y),  %% <-  X = Y
    U = true.
equal1(X,Y,false,Q,R):-
    bind_hook(Q,(Q = false)),
    bind_hook(P,P=Q), %% <- P = Q
    bind_hook(X,bind_hook(Y,equal(X,Y,P,Q,R) ) ).

%
% t_equal/5
%
t_equal(_,_,_,_,W):-
    bound(W),!.
t_equal(X,Y,V,R,W):-
    bound(V),!,
    t_equal1(X,Y,V,R,W).
t_equal(X,Y,Z,R,S):-
    unbound(X),!,
    freeze(X,Z,t_equal(X,Y,Z,R,S)).
t_equal(X,Y,Z,R,_):-
    unbound(Y),!,
    X = Y,
    Z = R,
    Z = true. 
t_equal(X,X,Z,R,_):-!,  %% ??? X,X OK ???
    Z = R,
    Z = true.
t_equal(_,_,false,false,_).

t_equal1(X,Y,true,U,_):-!,
    X = Y,
    U = true.
t_equal1(X,Y,false,Q,R):-
    bind_hook(Q,(Q = false)),
    bind_hook(P,P = Q), %% <- P = Q
    bind_hook(X,bind_hook(Y,t_equal(X,Y,P,Q,R))).


assign(X,Y,Z):-
    assign(X,Y,Z,Z).
    % Y must not have multiple occurrences of a variable

assign(_,_,_,S):-
    bound(S),!.
assign(X,Y,Z,_):-
    unbound(Y),!,
    X = Y,
    Z = true.
assign(X,Y,Z,S):-
    unbound(X),!,
    bind_hook(X,assign(X,Y,Z,S)).
assign([X|P],[Y|Q],Z,S):-!,
    and_cil(A,B,Z),
    assign(X,Y,A,S),!,
    assign(P,Q,B,S).
assign(X,Y,Z,S):-
    stack_vector(X,N),
    stack_vector(Y,N),!,
    assign(N,X,Y,Z,S).
assign(X,Y,Z,S):-       % atomic,heap,string,etc
    X == Y,
    assign(0,X,Y,Z,S),!.
assign(_,_,Z,_):-
    unify(Z,false).

assign(_,_,_,_,S):-
    bound(S),!.
assign(0,_,_,Z,_):-!,
    unify(Z,true).
assign(I,X,Y,Z,S):-!,
    J is I - 1,
    and_cil(C,D,Z),
    vector_element(X,J,A),
    vector_element(Y,J,B),
    assign(A,B,C,S),
    assign(J,X,Y,D,S).


%
% boolean predicate
%

% not/2
not(X,Y):-
    bind_hook(X,pv(F,if(X,(Y = false),(Y = true)))),
    bind_hook(Y,pv(F,if(Y,(X = false),(X = true)))).

% and/3
and_cil(X,Y,Z):-
    bind_hook(X,pv(F,if(X,(Y = Z),(Z = false)))),
    bind_hook(Y,pv(F,if(Y,(X = Z),(Z = false)))),
    bind_hook(Z,pv(F,if(Z,((X = true),(Y = true)),andx(X,Y)))).

    andx(X,Y) :-
        bind_hook(X,if(X,(Y = false))),
        bind_hook(Y,if(Y,(X = false))).

% or/3
or_cil(X,Y,Z):-
    bind_hook(X,pv(F,if(X,(Z = true),(Y = Z)))),
    bind_hook(Y,pv(F,if(Y,(Z = true),(X = Z)))),
    bind_hook(Z,pv(F,if(Z,orx(X,Y),((X = false),(Y = false))))).

    % orx/2
    orx(X,Y) :-
        bind_hook(X,if(X,true,(Y = true))),
        bind_hook(Y,if(Y,true,(X = true))).

% exor/3
exor(X,Y,Z):-
    not(X,NX),
    not(Y,NY),
    and_cil(NX,Y,V),
    and_cil(X,NY,U),
    or_cil(V,U,Z).

% nand/3
nand(X,Y,Z):-
    and_cil(X,Y,U),
    not(U,Z).

% nor/3
nor(X,Y,Z):-
    or_cil(X,Y,U),
    not(U,Z).

% imply/3
imply(X,Y,Z):-
    not(X,U),
    or_cil(U,Y,Z).

% iff/3
iff(X,Y,Z):-
    imply(X,Y,U),
    imply(Y,X,V),
    and_cil(U,V,Z).

%
% arithmetic predicate
%

%%%%%% BASIC ARITHMETIC CONSTRAINT %%%%%%%
add(X,Y,Z,P,Q,R):-
    bind_hook(X,bind_hook(Y,pv(F,evaluate(`(X + Y),Z,P,Q,R)))),
    bind_hook(Z,bind_hook(Y,pv(F,evaluate(`(Z - Y),X,P,Q,R)))),
    bind_hook(Z,bind_hook(X,pv(F,evaluate(`(Z - X),Y,P,Q,R)))).

multiply(X,Y,Z,P,Q,R):-
    bind_hook(X,bind_hook(Y,pv(F,evaluate(`(X * Y),Z,P,Q,R)))),
    bind_hook(Z,bind_hook(Y,pv(F,evaluate(`(Z / Y),X,P,Q,R)))),
    bind_hook(Z,bind_hook(X,pv(F,evaluate(`(Z / X),Y,P,Q,R)))).

mod(X,Y,Z,P,Q,R):-
    bind_hook(X,bind_hook(Y,evaluate(`(X mod Y),Z,P,Q,R))).

    % for temporary equality no_debug
t_add(X,Y,Z,P,Q,R):-
    bind_hook(X,bind_hook(Y,pv(F,t_evaluate(`(X + Y),Z,P,Q,R)))),
    bind_hook(Z,bind_hook(Y,pv(F,t_evaluate(`(Z - Y),X,P,Q,R)))),
    bind_hook(Z,bind_hook(X,pv(F,t_evaluate(`(Z - X),Y,P,Q,R)))).

t_multiply(X,Y,Z,P,Q,R):-
    bind_hook(X,bind_hook(Y,pv(F,t_evaluate(`(X * Y),Z,P,Q,R)))),
    bind_hook(Z,bind_hook(Y,pv(F,t_evaluate(`(Z / Y),X,P,Q,R)))),
    bind_hook(Z,bind_hook(X,pv(F,t_evaluate(`(Z / X),Y,P,Q,R)))).

t_mod(X,Y,Z,P,Q,R):-
    bind_hook(X,bind_hook(Y,t_evaluate(`(X mod Y),Z,P,Q,R))).

% EVALUATE ARITHMETIC OPERATOR
% evaluate/5
evaluate(E,V,P,Q,R):-
    evaluate(E,T),
    equal(V,T,P,Q,R).
% temporary evaluate/5
t_evaluate(E,V,P,Q,R):-
    evaluate(E,T),
    t_equal(T,V,P,Q,R).

% :- public evaluate/2.
evaluate(`(X + Y),Z):-!,
    Z is X + Y.
evaluate(`(X - Y),Z):-!,
    Z is X - Y.
evaluate(`(X / Y),Z):-!,
    Z is X / Y.
evaluate(`(X * Y),Z):-!,
    Z is X * Y.
evaluate(`(X mod Y),Z):-!,
    Z is X mod Y.


%%%%%% ARITHMETIC EXPRESSION 

% exp/5
exp(X,Y,P,Q,R):-
    unbound(X),!,
    t_equal(X,Y,P,Q,R).
exp(`(X + Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    add(T,U,Z,P3,Q3,R).
exp(`(X - Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    add(Z,U,T,P3,Q3,R).
exp(`(X * Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    multiply(T,U,Z,P3,Q3,R).
exp(`(X / Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    multiply(U,Z,T,P3,Q3,R).
exp(`(X mod Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    exp(X,T,P1,Q1,R),
    exp(Y,U,P2,Q2,R),
    mod(T,U,Z,P3,Q3,R).
exp(X,Y,P,Q,R):-
    t_equal(X,Y,P,Q,R).

    % exp/5 of temporary equality
t_exp(X,Y,P,Q,R):-
    unbound(X),!,
    t_equal(X,Y,P,Q,R).
t_exp(`(X + Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_add(T,U,Z,P3,Q3,R).
t_exp(`(X - Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_add(Z,U,T,P3,Q3,R).
t_exp(`(X * Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_multiply(T,U,Z,P3,Q3,R).
t_exp(`(X / Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_multiply(U,Z,T,P3,Q3,R).
t_exp(`(X mod Y),Z,P,Q,R):-!,
    and_cil(P1,P2,A),
    and_cil(A,P3,P),
    and_cil(Q1,Q2,S),
    and_cil(S,Q3,Q),
    t_exp(X,T,P1,Q1,R),
    t_exp(Y,U,P2,Q2,R),
    t_mod(T,U,Z,P3,Q3,R).
t_exp(X,Y,P,Q,R):-
    t_equal(X,Y,P,Q,R).

%
% utilities
%

add_variables(X,L,L):-
    eq_mem(X,L),!.
add_variables(X,L,[X|L]).

get_variables(X,L,LL):-
    unbound(X),!,
    add_variables(X,L,LL).
get_variables([X|R],L,LL):-!,
    get_variables(X,L,LT),!,
    get_variables(R,LT,LL).
get_variables(X,L,LL):-
    pst(X),!,
    get_pst_variables(X,L,LL).
get_variables(X,L,LL):-
    stack_vector(X,N),!,
    get_variables_vector(N,X,L,LL).
get_variables(_,L,L).

get_pst_variables(x(P,T),L,LL):-
    unbound(P),!,
    get_tree_variables(T,L,LL).
get_pst_variables(x(P,_),L,LL):-!,
    get_pst_variables(P,L,LL).

get_tree_variables(T,L,L):-
    unbound(T),!.
get_tree_variables(t((_,V),S,B),L,LL):-
    get_tree_variables(S,L,TL1),
    get_variables(V,TL1,TL2),!,
    get_tree_variables(B,TL2,LL).

pst(X):-
    stack_vector(X,3),
    first(X,F),
    F == x,!.


get_variables_vector(0,_,L,L):-!.
get_variables_vector(N,X,L,LL):-
    M is N - 1,
    vector_element(X,M,E),
    get_variables(E,L,LT),!,
    get_variables_vector(M,X,LT,LL).

eq_mem(X,[Y|_]):-
    X =:= Y,!.
eq_mem(X,[_|R]):-
    eq_mem(X,R).


%
% equal/5
%
equal(_,_,_,_,W):-
    bound(W),!.
equal(X,Y,V,R,W):-
    bound(V),!,
    equal1(X,Y,V,R,W).
equal(X,Y,Z,R,S):-
    unbound(Y),!,
    freeze(Y,Z,equal(X,Y,Z,R,S)).
equal(X,Y,Z,R,S):-
    unbound(X),!,
    freeze(X,Z,equal(X,Y,Z,R,S)).
equal([A|X],[B|Y],Z,R,S):-!,
    and_cil(C,D,Z),
    and_cil(P,Q,R),
    equal(A,B,C,P,S),!,
    equal(X,Y,D,Q,S).
equal(X,Y,Z,R,S):-
    stack_vector(X,N),
    stack_vector(Y,N),!,
    equal(N,X,Y,Z,R,S). 
equal(X,Y,Z,R,S):-     % atomic,string,heap,etc
    X = Y,!,
    equal(0,X,Y,Z,R,S).
equal(_,_,false,false,_).

equal(_,_,_,_,_,S):-
    bound(S),!.
equal(0,_,_,T,T,_):-!,
    T = true.
equal(I,X,Y,Z,R,S):-!,
    J is I - 1,
    and_cil(C,D,Z),
    and_cil(P,Q,R),
    vector_element(X,J,A),
    vector_element(Y,J,B),
    equal(A,B,C,P,S),
    equal(J,X,Y,D,Q,S).

equal1(X,Y,true,U,_):-!,
    unify_cil(X,Y),  %% <-  X = Y
    U = true.
equal1(X,Y,false,Q,R):-
    bind_hook(Q,(Q = false)),
    bind_hook(P,P=Q), %% <- P = Q
    bind_hook(X,bind_hook(Y,equal(X,Y,P,Q,R) ) ).

%
% t_equal/5
%
t_equal(_,_,_,_,W):-
    bound(W),!.
t_equal(X,Y,V,R,W):-
    bound(V),!,
    t_equal1(X,Y,V,R,W).
t_equal(X,Y,Z,R,S):-
    unbound(X),!,
    freeze(X,Z,t_equal(X,Y,Z,R,S)).
t_equal(X,Y,Z,R,_):-
    unbound(Y),!,
    X = Y,
    Z = R,
    Z = true. 
t_equal(X,X,Z,R,_):-!,  %% ??? X,X OK ???
    Z = R,
    Z = true.
t_equal(_,_,false,false,_).

t_equal1(X,Y,true,U,_):-!,
    X = Y,
    U = true.
t_equal1(X,Y,false,Q,R):-
    bind_hook(Q,(Q = false)),
    bind_hook(P,P = Q), %% <- P = Q
    bind_hook(X,bind_hook(Y,t_equal(X,Y,P,Q,R))).


assign(X,Y,Z):-
    assign(X,Y,Z,Z).
    % Y must not have multiple occurrences of a variable

assign(_,_,_,S):-
    bound(S),!.
assign(X,Y,Z,_):-
    unbound(Y),!,
    X = Y,
    Z = true.
assign(X,Y,Z,S):-
    unbound(X),!,
    bind_hook(X,assign(X,Y,Z,S)).
assign([X|P],[Y|Q],Z,S):-!,
    and_cil(A,B,Z),
    assign(X,Y,A,S),!,
    assign(P,Q,B,S).
assign(X,Y,Z,S):-
    stack_vector(X,N),
    stack_vector(Y,N),!,
    assign(N,X,Y,Z,S).
assign(X,Y,Z,S):-       % atomic,heap,string,etc
    X == Y,
    assign(0,X,Y,Z,S),!.
assign(_,_,Z,_):-
    unify(Z,false).

assign(_,_,_,_,S):-
    bound(S),!.
assign(0,_,_,Z,_):-!,
    unify(Z,true).
assign(I,X,Y,Z,S):-!,
    J is I - 1,
    and_cil(C,D,Z),
    vector_element(X,J,A),
    vector_element(Y,J,B),
    assign(A,B,C,S),
    assign(J,X,Y,D,S).

%
% syntax
%
:-define constraint/syntax.
:-view predicate_syntax/2.

predicate_syntax(constr(C), ``('$skip'(constr(``(C))))):-!.
predicate_syntax(constr(C,R), ``('$skip'(constr(``(C),R)))):-!.
predicate_syntax(constr(C,E,R), ``('$skip'(constr(``(C),E,R)))):-!.
predicate_syntax(constr(C,E,R,S), ``('$skip'(constr(``(C),E,R,S)))):-!.

%
% constraint operator
%
:-define constraint/operator.
:-view op/3.
op(700,xfx,(=)).
op(700,xfx,(==)).
op(700,xfx,(\==)).
op(700,xfx,(=\=)).
op(700,xfx,(=:=)).
op(700,xfx,(>)).
op(700,xfx,(<)).
op(500,yfx,(+)).
op(500,yfx,(-)).
op(400,yfx,(*)).
op(400,yfx,(/)).
op(400,xfx,(mod)).
op(1040,yfx,('->')).
op(1040,yfx,('<->')).
