/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
#include <math.h>
#include <stdlib.h>
#include <stream.h>
#include "matrix.h"


matrix::matrix(int M,int N)
{
  if (M <= 0 || N <= 0) {
    ::cerr << "wrong matix size\n";
    return;
  }
  Nsize = M * N;
  DimI = M; 
  DimJ = N;
  D = new double[Nsize];
  for (int I = 0;I < M;++I) {
    for (int J = 0;J < N;++J) {
      D[I*DimJ + J] = 0.0;
    }
  }
}

matrix::matrix(void)
{
  Nsize = 0;
  D = (double*)0;
  DimI = 0;
  DimJ = 0;
}

matrix::matrix(matrix& A)
{
  Nsize = 0;
  copy(A);
}

matrix::~matrix(void)
{
  if (Nsize >0) {
    delete D;
  }
}

void matrix::zap(void)
{
  delete D;
}

void matrix::copy(matrix& A)
{
  int masize = A.sizeN();
  if (Nsize >0) {
    if (Nsize != masize) {
      delete D;
      D = new double [masize];
    }
  } else if (Nsize == 0) {
    D = new double [masize];
  }
  DimI = A.sizeI(); 
  DimJ = A.sizeJ(); 
  Nsize = masize;
  for (int I = 0; I < DimI; I++) {
    for (int J = 0; J < DimJ; J++) {
      D[I*DimJ + J] = A(I,J);
    }
  }
}

void operator *=(matrix& A,double B)
{
  for (int I = 0; I < A.DimI; I++) {
    for (int J = 0; J < A.DimJ; J++) {
      A.D[I*A.DimJ + J] *= B;
    }
  }
}

void operator /=(matrix& A,double B)
{
  for (int I = 0; I < A.DimI; I++) {
    for (int J = 0; J < A.DimJ; J++) {
      A.D[I*A.DimJ + J] /= B;
    }
  }
}

double & matrix::operator ()(int I, int J)
{
  if (I < 0 || I >= DimI || J < 0 || J >= DimJ) {
      cerr << "out of range. \n";
      exit(1);
  }
  return D[I*DimJ + J];
}      


void matrix::print(void)
{
  cout << "{\n";
  for (int I = 0; I < DimI; ++I) {
    cout << "  {";
    for (int J = 0; J < DimJ; ++J) {
      cout.form("%12.4e", D[I*DimJ + J]);
    }
    cout << "}\n";
  }
  cout <<  "}\n";
}

matrix matrix::tr(void)
{
  matrix C(DimJ,DimI);
  for ( int I = 0; I < DimI; ++I) {
    for ( int J = 0; J < DimJ; ++J) {
      C(J,I) = D[I*DimJ + J];
    }
  }
  return C;
}

matrix  matrix::inverse(void)
{
  if (DimI != DimJ) {
    cerr.form("matrix sizes not matching \n");
    exit(1);	  
  }
  matrix C(DimI,DimJ);
  for (int I = 0; I < DimI; ++I) {
    for (int J = 0; J < DimJ; ++J) {
      C(I,J) = D[I*DimJ + J];
    }
  }
  int p[DimI],q[DimI];
  int L = 0, k = 0;
  for (int i=0 ; i<DimI; ++i) {
    p[i] = q[i] = 0;
  }
  for (k=0 ; k<DimI; ++k) {
    L = pivotsearch(C,q,k);
    p[k]=L;
    q[L]=k;
    for (int j=0;j<DimI;++j) {         /* Sweep out begin */
	if (j==k) {
	  continue;
	}
	C(L,j) /= C(L,k);
	for (int i=0;i<DimI; ++i) {
	  if (i==L) {
	    continue;
	  }
	  C(i,j) -= ( C(L,j) * C(i,k) );
	}
      }                         /* Sweep out end */
      C(L,k) = (1 / C(L,k));      /* K-th Column Begin */
      for (j=0; j<DimI; ++j) {
	if (j==L) {
	  continue;
	}
	C(j,k) = -(C(j,k) * C(L,k));
      }                                 /* K-th Column End */
    }
  
  for (k=0; k<DimI; ++k) {                   /* ReArrange Begin */
    L = p[k];
    if (k==L) {
      continue;
    }
    for (int j=0; j<DimI; ++j) {
      double sp = C(L,j);
      C(L,j) = C(k,j);
      C(k,j) = sp;
    }
    for (int i=0; i<DimI; ++i)  {
      double sp = C(i,k);
      C(i,k) = C(i,q[k]);
      C(i,q[k]) = sp;
    }
    p[q[k]] = L;
    p[k] = k;
    q[L] = q[k];
    q[k] = k;
  }                                   /* ReArrange End */
  return C;
}


void matrix::changeline(int m1, int m2)
{
  double temp;
  for (int i=0; i<DimJ; i++) {
    temp = D[DimJ*m1 + i];
    D[DimJ*m1 + i] = D[DimJ*m2 + i];
    D[DimJ*m2 + i] = temp;
  }
}

double matrix::innerproduct(int n, double* u, double* v)
{
  double s = 0.0;
  int n5 = n % 5;
  double* wu;
  double* wv;
  wu = u;
  wv = v;
  for (int i=0; i<n5; i++,wu++,wv++){
    s += (*wu)*(*wv);
  }
  for (i = n5; i<n; i+=5) {
    wu = (u+i);
    wv = (v+i);
    s += (*wu)*(*wv); wu++;wv++;
    s += (*wu)*(*wv); wu++;wv++;
    s += (*wu)*(*wv); wu++;wv++;
    s += (*wu)*(*wv); wu++;wv++;
    s += (*wu)*(*wv); 
  }
  return s;
}

double matrix:: householder(int n,double* x)
{
  double s = sqrt(innerproduct(n,x,x));
  /*  $@Fb@Q$NJ?J}:,!"$9$J$o$A%Y%/%H%k(Jx$@$NBg$-$5(J */
  if ( s != 0) {
    if ( x[0] < 0) s = -s;
    x[0] += s;
    double t = 1/sqrt(x[0] * s);
    for (int i=0; i<n ; i++) x[i] *= t;
  }
  return -s;
}

void matrix::tridiagonalize(double* d, double* e)
{
  double* v;
  double* w;
  v = new double[DimJ];
  w = new double[DimJ];
  for (int k=0; k<DimI-2; ++k) {
    for (int temp=0;temp<DimJ;temp++) {
      v[temp] = D[DimJ*k + temp];
    }
    d[k] = v[k];
    e[k] = householder(DimI-k-1,&v[k+1]);
    if (e[k] == 0) {
      continue;
    }
    for (int i=k+1; i<DimI; ++i) {
      double s = 0.0;
      for (int j=k+1; j<i; j++) s += D[DimJ*j+i] * v[j];
      for (j=i; j<DimI; j++)    s += D[DimJ*i+j] * v[j];
      d[i] = s;
    }
    double t = innerproduct(DimI-k-1, &v[k+1], &d[k+1]) /2;
    for (i=DimI-1; i>k; i--) {
      double p = v[i];
      double q = d[i] - t * p;
      d[i] = q;
      for (int j=i; j<DimI; j++) {
	D[DimJ*i+j] -= p * d[j] + q * v[j];
      }
    }
    for (temp=0;temp<DimJ;temp++) {
      D[DimJ*k + temp] = v[temp];
    }
  }
  if (DimI >= 2) {
    d[DimI-2] = D[DimJ*(DimI-2) + (DimI-2)];
    e[DimI-2] = D[DimJ*(DimI-2) + (DimI-1)];
  }
  if (DimI >= 1) d[DimI-1] = D[DimJ*(DimI-1) + (DimI-1)];
  for (k=DimI-1; k>=0; k--) {
    for (int temp=0;temp<DimJ;temp++) {
      v[temp] = D[DimJ*k + temp];
    }
    if (k < DimI-2) {
      for (int i=k+1; i<DimI; i++) {
	for (int temp=0;temp<DimJ;temp++) w[temp] = D[DimJ*i + temp];
	double t = innerproduct(DimI-k-1, &v[k+1], &w[k+1]);
	for (int j=k+1; j<DimI; ++j) {
	  w[j] -= t * v[j];
	}
	for (temp=0; temp<DimJ; temp++) {
	  D[DimJ*i + temp] = w[temp];
	}
      }
    }
    for (int i=0; i<DimI; i++) {
      v[i] = 0;
    }
    v[k] = 1;
    for (temp=0;temp<DimJ;temp++) {
      D[DimJ*k + temp] = v[temp];
    }
  }
}

matrix operator-(matrix& A)
{
  int mai = A.sizeI(); 
  int maj = A.sizeJ(); 
  matrix C(mai,maj);
  for ( int I = 0; I < mai; ++I) {
    for ( int J = 0; J < maj; ++J) {
      C(I,J) = - A(I,J);
    }
  }
  return C;
}      

matrix operator+(matrix& A)
{
  return matrix(A);
}      

matrix operator+(matrix& A, matrix& B)
{
  int mai = A.sizeI(); 
  int maj = A.sizeJ(); 
  int mbi = B.sizeI(); 
  int mbj = B.sizeJ(); 
  if (mai != mbi || maj != mbj) {
    cerr << "matrix operator+ \nmatrix sizes not matching \n";
    exit(1);
  }
  matrix C(mai,maj);
  for ( int I = 0; I < mai; I++) {
    for ( int J = 0; J < maj; J++) {
      C(I,J) = A(I,J) + B(I,J);
    }
  }
  return C;
}

matrix operator-(matrix& A, matrix& B)
{
  int mai = A.sizeI(); 
  int maj = A.sizeJ(); 
  int mbi = B.sizeI(); 
  int mbj = B.sizeJ(); 
  if (mai != mbi || maj != mbj) {
    cerr << "matrix operator- \nmatrix sizes not matching \n";
    exit(1);
  }
  matrix C(mai,maj);
  for ( int I = 0; I < mai; ++I) {
    for ( int J = 0; J < maj; ++J) {
      C(I,J) = A(I,J) - B(I,J);
    }
  }
  return C;
}

matrix operator*(matrix& A, matrix& B)
{
  int mai = A.sizeI(); 
  int maj = A.sizeJ(); 
  int mbi = B.sizeI(); 
  int mbj = B.sizeJ(); 
  if (maj != mbi) {
    cerr << "matrix operator* \nmatrix sizes not matching \n";
    exit(1);
  }
  matrix C(mai,mbj);
  for (int I = 0; I < mai; I++) {
    for (int J = 0; J < mbj; J++) {
      double Cij = 0.00;
      for (int K = 0; K < maj; K++) {
	Cij += (A(I,K) * B(K,J));
      }
      C(I,J) = Cij;
    }
  }
  return C;
}

matrix operator*(double &K, matrix& A)
{
  int mai = A.sizeI(); 
  int maj = A.sizeJ(); 
  matrix C(mai,maj);
  for (int i=0;i<mai; ++i) {
    for (int j=0; j<maj ; ++j) {
      C(i,j) = K * A(i,j);
    }
  }
  return C;
}

matrix operator*(matrix& A, double& K)
{
  return K * A;
}

matrix operator/(matrix& A, double& K)
{
  return (1.0/K) * A;
}

void operator=(matrix& A,matrix& B)
{
  A.copy(B);
}
