/* ---------------------------------------------------------- 
%   (C)1992 Institute for New Generation Computer Technology 
%       (Read COPYRIGHT for detailed information.) 
----------------------------------------------------------- */
/* 
 *  Error handling routines.
 *  
 *  The functions in this file are independent of any application
 *  variables, and may be used with any C program.
 *  Either of the names CLIENT or SERVER may be defined when compiling
 *  this function. If neither are defined, we assume CLIENT.
 */

#include  <stdio.h>
#include  <varargs.h>

#ifdef  CLIENT
#ifdef  SERVER
cant define both CLIENT and SERVER
#endif
#endif


#ifndef  CLIENT
#ifndef  SERVER
#define  CLIENT
#endif
#endif


char      *pname = NULL;

#ifdef  CLIENT     /* these all output to stderr */

/*
 *  Fatal error. Print a message and terminate.
 *  Don't dump core and don't print the system's errno value.
 */

/*VARARGS1*/
err_quit(va_alist)
va_dcl
{
  va_list   args;
  char      *fmt;

  va_start(args);
  if(pname != NULL)  fprintf(stderr,"%s: ",pname);
  fmt = va_arg(args, char *);
  vfprintf(stderr, fmt, args);
  fputc('\n',stderr);
  va_end(args);
  socketClose();
  exit(1);
}


/*
 *  Fatal error related to a system call. Print a message and terminate.
 *  Don't dump core, but do print the system's errno value and its
 *  associated message.
 */

/*VARARGS1*/
err_sys(va_alist)
va_dcl
{
  va_list   args;
  char      *fmt;

  va_start(args);
  if(pname != NULL)  fprintf(stderr,"%s: ",pname);
  fmt = va_arg(args, char *);
  vfprintf(stderr, fmt, args);
  va_end(args);
  my_perror();
  socketClose();
  exit(1);
}


/*
 *  Recoverable error.  Print a message, and return to caller.
 */

/*VARARGS1*/
err_ret(va_alist)
va_dcl
{
  va_list   args;
  char      *fmt;

  va_start(args);
  if(pname != NULL)  fprintf(stderr,"%s: ",pname);
  fmt = va_arg(args, char *);
  vfprintf(stderr, fmt, args);
  va_end(args);
  my_perror();

  fflush(stdout);    fflush(stderr);
  return;
}



/*
 *  Fatal error. Print a message, dump core and terminate.
 */

/*VARARGS1*/
err_dump(va_alist)
va_dcl
{
  va_list   args;
  char      *fmt;

  va_start(args);
  if(pname != NULL)  fprintf(stderr,"%s: ",pname);
  fmt = va_arg(args, char *);
  vfprintf(stderr, fmt, args);
  va_end(args);
  my_perror();
  fflush(stdout);    fflush(stderr);
  abort();
}


/*
 *  Print the UNIX error value.
 */
my_perror()
{
  char  *sys_err_str();
  
  fprintf(stderr,"%s\n", sys_err_str());
}

#endif   /* CLIENT */

#ifdef  SERVER
#ifdef  BSD
/* 
 *  Under BSD, these server routines use the syslog(3) facility.
 *  They don't append a newline, for example.
 */
#include  <syslog.h>
#else   /* not BSD */
/* 
 *  There really ought to be a better way to handle server logging 
 *  under SYSTEM V.
 */
#define  syslog(a,b)      fprintf(stderr,"%s\n",(b));
#define  openlog(a,b,c)   fprintf(stderr,"%s\n",(a));
#endif  /* BSD */

char  emesgstr[255] = {0};   /* used by all server routines */

/* 
 * Identify ourself, for syslog() messages.
 *
 * LOG_PID is an option that says prepend each message with our pid.
 * LOG_CONS is an option that says write to console if unable to send 
 * the message to syslogd.
 * LOG_DAEMON is our facility.
 */

err_init(ident)
char  *ident;
{
  openlog(ident,(LOG_PID|LOG_CONS), LOG_DAEMON);
}


/*
 *  Fatal error. Print a message and terminate.
 *  Don't dump core and don't print the system's errno value.
 */

/*VARARGS1*/
err_quit(va_alist)
va_dcl
{
  va_list   args;
  char      *fmt;

  va_start(args);
  fmt = va_arg(args, char *);
  vsprintf(emesgstr, fmt, args);
  va_end(args);
  syslog(LOG_ERR,emesgstr);
  socketClose();
  exit(1);
}


/*
 *  Fatal error related to a system call. Print a message and terminate.
 *  Don't dump core, but do print the system's errno value and its
 *  associated message.
 */

/*VARARGS1*/
err_sys(va_alist)
va_dcl
{
  va_list   args;
  char      *fmt;

  va_start(args);
  fmt = va_arg(args, char *);
  vsprintf(emesgstr, fmt, args);
  va_end(args);
  my_perror();
  syslog(LOG_ERR,emesgstr);
  socketClose();
  exit(1);
}


/*
 *  Recoverable error.  Print a message, and return to caller.
 */

/*VARARGS1*/
err_ret(va_alist)
va_dcl
{
  va_list   args;
  char      *fmt;

  va_start(args);
  fmt = va_arg(args, char *);
  vsprintf(emesgstr, fmt, args);
  va_end(args);
  my_perror();
  syslog(LOG_ERR,emesgstr);
  return;
}


/*
 *  Fatal error. Print a message, dump core and terminate.
 */

/*VARARGS1*/
err_dump(va_alist)
va_dcl
{
  va_list   args;
  char      *fmt;

  va_start(args);
  fmt = va_arg(args, char *);
  vsprintf(emesgstr, fmt, args);
  va_end(args);
  my_perror();
  syslog(LOG_ERR,emesgstr);
  abort();
}


/*
 *  Print the UNIX error value.
 *  We just append it to the end of the emesgstr[] array.
 */
my_perror()
{
  char  *sys_err_str();
  int   len;

  len = strlen(emesgstr);
  sprintf(emesgstr+len," %s", sys_err_str());
}

#endif   /* SERVER */


/*  remainder is for both CLIENT and SERVER  */

extern int  errno;           /* UNIX error number */
extern int  sys_nerr;        /* # of error message strings in sys table */ 
extern char *sys_errlist[];  /* the system error message table */

#ifdef  SYS5
int  t_errno;  /* in case call is using TLI, these are "tentative definition"
                  else they are "definition" */
int  t_nerr;
#endif /* SYS5 */

char  *sys_err_str()
{
  static char msgstr[200];
  
  if(errno!=0) {
     if(errno>0  &&  errno<sys_nerr)
        sprintf(msgstr,"(%s)",sys_errlist[errno]);
     else
        sprintf(msgstr,"(errno = %d)", errno);
  }
  else
     msgstr[0]=0;

#ifdef  SYS5
  if(t_errno!=0) {
     char  tmsgstr[100];

     if(t_errno>0  &&  t_errno<t_nerr)
        sprintf(tmsgstr,"(t_errno = %d)", t_errno);
     strcat(msgstr,tmsgstr);
  }
#endif /* SYS5 */

  return msgstr;   /* For this reason, msgstr[] must be static */
}


/*
 *  Return a string containing some additional information after a
 *  host name or address lookup error --- gethostbyname(), gethostbyaddr().
 */

int  h_errno;         /* host error number */
int  h_nerr;          /* # of error messag strings */ 

char *host_err_str()
{
  static char msgstr[200];

  if(h_errno!=0) {
     sprintf(msgstr,"(h_errno = %d)", h_errno);
  }
  else
     msgstr[0]=0;

  return msgstr;
}


