;;; $Id: commands.scm,v 1.7 1994/01/24 09:57:55 queinnec Exp $
;;; Copyright (c) 1990-93 by Christian Queinnec. All rights reserved.
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo
;;;                        LiSP2TeX
;;;   Christian Queinnec             or to:  Christian Queinnec
;;;   <queinnec@polytechnique.fr>            <Christian.Queinnec@inria.fr>
;;;   Laboratoire d'Informatique de l'X      INRIA -- Rocquencourt
;;;   Ecole Polytechnique                    Domaine de Voluceau, BP 105
;;;   91128 Palaiseau                        78153 Le Chesnay Cedex
;;;   France                                 France
;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo

;;; This program is distributed in the hope that it will be useful.
;;; Use and copying of this software and preparation of derivative works
;;; based upon this software are permitted, so long as the following
;;; conditions are met:
;;;      o credit to the authors is acknowledged following current
;;;        academic behaviour
;;;      o no fees or compensation are charged for use, copies, or
;;;        access to this software
;;;      o this copyright notice is included intact.
;;; This software is made available AS IS, and no warranty is made about
;;; the software or its performance.

;;;oooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooooo

;;; This file defines the commands offered by LiSP2TeX
;;; These commands are:
;;;    FromFile                 ( file keys... )
;;;    Print                    Sexpression
;;;    PrettyPrint              Sexpression
;;;    PrettyFromFile           ( file keys... )
;;;    Eval                     Sexpression
;;;    ShowEvaluationOf		Sexpression

;;; Any command <C> is associated to two functions: run-<C> and
;;; command-<C>. Whenever a directive is found in the input stream,
;;; the function run-<C> is invoked to read some parameters and apply
;;; command-<C> on them.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; Print
;;; Read and print a Sexpression. By default (see port-XXX.scm) try to
;;; respect input case.

(define (run-Print stream)
  (command-Print (numbering-read stream)) )

(define (command-Print exp)
  (pp-format stdout-port *lisp-excerpt* exp) )

(define *lisp-excerpt* "{\\WithLispFont{~A}}")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; PrettyPrint
;;; Read and pretty-print a Sexpression. By default (see port-XXX.scm)
;;; try to respect input case.

(define (run-PrettyPrint stream)
  (command-PrettyPrint (numbering-read stream)) )

(define (command-PrettyPrint exp)
  (pp-greekify exp stdout-port) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; PrettyPrintFromFile
;;; Read from a file and pretty-print a Sexpression. As for FromFile
;;; there is a concept of current file (held in the global variable
;;; *the-current-scanned-file*).

(define *prettyfromfile-error1*
  "~%LiSP2TeX (PrettyFromFile) error: Does not begin with a string: ~A " )
(define *prettyfromfile-error2*
"~%LiSP2TeX (PrettyFromFile) error: invalid argument ~A " )

(define (run-PrettyFromFile stream)
  (let ((e (numbering-read stream)))
    (if (pair? e)
        (if (string? (car e))
            (command-PrettyFromFile (car e) (cdr e))
            (command-error *prettyfromfile-error1* e) )
        (command-error *prettyfromfile-error2* e) ) ) )

(define *prettyfromfile-error3*
  "~%LiSP2TeX (PrettyFromFile) error: No current scanned file " )
(define *prettyfromfile-error4*
  "~%LiSP2TeX (PrettyFromFile) error: Cannot find file ~A~%"  )

(define (command-PrettyFromFile filename keys)
  (let ((sf (if (string=? filename "")
		(or *the-current-scanned-file*
		    (begin (command-error *prettyfromfile-error3*)
                           #f ) )
		(find-scanned-file filename) )))
    (if sf
	(let ((ses (mapcan (lambda (key) (find-scanned-expression key sf))
                           keys )) )
	  (set! *the-current-scanned-file* sf)
	  (for-each (lambda (se)
                      (if (eq? se '***MISSING-EXPRESSION***)
                          (pp-greekify (symbol->string se) stdout-port)
                          (pp-greekify (scanned-expression-original se) 
                                       stdout-port ) ) )
		    ses ) )
	(command-error *prettyfromfile-error4* filename) ) ) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; ShowEvaluationOf
;;; Read, print, evaluate an Sexpression then print its result.  Since
;;; there is an evaluation performed by the underlying Scheme, the
;;; expression is read with the preferred case sensibility of the
;;; underlying Scheme.

(define (run-ShowEvaluationOf stream)
  ;; save the current settings of the reader
  (let* ((ri *respect-input*)
         (ui *uppercase-input*)
         (li *lowercase-input*)
         (start-line (numbering-stream-line stream)) )
    ;; sets the reader to the way eval prefers
    (set! *respect-input*   *evaluator-preferred-respect-input*)
    (set! *uppercase-input* *evaluator-preferred-uppercase-input*)
    (set! *lowercase-input* *evaluator-preferred-lowercase-input*)
    (let* ((exp (numbering-read stream))
           (end-line (numbering-stream-line stream)) )
      (pp-format stdout-port *showevaluationof-input* exp)
      ;; restaure the reader
      (set! *respect-input*   ri)
      (set! *uppercase-input* ui)
      (set! *lowercase-input* li)
      ;; and only now evaluates the read expression
      (let ((result (LiSP2TeX-eval exp)))
        (if (and (pair? result)
                 (eq? (car result) '***EVALUATION-ERROR***) )
            (command-error *eval-error-format*
                           start-line
                           end-line
                           exp
                           (cdr result) )
            (pp-format stdout-port 
                       *showevaluationof-output* 
                       result ) ) ) ) ) )

(define *showevaluationof-input*    "~A ")
(define *showevaluationof-output*   " gives ~A")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; Eval
;;; Read and evaluates a Sexpression. If an error is detected then
;;; LiSP2TeX-eval recturns a specific list. LiSP2TeX-eval is a non
;;; portable function defined in port-XXX.scm. Since there is an
;;; evaluation performed by the underlying Scheme, the expression is
;;; read with the preferred case sensibility of the underlying Scheme.

(define *eval-error-format*
  "~%LiSP2TeX (Eval) error (line ~A-~A): ~A~%original message: ~A " )

(define (run-Eval stream)
  ;; save the current settings of the reader
  (let* ((ri *respect-input*)
         (ui *uppercase-input*)
         (li *lowercase-input*)
         (start-line (numbering-stream-line stream)) )
    ;; sets the reader to the way eval prefers
    (set! *respect-input*   *evaluator-preferred-respect-input*)
    (set! *uppercase-input* *evaluator-preferred-uppercase-input*)
    (set! *lowercase-input* *evaluator-preferred-lowercase-input*)
    (let* ((exp (numbering-read stream))
           (end-line (numbering-stream-line stream)) )
      ;; restaure the reader
      (set! *respect-input*   ri)
      (set! *uppercase-input* ui)
      (set! *lowercase-input* li)
      ;; and only now evaluates the read expression
      (let ((result (LiSP2TeX-eval exp)))
        (when (and (pair? result)
                   (eq? (car result) '***EVALUATION-ERROR***) )
          (command-error *eval-error-format*
                         start-line
                         end-line
                         exp
                         (cdr result) ) ) ) ) ) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; FromFile
;;; There is a concept of current file. Whenever you use the FromFile
;;; command a null string in lieu of the filename refers to the current file.

(define *the-current-scanned-file* #f)

;;; Find a scanned file and scan it if not yet done. Set it as the
;;; current file. It is left to the implementation to detect if the file
;;; does not exist since it is not possible in standard Scheme to know it.

(define find-scanned-file 
  (let ((already-scanned-files '()))
    (lambda (filename)
      (define (lookup sfs)
        (cond ((null? sfs)
	       (let ((file (search-file filename *path*)))
		 (cond 
		  (file 
		   (command-echo "[Scan ~U" filename)
		   (set! already-scanned-files
			 (cons (scan-file filename)
			       already-scanned-files ) )
		   (command-echo "]")
		   (set! *the-current-scanned-file*
			 (car already-scanned-files) ) )
		  (else (set! *the-current-scanned-file* #f) ) ) )
	       *the-current-scanned-file* )
	      ((string=? filename (scanned-file-filename (car sfs)))
               (car sfs) )
              (else (lookup (cdr sfs))) ) )
      (lookup already-scanned-files) ) ) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; FromFile
;;; Include a definition, it is reproduced as it is.

(define *fromfile-error1* 
  "~%LiSP2TeX (FromFile) error: Does not begin with a string: ~A " )
(define *fromfile-error2*
  "~%LiSP2TeX (FromFile) error: Invalid number of arguments ~A " )

(define (run-FromFile stream)
  (let ((e (numbering-read stream)))
    (if (pair? e)
        (if (string? (car e))
	    (command-FromFile (car e) (cdr e))
	    (command-error *fromfile-error1* e) )
        (command-error *fromfile-error2* e) ) ) )

(define *fromfile-error3*
  "~%LiSP2TeX (FromFile) error: No current scanned file ")
(define *fromfile-error4*
  "~%LiSP2TeX (FromFile) error: Cannot find file ~A~%" )
                       
(define (command-FromFile filename keys)
  (let ((sf (if (string=? filename "")
		(or *the-current-scanned-file*
		    (begin (command-error *fromfile-error3*)
                           #f ) )
		(find-scanned-file filename) )))
    (if sf
	(let ((ses (mapcan (lambda (key) (find-scanned-expression key sf))
                           keys )) )
	  (set! *the-current-scanned-file* sf)
	  (for-each (lambda (se)
                      (unless (eq? se '***MISSING-EXPRESSION***)
                        (pp-format 
                         stdout-port 
                         *lisp-citation*
                         (scanned-expression-type se)
                         (scanned-expression-key se) ) ) )
		    ses )
	  (when (pair? ses) (pp-format stdout-port *beginlisp-block*))
	  (for-each (lambda (se)
                      (pp-format stdout-port *interlisp-block*)
                      (if (eq? se '***MISSING-EXPRESSION***)
                          (pp-format stdout-port "~U" (symbol->string se))
                          (pp-format stdout-port "~U" 
                           (substring (scanned-file-string sf)
                                      (scanned-expression-real-start se)
                                      (scanned-expression-end se) ) ) ) )
		    ses )
	  (when (pair? ses) (pp-format stdout-port *endlisp-block*)) )
	(command-error *fromfile-error4* filename ) ) ) )

(define *lisp-citation*   "\\LispCite{~U}{~U}~%")
(define *beginlisp-block* "\\Lisp")
(define *interlisp-block* " ~%")
(define *endlisp-block*   " \\EndLisp ")

;;; Find the expressions matched by key in scanned-file.
;;; A key can be  
;;;    a symbol : find THE Sexpression that defines this name,
;;;    a pair (considered as a pattern): find ALL Sexpressions matched.

(define (find-scanned-expression key scanned-file)
  (let* ((ses (scanned-file-scanned-expressions scanned-file))
         (found #f)
         (max (vector-length ses)) )
    (define (lookup i)
      (if (< i max)
          (let ((se (vector-ref ses i)))
            (cond ((and (symbol? key)
                        (scanned-expression-key se)
                        (string-ci=? (symbol->string key)
                                     (scanned-expression-key se) ) )
                   (set! found #t)
                   (list se) )
                  ((and (pair? key)
                        (naive-match (scanned-expression-original se) key) )
                   (set! found #t)
                   (cons se (lookup (+ 1 i))) )
                  (else (lookup (+ 1 i))) ) )
          (if found
              '()
              (begin
                (command-error *find-definition-error* key)
                '(***MISSING-EXPRESSION***) ) ) ) )
    (lookup 0) ) )

(define *find-definition-error*
  "~%LiSP2TeX ([Pretty]FromFile) error: Cannot find definition of ~A " )

;;; This naive pattern matcher recognizes ?- and ... at any level.
;;; ?- matches any Sexp while ... matches any sequence of Sexp. This
;;; latter can only be used in the context of a list.

(define *mono-Sexp-pattern* '?-)
(define *poly-Sexp-pattern* '...)

(define (naive-match expression pattern)
  (define (naive-match-list expressions patterns)
    (if (pair? patterns)
        (if (equal? (car patterns) *poly-Sexp-pattern*)
             ; accepts any sequence of things
            (or (naive-match-list expressions (cdr patterns))
                (and (pair? expressions)
                     (naive-match-list (cdr expressions) patterns) ) )
            (and (pair? expressions)
                 (naive-match (car expressions) (car patterns))
                 (naive-match-list (cdr expressions) (cdr patterns)) ) )
        (naive-match expressions patterns) ) )
  (or (equal? pattern *mono-Sexp-pattern*)              ; accepts anything
      (if (equal? pattern *poly-Sexp-pattern*)
          (command-error *match-...-error*
                         *poly-Sexp-pattern* pattern )
          (if (pair? pattern) 
              (naive-match-list expression pattern)
              (equal? expression pattern) ) ) ) )

(define *match-...-error*
  "Bad use of ~A in pattern" )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; Utility functions

;;; These options allow to display the progress of the LiSP2TeX filter.
;;; Progress is reported on stderr not to be confused with the regular
;;; output.

(define *verbose* #f)
(define *ultra-verbose* #f)

;;; Reports some progress
(define (command-echo fmt . args)
  (when *verbose*
    (LiSP2TeX-apply pp-format stderr-port fmt args)
    (flush-all-buffers) ) )

;;; reports some error on both streams (to see it in the output TeX)
(define (command-error fmt . arguments)
  (LiSP2TeX-apply pp-format stdout-port fmt arguments)
  (LiSP2TeX-apply pp-format stderr-port fmt arguments)
  (flush-all-buffers) )

;;; The list of directories where to search (customization or regular) files. 
;;; Only the current directory by default.
(define *path* '("."))

;;; Emit a header (a TeX comment) identifying the version of LiSP2TeX.
;;; It is output on stdout as a TeX comment.

(define *header-format*
  "%%% This file was automatically produced by LiSP2TeX (~U) ~U
%%%                   --- PLEASE DO NOT EDIT --- ~%" )
   
(define (emit-header)
  (pp-format stdout-port *header-format* scm-name version) )

;;; Identify the current version of LiSP2TeX.
(define (emit-version-header)
  (pp-format 
   stderr-port
   "LiSP2TeX (~U) ~U [Christian.Queinnec@INRIA.fr]~%" 
   scm-name version ) )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;; MAIN
;;; The main entry point, handles options then filters stdin.
;;; Scheme->C calls it from the shell with the shell arguments in args.

(define *no-option*
  "~%LiSP2TeX option error: empty option " )
(define *too-much---options*
  "~%LiSP2TeX option error: too much -- options " )
(define *empty-option*
  "~%LiSP2TeX option error: No option after dash" )
(define *cannot-find-file*
  "~%LiSP2TeX option error: Cannot find file ~A " )
(define *unknown-option*
  "~%LiSP2TeX option error: Unrecognized option: ~A " )

(define (LiSP2TeX args)
  ;; Analyse options
  (let ((does--appear? #f)
        (settings '())
        (files '()) )
    (do ((args args (cdr args)))
        ((null? args) #f)
      (cond 
       ;; refuses empty options
       ((= 0 (string-length (car args)))
        (command-error *no-option*) )
       ;; outputs the version identifier
       ((equal? (car args) "-version")(emit-version-header))
       ;; separates the parameter files from the files to filter
       ((equal? (car args) "--")
        (cond (does--appear? 
               (command-error *too-much---options*) )
              (else
               (set! does--appear? #t)
               (set! settings files)
               (set! files '()) ) ) )
       ;; A real option (or combination of options)
       ((char=? (string-ref (car args) 0) #\-)
        (if (> (string-length (car args)) 1)
            (analyze-option (car args) 1)
            (command-error *empty-option*) ) )
       (else
        (set! files (cons (car args) files)) ) ) )
    ;; Reverse the order of directories pushed onto *path*
    (set! *path* (reverse! *path*))
    ;; Identifies the current version of LiSP2TeX on stdout
    (emit-header)
    ;; Load the parameter files. They use *path* 
    (when does--appear?
          (for-each (lambda (filename) 
                      (let ((file (search-file filename *path*)))
                        (if file 
                            (begin
                              (command-echo "[Read customization ~U..."
                                            file )
                              (load-parameter-file file)
                              (command-echo "]~%") )
                            (command-error *cannot-find-file*
                                           filename ) ) ) )
                    (reverse! settings) ) )
    ;; Now handle files to filter or the standard input
    (if (pair? files)
        ;; Handles all mentionned files
        (for-each (lambda (filename)
                    (let ((file (search-file filename *path*)))
                      (if file
                          (call-with-input-file
                              file
                            (lambda (in)
			      (command-echo "[Process ~U..." filename)
                              (look (create-numbering-stream in)
                                    numbering-stream-read-char )
			      (command-echo "]~%") ) )
                          (command-error *cannot-find-file*
                                         filename ) ) ) )
                  (reverse! files) )
        ;; Handles stdin
        (look (create-numbering-stream stdin-port)
              numbering-stream-read-char ) )
    ;; Terminates.
    (flush-all-buffers) ) )

;;; analyze options (options start with "-")

(define (analyze-option arg i)
  (when (< i (string-length arg))
        (cond 
         ;; Add a directory to search
	 ((char=? (string-ref arg i) #\I)
	  (set! *path* (cons (substring arg 
					(+ 1 i)
					(string-length arg) )
			     *path* )) )
         ;; be verbose
	 ((char=? (string-ref arg i) #\v)
	  (set! *verbose* #t)
	  (analyze-option arg (+ 1 i)) )
         ;; be more verbose if possible
	 ((char=? (string-ref arg i) #\V)
	  (set! *ultra-verbose* #t)
	  (set! *verbose* #t)
	  (analyze-option arg (+ 1 i)) )
         ;; converts what is read to uppercase
	 ((char=? (string-ref arg i) #\U)
          (set! *respect-input*    #f)
	  (set! *lowercase-input*  #f)
	  (set! *uppercase-input*  #t) )
         ;; converts what is printed to uppercase
	 ((char=? (string-ref arg i) #\u)
          (set! *respect-output*   #f)
	  (set! *lowercase-output* #f)
	  (set! *uppercase-output* #t) )
	 ((char=? (string-ref arg i) #\L)
          (set! *respect-input*    #f)
	  (set! *lowercase-input*  #t)
	  (set! *uppercase-input*  #f) )
	 ((char=? (string-ref arg i) #\l)
          (set! *respect-output*   #f)
	  (set! *lowercase-output* #t)
	  (set! *uppercase-output* #f) )
	 ((char=? (string-ref arg i) #\=)
          (set! *respect-input*    #t)
	  (set! *lowercase-input*  #f)
	  (set! *uppercase-input*  #f)
          (set! *respect-output*   #t)
	  (set! *lowercase-output* #f)
	  (set! *uppercase-output* #f) )
         ;; Do not emit header
	 ((char=? (string-ref arg i) #\h)
	  (set! emit-header (lambda () #f)) )
	 (else 
	  (command-error *unknown-option*
			 (string-ref arg i) ) ) ) ) )

;;; end of command.scm
