;;; $Id: read.scm,v 1.10 1992/04/03 21:45:10 queinnec beta $
;;; This file contains a Scheme Reader which I hope is conformant to 
;;; Scheme Standard. it is written with light dependencies on the
;;; data structures that are used by the implementation so it is 
;;; probably portable.

;;; The functions read-XYZ have a similar structure:
;;; they take a character (or eof) as first argument, the
;;; stream from which they read and they return whatever 
;;; object is read. 
;;; They are usually programmed as a test against eof followed
;;; by a big selection driven by the current character.

;;; Whenever an end of file is found, the reader tries to finish
;;; its work and usually return this end-of-file or report an error
;;; if the read object is not complete.

;;; This huge function creates a reader, streams are assumed to answer
;;; to peek-char, read-char and read-error.

(define (make-reader peek-char read-char)
  ;; Current-string is a working area where are collected characters of
  ;; symbols or string while being scanned.
  (define current-string (make-extensible-string 100))
  ;; It is reused all the time
  (define (reset-current-string)
    (set-extensible-string-index! current-string 0)
    current-string )
  
  ;; pipe two operations in one: consume the head of the stream 
  ;; and peek the next character. At any time the first character of 
  ;; stream is the poken char.
  (define (read-then-peek-char stream)
    (read-char stream)
    (peek-char stream) )
  
  (lambda (kind stream read-error)   ; kind is 'read or 'skip

    ;; Two operations in one: skip spaces then read an object. 
    (define (skip-and-read-object char stream)
      (read-object (skip-until-object char stream) stream) )

    ;; skip characters until an expression starts, return the last poken char
    (define (skip-until-object char stream)
      (if (eof-object? char)
          char
          (case char
            ((#\space #\newline) 
             (skip-until-object (read-then-peek-char stream) stream) )
            ((#\;)
             (read-after-end-of-line 
              (read-then-peek-char stream) stream skip-until-object ) )
            (else char) ) ) )
 
    ;; Read an object that starts with char.
    (define too-much-closing-bracket
      "Cannot read a closing bracket" )
    (define (read-object char stream)
      (if (eof-object? char)
          char
          (case char
            ((#\')
             (let ((o (skip-and-read-object 
                       (read-then-peek-char stream) stream )))
               (if (eof-object? o) o `(quote ,o)) ) )
            ((#\`)
             (let ((o (skip-and-read-object 
                       (read-then-peek-char stream) stream )))
               (if (eof-object? o) o (list 'quasiquote o)) ) )
            ((#\,)
             (let ((ch (read-then-peek-char stream)))
               (case ch
                 ((#\@)
                  (let ((o (skip-and-read-object (read-then-peek-char stream) 
                                                 stream )))
                    (if (eof-object? o) o 
                        (list 'unquote-splicing o) ) ) )
                 (else 
                  (let ((o (skip-and-read-object ch stream)))
                    (if (eof-object? o) o 
                        (list 'unquote o )) ) ) ) ) )
            ((#\()
             (skip-and-read-list (read-then-peek-char stream) stream) )
            ((#\))
             (read-char stream)		; consume the offending parenthesis.
             (read-error stream too-much-closing-bracket) )
            ((#\")
             (read-string (read-then-peek-char stream) 
                          stream
                          (make-extensible-string 60) ) )
            ((#\#)
             (read-sharp-object (read-then-peek-char stream) stream) )
            ((#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)
             (read-number-or-symbol 
              char stream (reset-current-string) #f #t ) )
            ((#\+ #\-)
             (read-number-or-symbol 
              (read-then-peek-char stream) 
              stream 
              (increment-string char (reset-current-string))
              char 
              #f ) )
            (else
             (read-symbol char stream (reset-current-string)) ) ) ) )

    ;; skip characters until the end of the line and resume the
    ;; fn function on the next character of the next line.
    (define (read-after-end-of-line char stream fn)
      (if (eof-object? char)
          char
          (case char
            ((#\newline)
             (fn (read-then-peek-char stream) stream) )
            (else (read-after-end-of-line 
                   (read-then-peek-char stream) stream fn )) ) ) )

    ;; Strings can be created with an instance of extensible string. The
    ;; string can be augmented with characters (increment-string)
    ;; and finalized with (finalize-string)

    ;; Read a string
    (define premature-eof-in-string
      "Stream ends up while reading a string" )
    (define (read-string char stream string)
      (if (eof-object? char)
          (read-error stream premature-eof-in-string)
          (case char
            ((#\") (read-char stream)
                   (finalize-string string) )
            ((#\\)
             (read-char stream);; consume the backslash
             (let ((ch (read-char stream)))
               (when (eof-object? ch)
                     (read-error stream premature-eof-in-string) )
               (increment-string ch string)
               (read-string (peek-char stream) stream string) ) )
            (else (increment-string char string)
                  (read-string 
                   (read-then-peek-char stream) stream string ) ) ) ) )

    ;; Read a symbol (backslash and bar are allowed).
    ;; Note that dot and quote do not finish a symbol.
    (define premature-eof-in-symbol
      "Stream ends up while reading a symbol" )
    (define (read-symbol char stream symbol)
      (if (eof-object? char)
          (finalize-symbol symbol)
          (case char
            ((#\space #\newline #\( #\) #\; #\") 
             (finalize-symbol symbol) )
            ((#\\)
             (read-char stream);; consume the backslash
             (let ((ch (read-char stream)))
               (when (eof-object? ch)
                     (read-error stream premature-eof-in-symbol) )
               (increment-symbol ch symbol #t)
               (read-symbol (peek-char stream) stream symbol) ) )
            ((#\|)
             (read-barred-symbol (read-then-peek-char stream) 
                                 stream symbol ) )
            (else 
             (increment-symbol char symbol #f)
             (read-symbol
              (read-then-peek-char stream) stream symbol ) ) ) ) )

    ;; Read the part of a symbol which is between bars.
    (define premature-eof-in-barred-symbol
      "Stream ends up while reading a barred symbol" )
    (define (read-barred-symbol char stream symbol)
      (if (eof-object? char)
          (read-error stream premature-eof-in-barred-symbol)
          (case char
            ((#\|) 
             (let ((newchar (read-then-peek-char stream)))
               (when (eof-object? newchar)
                     (read-error stream premature-eof-in-barred-symbol) )
               (case newchar
                 ((#\|)
                  (increment-symbol newchar symbol #t)
                  (read-barred-symbol (read-then-peek-char stream)
                                      stream symbol ) )
                 (else (read-symbol newchar stream symbol)) ) ) )
            ((#\\)
             (read-char stream);; consume the backslash
             (let ((ch (read-char stream)))
               (when (eof-object? ch)
                     (read-error stream premature-eof-in-symbol) )
               (increment-symbol ch symbol #t)
               (read-barred-symbol 
                (peek-char stream) stream symbol ) ) )
            (else (increment-symbol char symbol #t)
                  (read-barred-symbol
                   (read-then-peek-char stream) stream symbol ) ) ) ) )

    ;; Reads a list
    (define premature-eof-in-list
      "Stream ends up while reading a list" )
    (define incorrect-dots
      "Two contiguous dots" )
    (define (skip-and-read-list char stream)
      (if (eof-object? char)
          (read-error stream premature-eof-in-list)
          (case char
            ((#\space #\newline)
             (skip-and-read-list (read-then-peek-char stream) stream) )
            ((#\;)
             (read-after-end-of-line
              (read-then-peek-char stream) stream skip-and-read-list ) )
            (else (read-list char stream)) ) ) )
    (define (read-list char stream)
      (case char
        ((#\))
         (read-char stream)             ; consume the close bracket
         '() )
        ((#\.)
         (let ((ch (read-then-peek-char stream)))
           (if (eof-object? ch)
               (read-error stream premature-eof-in-list)
               (case ch
                 ((#\.)
                  (let ((other-ch (read-then-peek-char stream)))
                    (if (eof-object? other-ch)
                        (read-error stream premature-eof-in-list)
                        (case other-ch
                          ((#\.)
                           (read-char stream) ; consumes third dot
                           (let ((symbol (reset-current-string)))
                             (increment-symbol ch symbol #t)
                             (increment-symbol ch symbol #t)
                             (increment-symbol ch symbol #t)
                             (let* ((s (finalize-symbol symbol))
                                    (d (skip-and-read-list (peek-char stream)
                                                           stream )) )
                               (if (eof-object? d)
                                   d
                                   (cons s d) ) ) ) )
                          (else (read-error stream incorrect-dots)) ) ) ) )
                 (else
                  (read-end-of-list 
                   (skip-and-read-list ch stream) 
                   stream ) ) ) ) ) )
        (else 
         (let ((a (skip-and-read-object char stream)))
           (if (eof-object? a)
               a
               (let ((d (skip-and-read-list (peek-char stream) stream)))
                 (if (eof-object? d)
                     d
                     (cons a d) ) ) ) ) ) ) )

    ;; Check if what was read is a regular end of list ie is a list
    ;; with exactly one term otherwise report an error.
    (define too-long-dotted-list
      "Incorrect dotted (too long) list" )
    (define too-short-dotted-list
      "Incorrect dotted (too short) list" )
    (define (read-end-of-list expressions stream)
      (if (pair? expressions)
          (if (pair? (cdr expressions))
              (read-error stream too-long-dotted-list expressions)
              (car expressions) )
          (read-error stream too-short-dotted-list expressions) ) )

    ;; Read a sharp notation
    (define premature-eof-in-sharp
      "Stream ends up while reading a sharp character" )
    (define incorrect-character-name
      "Incorrect name of character" )
    (define invalid-dispatch-character
      "Invalid dispatch character" )
    (define named-characters
      (list (cons (string->symbol "space") 32)
            (cons (string->symbol "SPACE") 32)
            (cons (string->symbol "newline") 10)
            (cons (string->symbol "NEWLINE") 10) ) )
    (define (read-sharp-object char stream)
      (case char
        ((#\\)
         (read-until-delimiter
          (read-then-peek-char stream)
          stream
          '()
          (lambda (chars delimiter)
            (case (length chars)
              ((0) (read-char stream))  ; -> delimiter
              ((1) (car chars))
              (else (let* ((n (length chars))
                           (string (make-string n))
                           (name (do ((i 0 (+ 1 i))
                                      (chars chars (cdr chars)) )
                                     ((= i n) (string->symbol string))
                                   (string-set! string i (car chars)) ))
                           (value (assq name named-characters)) )
                      (if value 
                          (integer->char (cdr value))
                          (read-error 
                           stream incorrect-character-name name ) ) )) ) ) ) )
        ((#\t #\T) (read-char stream) #t)
        ((#\f #\F) (read-char stream) #f)
        ((#\b #\B)
         (read-number (read-then-peek-char stream) stream 2 0) )
        ((#\o #\O)
         (read-number (read-then-peek-char stream) stream 8 0) )
        ((#\d #\D)
         (read-number (read-then-peek-char stream) stream 10 0) )
        ((#\x #\X)
         (read-number (read-then-peek-char stream) stream 16 0) )
        ((#\()
         (let ((items
                (skip-and-read-vector (read-then-peek-char stream) stream) ))
           (apply vector items) ) )
        (else (read-char stream)
              (read-error stream invalid-dispatch-character char) ) ) )

    (define premature-eof-in-vector
      "Stream ends up while reading a vector" )
    (define dot-in-vector
      "Vectors cannot be dotted" )
    (define (skip-and-read-vector char stream)
      (if (eof-object? char)
          (read-error stream premature-eof-in-vector)     
          (case char
            ((#\space #\newline)
             (skip-and-read-vector (read-then-peek-char stream) stream) )
            ((#\;)
             (read-after-end-of-line
              (read-then-peek-char stream) stream skip-and-read-vector ) )
            (else (read-vector char stream)) ) ) )
    (define (read-vector char stream)
      (case char
        ((#\))
         (read-char stream)
         '() )
        ((#\.)
         (read-char stream)             ; consume the dot
         (read-error stream dot-in-vector) )
        (else 
         (let ((item (skip-and-read-object char stream)))
           (cons item (skip-and-read-vector (peek-char stream) 
                                            stream )) ) ) ) )

    ;; Returns a list of characters until a delimiter.
    ;; Useful to read #\space for instance.
    (define (read-until-delimiter char stream chars cont)
      (if (eof-object? char)
          (read-error stream premature-eof-in-sharp)
          (case char
            ((#\( #\) #\; #\" #\space #\newline)
             (cont (reverse! chars) char) )
            (else (read-until-delimiter
                   (read-then-peek-char stream)
                   stream
                   (cons char chars)
                   cont )) ) ) )
           
    ;; Read a number
    (define premature-eof-in-number
      "Stream ends up while reading a number" )
    (define incorrect-figure-in-number
      "Incorrect figure while reading number" )
    (define correct-figures "0123456789ABCDEF")
    (define (convert-figure char base result)
      (if (< result base)
          (if (char-ci=? char (string-ref correct-figures result))
              result
              (convert-figure char base (+ 1 result)) )
          #f ) )
    (define (read-number char stream base result)
      (if (eof-object? char)
          (or result (read-error stream premature-eof-in-number))
          (let ((figure (convert-figure char base 0)))
            (if figure
                (read-number (read-then-peek-char stream) 
                             stream base (+ figure (* base result)) )
                ;; numbers are finished by the first non-figure character 
                ;; This is questionable !?
                result ) ) ) )

    ;; It is not yet known if we are reading a number or a symbol.
    ;; string is the extensible string where are stored characters
    ;; sign is #f or #\+ or #\- if the sign has been read
    ;; number? is #t or #f if at least one figure has been read (useful 
    ;; to read + as a symbol and +0 as zero).
    (define (read-number-or-symbol char stream string sign number?)
      (case char
        ((#\0 #\1 #\2 #\3 #\4 #\5 #\6 #\7 #\8 #\9)
         (increment-string char string)
         (read-number-or-symbol 
          (read-then-peek-char stream) stream string sign #t ) )
        ((#\space #\newline #\( #\) #\;)
         (if number?
             (do ((i (if sign 1 0) (+ 1 i))
                  (char 'wait)
                  (result 0) )
                 ((= i (extensible-string-index string))
                  (case sign
                    ((#\-) (- result))
                    (else result) ) )
               (set! char (string-ref (extensible-string-characters string) i))
               (set! result (+ (convert-figure char 10 0) (* 10 result))) )
             (read-symbol char stream string) ) )
        (else
         (read-symbol char stream string) ) ) )

    ;; The generated functions
    (case kind
      ((read) (skip-and-read-object (peek-char stream) stream))
      ((skip) (skip-until-object (peek-char stream) stream)) ) ) )
