/* Lisp mode for Epsilon.  This editor mode is intended to be used with
   programs written in Common Lisp or Scheme.  It attempts to be compatible
   with Symbolics Zmacs and GNU Emacs.

   (c) Copyright 1990 Carl W. Hoffman.  All rights reserved.

   This file may be freely copied, distributed, or modified for non-commercial
   use provided that this copyright notice is not removed.  For further
   information about other Common Lisp and Scheme utilities, contact the
   following address:

   Carl W. Hoffman, 363 Marlborough Street, Boston, MA 02115, U.S.A.
   Internet: CWH@AI.MIT.EDU    CompuServe: 76416,3365    Fax: 617-262-4284

   This code has been tested with Epsilon version 4.13.

   This file expects that FILLPREF.E will also be loaded so as to enable the
   filling of Lisp comments. */

#include <eel.h>

#define OPEN_PAREN  '('
#define CLOSE_PAREN ')'

buffer char fill_prefix[60];
buffer char comment_start[10];
keytable lisp_tab;

#define NOT_WHITESPACE           "[^ \t\n]"
#define NOT_WHITESPACE_OR_OPEN   "[^ \t\n(]"
#define NOT_WHITESPACE_OR_CLOSE  "[^ \t\n)]"
#define LISP_BREAK               "[^\\][ \t\n\"|()]"

forward_one_sexp()
{
  int start = point;
  int level = 0;

  /* Jump over whitespace and close parentheses.
     Abort if we reach the end of the buffer.
     Leave point on the first non-whitespace-or-close character we see. */

  if (!re_search(1, NOT_WHITESPACE_OR_CLOSE)) {
    say("At end of buffer");
    point = start;
    return 1;
    }
  point = matchstart;

  /* Loop skipping forward over Lisp tokens.  The variable LEVEL keeps
     track of the current nesting level. */

  while (1) {
    switch (curchar()) {

      case OPEN_PAREN:
        point++;
        level++;
        break;

      case CLOSE_PAREN:
        point++;
        if (level > 0) level--;
        break;

      case ';':
        nl_forward();
        goto next_token;

      case '\'':
      case '`':
      case ',':
        point++;
        goto next_token;

      case '#':
        point++;
        switch (curchar()) {

          case OPEN_PAREN:                    /* Scheme and CL #( */
            point++;
            level++;
            break;

          case '\\':                          /* Scheme and CL #\ */
            point++;
            if (curchar() == '\\')            /* Must treat #\\ specially */
              point++;
            else if (re_search(1, LISP_BREAK))
              point--;
            break;

          case 'T':                           /* Scheme #T and #F */
          case 't':
          case 'F':
          case 'f':
            point++;
            break;

          case 'B':                           /* Scheme and CL #B #O #X */
          case 'b':
          case 'O':
          case 'o':
          case 'X':
          case 'x':

          case 'D':                           /* Scheme #D */
          case 'd':

          case 'R':
          case 'r':                           /* CL #R */

          case '\'':                          /* CL #' #. #, #+ #- */
          case '.':
          case ',':
          case '+':
          case '-':
            point++;
            goto next_token;

          case '|':                           /* CL #| ... |# */
            if (!re_search(1, "%|#")) {
              say("Unmatched comment");
              point = start;
              return 0;
              }
            break;

          default:
            break;
          }
        break;

      /* We're inside a string. */

      case '"':

        /* Move over the starting double quote. */
        point++;

        /* Move forward looking for backslash and double quote characters.
           If we see a double quote, we've reached the end of the string.
           If we see a backslash, skip over the immediately following
           character and continue parsing. */        

        while (1) {
          if (!re_search(1, "[\\\"]")) {
            say("Unmatched string");
            point = start;
            return 0;
            }
          if (character(point-1) == '"')
            break;
          /* Skip over character quoted by backslash. */
          point++;
          }

        break;

      /* We're inside a quoted symbol. */

      case '|':

        /* Move over the starting vertical bar. */
        point++;

        /* Move forward looking for backslash and vertical bar characters.
           If we see a vertical bar, we've reached the end of the symbol.
           If we see a backslash, skip over the immediately following
           character and continue parsing. */

        while (1) {
          if (!re_search(1, "[\\|]")) {
            say("Unmatched quoted symbol");
            point = start;
            return 0;
            }
          if (character(point-1) == '|')
            break;
          /* Skip over the character quoted by backslash. */
          point++;
          }

        break;

      /* We're inside an ordinary symbol.  Search for the next two-character
         sequence where the first character is not a backslash and the second
         character is a Lisp token break character. */

      default:
        if (re_search(1, LISP_BREAK))
          point--;
        break;
      }

    if (level == 0) return 1;

    /* Skip over whitespace to find the start of the next token.  Leave point
       on the break character, if we find one, or at the end of the buffer. */

    next_token:
    if (!re_search(1, NOT_WHITESPACE))
      break;
    point = matchstart;
    }

  say("Unmatched parentheses");
  point = start;
  return 0;
  }

/* Bugs:  (1) This recognizes any sequence of the characters ' ` , # as being
   a legal sequence of macro characters.  (2) This doesn't recognize ,@ as
   being a legal sequence of macro characters. */

backward_over_macro_chars()
{
  if (index("+-", character(point-1)) &&
      (character(point-2) == '#'))
    point -= 2;
  else while (1) {
    if (!index("'`,#", character(point-1)))
      break;
    point--;
    }
  }

/* It is impossible to parse backwards correctly in all cases, so
   this tries to win only on the common ones.  To really win all the
   time one must start from the beginning of the buffer and parse forward. */

backward_one_sexp()
{
  int start = point;
  int level = 0;

  /* Jump over whitespace and open parentheses.
     Abort if we reach the beginning of the buffer. */

  if (!re_search(-1, NOT_WHITESPACE_OR_OPEN)) {
    say("At beginning of buffer");
    point = start;
    return 1;
    }

  /* Loop skipping backward over Lisp tokens.  The variable LEVEL keeps
     track of the current nesting level. */

  while (1) {
    switch (curchar()) {

      /* If we see a slashified open paren at the end of a token,
         it is either a character constant or a symbol ending in a
         slashified paren. */

      case CLOSE_PAREN:
        if ((character(point-1) == '\\') && (character(point-2) != '\\')) {
          re_search(-1, LISP_BREAK);
          point++;
          }
        else
          level++;
        break;

      case OPEN_PAREN:
        if ((character(point-1) == '\\') && (character(point-2) != '\\')) {
          re_search(-1, LISP_BREAK);
          point++;
          }
        else {
          if (level > 0) level--;
          backward_over_macro_chars();
          }
        break;

      /* We're inside a string.  Search for the next two-character sequence
         where the first character is not backslash and the second character
         is double quote. */

      case '"':
        re_search(-1, "[^\\]\"");
        point++;
        break;

      /* We're inside a quoted symbol.  Search for the next two-character
         sequence where the first character is not backslash and the second
         character is vertical bar. */

      case '|':
        re_search(-1, "[^\\]%|");
        if (!index("'`,", curchar()))
          point++;
        break;

      default:

        /* We're inside a comment. */

        if ((curchar() == '#') && (character(point-1) == '|')) {
          if (!re_search(-1, "#%|")) {
            say("Unmatched comment");
            point = start;
            return 0;
            }
          }

        /* We're inside an ordinary symbol.  Search for the next two-character
           sequence where the first character is not a backslash and the second
           character is a Lisp token break character. */

        else if (re_search(-1, LISP_BREAK))
          point = matchstart;

        /* We must treat the case of a single Lisp break character at the
           beginnning of the buffer specially, since we won't find it as
           a two-character sequence. */

        else if (index("()|\"#", curchar()))
          point++;
        break;
      }

    if (level == 0) return 1;

    if (!re_search(-1, NOT_WHITESPACE))
      break;
    }

  say("Unmatched parentheses");
  point = start;
  return 0;
  }

command forward_sexp() on lisp_tab[ALT(CTRL('F'))]
{
  if (iter < 0)
    while (iter++ < 0)
      backward_one_sexp();
  else
    while (iter-- > 0)
      forward_one_sexp();
  }

command backward_sexp() on lisp_tab[ALT(CTRL('B'))]
{
  if (iter < 0)
    while (iter++ < 0)
      forward_one_sexp();
  else
    while (iter-- > 0)
      backward_one_sexp();
  }

command kill_sexp() on lisp_tab[ALT(CTRL('K'))]
{
  int start = point;
  forward_sexp();
  do_save_kill(start, point);
  }

command up_sexp() on lisp_tab[ALT(CTRL('U'))]
{
  while (1) {
    if (!re_search(-1, NOT_WHITESPACE))
      break;
    if (curchar() == OPEN_PAREN) {
      backward_over_macro_chars();
      break;
      }
    point++;
    if (!backward_one_sexp())
      break;
    if (current_column() == 0)
      break;
    }
  }

/* What should this command do when the cursor is on an open paren?
   At first I thought it should move forward one character and then
   try to go down a level.  However, this means that C-M-D followed
   by C-M-U doesn't leave you where you began.  So, now I have it
   defined to just go forward one character. */

command down_sexp() on lisp_tab[ALT(CTRL('D'))]
{
  int start = point;
  /* Must treat this as a special case since the re_search
     will only look for two character sequences. */
  if (curchar() == OPEN_PAREN) {
    point++;
    return;
    }
  if (!re_search(1, "[^\\][()]"))
    return;
  if (character(point-1) == CLOSE_PAREN) {
    point = start;
    }
  }

char *trans_temp;		/* name of temporary buffer */

command transpose_sexp() on lisp_tab[ALT(CTRL('T'))]
{
  int first, second;
  trans_temp = temp_buf();
  backward_one_sexp();
  first = point;
  forward_one_sexp();
  save_away(first, point);
  forward_one_sexp();
  second = point;
  grab_back();
  point = second;
  backward_one_sexp();
  save_away(point, second);
  point = first;
  grab_back();
  delete_buffer(trans_temp);
  iter = 0;
  }

command kill_sexp_up() on lisp_tab[ALT('K')]
{
  int first;
  trans_temp = temp_buf();
  forward_one_sexp();
  first = point;
  backward_one_sexp();
  save_away(point, first);
  up_sexp();
  first = point;
  forward_one_sexp();
  delete(first, point);
  grab_back();
  backward_one_sexp();
  delete_buffer(trans_temp);
  iter = 0;
  }

command begin_defun() on lisp_tab[ALT(CTRL('A'))]
{
  while (1) {
    if (!search(-1, "("))
      break;
    if (current_column() == 0)
      break;
    }
  }

command reposition_defun() on lisp_tab[ALT(CTRL('R'))]
{
  int try_window, try_defun;
  int save = point;
  try_window = prev_screen_line(window_size - 1);
  if (current_column() != 0 || curchar() != '(')
    begin_defun();
  try_defun = prev_screen_line(0);
  window_start = (try_defun > try_window) ? try_defun : try_window;
  point = save;
  screen_messed();
  build_first = 1;
  }

/* This information should be placed in some sort of database.  For now,
   the user can redefine this function to customize the indenter. */

int compute_operator_indent (operator, start_column, siblings)
char *operator;
int start_column, siblings;
{

  if (!(   strfcmp(operator, "do")
        && strfcmp(operator, "do*")
        && strnfcmp(operator, "defun", 5)
        && strnfcmp(operator, "defmacro", 8)))
    if (siblings != 2)
      return start_column + 2;
    else
      return current_column();

  if (!(   strnfcmp(operator, "def", 3)
        && strnfcmp(operator, "let", 3)
        && strnfcmp(operator, "with", 4)
        &&  strfcmp(operator, "case")
        &&  strfcmp(operator, "flet")
        &&  strfcmp(operator, "when")
        &&  strfcmp(operator, "catch")
        &&  strfcmp(operator, "ccase")
        &&  strfcmp(operator, "ecase")
        &&  strfcmp(operator, "labels")
        &&  strfcmp(operator, "lambda")
        &&  strfcmp(operator, "unless")
        &&  strfcmp(operator, "dolist")
        &&  strfcmp(operator, "dotimes")
        &&  strfcmp(operator, "macrolet")
        &&  strfcmp(operator, "ctypecase")
        &&  strfcmp(operator, "etypecase")
        &&  strfcmp(operator, "destructuring-bind")
        &&  strfcmp(operator, "multiple-value-bind")
        ))
    return start_column + 2;
  else
    return current_column();
  }

lisp_compute_indent()
{
  int indent = -1;
  int start = point;
  int siblings = 0;
  char operator[80];
  int operator_start, operator_end;
  int open_paren, open_paren_column;

  /* Find first non-whitespace character above the current line. */

  to_begin_line();
  if (!re_search(-1, NOT_WHITESPACE)) {
    point = start;
    to_indentation();
    indent = current_column();
    point = start;
    return indent;
    }

  /* Skip backward over the preceding S-expressions until we find the operator
     of the current form.  If the operator of the current form is a list
     rather than a symbol, then move up a level and try again. */

  /* Can we rewrite this to use up_sexp?  We would have to make up_sexp
     (or an internal subroutine) return a number which was the current
     sibling number. */

  if (curchar() != OPEN_PAREN)
    while (1) {
      point++;
      backward_one_sexp();

      /* If an S-expression begins in column 0, then give up and indent
         from column 0.  This also handles the case of comments beginning
         in column 0 since we can't recognize them when parsing backwards. */

      if (current_column() == 0) {
        point = start;
        return 0;
        }

      siblings++;
      if (indent == -1)
        indent = current_column();

      /* If no more non-whitespace characters before point, then the current
         form must be toplevel.  Just indent below the previous form. */
      if (!re_search(-1, NOT_WHITESPACE)) {
        point = start;
        return indent;
        }
      if (curchar() == OPEN_PAREN)
        break;
      }

  /* The point is just before our parent's open paren.  Find the indentation
     of the first S-expression following the operator.  Also check the
     operator name for certain special forms. */

  open_paren = point;
  open_paren_column = current_column();
  point++;
  re_search(1, "[ \t]*");

  /* If there is whitespace after our parent's open paren, then
     it is probably some sort of constant list structure.  Don't try
     to interpret it in any special way.  If there is an open paren
     after our parent's open paren, then just align with it. */

  if ((point > open_paren+1) || (curchar() == '(')) {
    indent = current_column();
    point = start;
    return indent;
    }

  /* Before looking at the current operator, go up two levels and
     check for FLET, LABELS, and MACROLET.  If we find one,
     then indent the current line two spaces in, as if it were a DEFUN.
     After going up one level, we check for two consecutive open
     parentheses.  This is a kludgey way of seeing if we are in the
     first subform of a FLET, LABELS, or MACROLET rather than in
     the second one.  A better way of doing this is to have up_sexp
     tell us how many expressions it had to go backward before reaching
     the car of the form. */

  point--;
  up_sexp();
  if (character(point+1) == '(') {
    up_sexp();
    point++;
    operator_start = point;
    re_search(1, LISP_BREAK);
    operator_end = point-1;
    grab(operator_start, operator_end, operator);
    if (!(   strfcmp(operator, "flet")
          && strfcmp(operator, "labels")
          && strfcmp(operator, "macrolet"))) {
      point = start;
      return open_paren_column + 2;
      }
    }

  /* Find the start and the end of the operator symbol. */

  point = open_paren + 1;
  operator_start = point;
  re_search(1, LISP_BREAK);
  point--;
  operator_end = point;

  /* Skip over any whitespace following the operator. */

  re_search(1, "[ \t]*");

  /* If whitespace and a newline immediately follow the operator, then just
     indent two spaces from it. */

  if (curchar() == '\n') {
    indent = open_paren_column + 2;
    point = start;
    return indent;
    }

  /* Point is now after the whitespace following the operator of the
     parent form. */

  grab(operator_start, operator_end, operator);
  indent = compute_operator_indent(operator, open_paren_column, siblings);
  point = start;
  return indent;
  }

lisp_indent() on lisp_tab['\t']
{
  int start = point;
  int offset = 0;
  to_indentation();
  if (point < start) offset = start - point;
  to_column(lisp_compute_indent());
  point += offset;
  }

command indent_sexp() on lisp_tab[ALT(CTRL('Q'))]
{
  int start = point;
  int *end = alloc_spot();
  forward_one_sexp();
  *end = point;
  point = start;
  while (1) {
    if (!nl_forward())
      break;
    if (point >= *end)
      break;
    if (!re_search(1, "[ \t]*"))
      break;
    if (curchar() == '\n')
      delete_horizontal_space();
    else {
      to_begin_line();
      to_column(lisp_compute_indent());
      }
    }
  point = start;
  }

/* The command show_matching_delimiter tests to see if move_level
   returns 1, and only then does a show_line.  Should we do the same? */

command show_matching_paren() on lisp_tab[CLOSE_PAREN]
{
  int start;
  normal_character();
  start = point;
  say("");
  backward_one_sexp();
  show_line();
  point = start;
  }

command insert_parens() on lisp_tab[ALT(OPEN_PAREN)]
{
  stuff("()");
  point--;
  }

command move_over_close_paren() on lisp_tab[ALT(CLOSE_PAREN)]
{
  re_search(1, ")");
  }

lisp_indenter() { to_column(lisp_compute_indent()); }

command lisp_mode ()
{
  mode_keys = lisp_tab;
  indenter = lisp_indenter;
  auto_indent = 1;
  margin_right = 79;
  fill_mode = 0;
  strcpy(fill_prefix, ";; ");
  strcpy(comment_start, "; ");
  major_mode = "Lisp";
  make_mode();
  }

suffix_lsp()  { lisp_mode(); }
suffix_scm()  { lisp_mode(); }

/* Tag all Lisp functions in this file */

/*
tag_suffix_lsp()
{
  }

tag_suffix_scm() { tag_suffix_lsp(); }

*/
