;;; receptionist.scm -- an object type for event mgt and dispatching.
;;; Brian Beckman, 13 Aug 1989

;;----------------------------------------------------------------------;;
;;                                                                      ;;
;;                        R E C E P T I O N I S T                       ;;
;;                                                                      ;;
;;----------------------------------------------------------------------;;

; The receptionist controls access to the event service resources of
; the application.  The receptionist also disciplines access to the
; system's event queue, hiding system-dependent details.  The
; receptionist keeps a data base of workers.  Each workers
; is a triple:
; 
;     (device-number event-fender event-handler), abbreviated to
;     (dn ef eh)
; 
; An event-handler is called by the receptionist when the device
; denoted by device-number generates a low-level event and when the
; event-fender returns true (#t).
; 
; Device-number is a system-dependent device number, denoting a
; physical or logical device such as a mouse button or keyboard.
; Device number 0 is reserved for null events.  Handlers for null
; events are ``background processes''.  They are called only when the
; low-level event queue is empty.  A low-level event is a pair
; 
;     (device-number device-data), abbreviated to
;     (dn dd)
; 
; Event-fender is a boolean function of one argument returning true
; when the corresponding event-handler is to be called.  The argument
; is the low-level event found by the receptionist.  Event-fenders are
; used to refine the definition of an event, for example by checking
; whether a mouse hit was in a certain on-screen object such as a
; button or slider.  A low-level event plus an event-fender together
; define a high-level, or user-defined event.
; 
; Event-handler is a function of one argument -- a low-level event --
; that processes the detected, or ``fended'' event.  The event-handler 
; performs whatever application-dependent actions are to be associated
; with a high-level event.  Given a low-level event, the receptionist
; will check all event services with a matching device-number, and
; will call all event-handlers whose fenders evaluate to true for that
; low-level event.
; 
; The event service data base is organized as a binary search table.
; 
;     #( (dn (ef eh) ...)  ...)

(set! **standard-breakout-continuation** '())

(define (new-receptionist . initial-workers)
  
  (let* ( (maxsize 16)
          (size 1)
          (tbl (make-vector maxsize)) 
          (dummy (vector-set! tbl 0 '(0)))
          (supers '()) )
    
    ; Global Invariant: element 0 of tbl always contains the
    ; workers for the null event.  That is, it always contains the
    ; list (0 (ef eh) ...).  

    ;;; local binary search invariants:
    ;;; min >= 0, max < size,
    ;;; min <= mid <= max.
    ;;; OK by the global invariant 

    (define (index-search key)
      (let iter ( (min 0) (max (-1+ size)) )
        (let* ( (mid  (quotient (+ min max) 2))
                (v    (vector-ref tbl mid))
                (test (car v)) )
          (cond
           ( (= key test)  mid                  )
           ( (>= min max)  '()                  )
           ( (< key test)  (iter min (-1+ mid)) )
           ( else          (iter (1+ mid) max)  )))))

    (define (search key)
      (let iter ( (min 0) (max (-1+ size)) )
        (let* ( (mid  (quotient (+ min max) 2))
                (v    (vector-ref tbl mid))
                (test (car v)) )
          (cond
           ( (= key test)  v                    )
           ( (>= min max)  '()                  )
           ( (< key test)  (iter min (-1+ mid)) )
           ( else          (iter (1+ mid) max)  )))))

 ;   (define search
 ;     (let ( (last-hit -1) )
 ;       (lambda (key)
 ;         (if (= key last-hit)
 ;             (vector-ref tbl last-hit)
 ;             (let iter ( (min 0) (max (- size 1)) )
 ;               (let* ( (mid  (quotient (+ min max) 2))
 ;                       (v    (vector-ref tbl mid))
 ;                       (test (car v)) )
 ;                 (cond
 ;                  ( (= key test)  (set! last-hit (car v)) v  )
 ;                  ( (>= min max)  '()                        )
 ;                  ( (< key test)  (iter min (-1+ mid))       )
 ;                  ( else          (iter (1+ mid) max) ))))))))

    (define (swap i j)
      (let ( (t (vector-ref tbl i)) )
        (vector-set! tbl i (vector-ref tbl j))
        (vector-set! tbl j t)))
    
    (define (check) ;;; check the invariant
      (unless (= (car (vector-ref tbl 0)) 0)
              (error "Receptionist must always have a null worker")))
    
    (define (sift-in worker)
      ;;; precondition -- maxsize > size
      (vector-set! tbl size (list (car worker) (cdr worker)))
      (let iter ( (i (-1+ size)) )
        (cond
         ( (= i 0)  (check) )
         ( (< (car (vector-ref tbl (1+ i)))
              (car (vector-ref tbl i)))
           (swap i (1+ i)) (iter (-1+ i)) )
         ( else (check) )))
      (set! size (1+ size)))

    (define (rif-department dev)
      (let ( (lookup (search dev)) )
        ;(pr lookup (index-search dev) (- size 2))
        (when lookup
          (for (i from (index-search dev) thru (- size 2))
            (vector-set! tbl i (vector-ref tbl (add1 i))))
          (set! size (sub1 size))
          (vector-set! tbl size '())
        )
      )
      (check)
      self
    )

    (define (fire worker)
      (let ( (lookup (search (car worker))) )
        (when lookup
          (set-cdr! lookup 
            (let ( (lyst (cdr lookup))
                   (obj  (cdr worker)) )
              (let iter ( (l lyst) )
                (cond
                  ( (null? l) '() )
                  ( (equal? obj (car lyst)) (iter (cdr lyst)) )
                  ( else (cons (car lyst) (iter (cdr lyst))) )))))))
      self)

    (define (install worker)
      (let ( (lookup (search (car worker))) )
        (cond

         ( lookup 
           (vector-set! 
            tbl 
            (index-search (car worker))
            (cons (car lookup) (cons (cdr worker) (cdr lookup)))) )

         ( (= size maxsize)
           (set! maxsize (* maxsize 2))
           (let ( (temp (make-vector maxsize)) )
             (for (i from 0 to size)
                  (vector-set! temp i (vector-ref tbl i)))
             (set! tbl temp))
           (sift-in worker) )

         ( else (sift-in worker) )))

      (qdevice (car worker))
      self)
    
    (define (show sym val)
      (display sym) (display " = ") (display val) (newline))
    
    (define (print)
      (show 'size size)
      (show 'maxsize maxsize)
      (display "#(") (newline)
      (for (i from 0 to size)
           (display " ")
           (display (vector-ref tbl i))
           (newline))
      (display ")")
      (newline))

    (define (iter-worker w data)
      (cond
        ( (null? w) )
        ( ((caar w) data) ((cadar w) data)
            (iter-worker (cdr w) data) )
        ( else
            (iter-worker (cdr w) data) ) ))

    (define (loop breakout-continuation)
      (qreset)
      (set! **standard-breakout-continuation** breakout-continuation)
      (while #t
        (let ( (esr (search (qtest))) )
          ;(display esr) (newline)  
          (cond
            ( (null? esr) (qread) ) ;;; trash the event
            ( (= 0 (car esr)) (iter-worker (cdr esr) '(0 0)) )
            ( else (iter-worker (cdr esr) (qread)) )))))

    (define (enter-loop)
      (call-with-current-continuation loop))

    (define (self msg)
      (cond
       ( (eq? msg 'install)      install         )
       ( (eq? msg 'retrieve)     search          )
       ( (eq? msg 'rif-dev)      rif-department  )
       ( (eq? msg 'rif)          fire            )
       ( (eq? msg 'print)        print           )
       ( (eq? msg 'table)        (lambda () tbl) )
       ( (eq? msg 'enter-loop)   enter-loop      )
       ( (search-supertypes supers msg)          )
       ( else  (make-error-method "Receptionist" msg) )))

    (for-each install initial-workers)

    self))
    
;;----------------------------------------------------------------------;;
;;                     g e n e r i c   f e n d e r s                    ;;
;;----------------------------------------------------------------------;;

(define (always data) #t)
(define (never data)  #f)
(define (button-down data) (= (cadr data) 1))
(define (button-up   data) (= (cadr data) 0))

;;----------------------------------------------------------------------;;
;;             g e n e r i c   Q u i t t e r   h a n d l e r            ;;
;;----------------------------------------------------------------------;;

(define (standard-quitter data)
   (**standard-breakout-continuation** #t))

(define standard-quit-worker (list ESCKEY always standard-quitter))

;;----------------------------------------------------------------------;;
;;              H a r d   l o o p i n g   c o n s t r u c t             ;;
;;----------------------------------------------------------------------;;

(define (do-until-event data thunk)

  (let ( (dev (car data))
         (val (cadr data))
       )

    (qdevice dev)

    (let iter ( (q (qtest)) )

      (cond

        ( (= q 0) (thunk) (iter (qtest)) ) ;; nothing in queue

        ( else (let ( (qr (qread)) )    ;; Read queue contents

                 (cond 

                   ( (and (= q dev) (= val (cadr qr))) 'done )

                   ( else (thunk) (iter (qtest)) ))) )))))


