;************************************************************
;                                                           *
; Copyright (c) 1990, California Institute of Technology.   *
; U.S. Government Sponsorhip under NASA Contract NAS7-918   *
; is acknowledged.                                          *
;                                                           *
;***********************************************************/

;;----------------------------------------------------------------------;;
;;                                                                      ;;
;;                        M A T R I C E S . S C M                       ;;
;;                                                                      ;;
;;----------------------------------------------------------------------;;
;;; Section 3.4,  page 242 ff of Abelson & Sussman
;;; Brian Beckman, 24 July 1989
;;;                 5 Aug  1989 -- polishing
;;----------------------------------------------------------------------;;

;;; CONTENTS

;;; (accum op init vec)
;;; (flatten stream)
;;; (filter-stream predicate? stream)
;;; (accum-n op init streams)
;;; (%m m)
;;; (*vv v w)
;;; (*mv m v)
;;; (*vm v m)
;;; (*mn m n)
;;; (*vmv v m)
;;; (@vv v w)
;;; (*vs v s)
;;; (*ms m s)
;;; (skip-elt stream k)
;;; (cofactor m i j)
;;; (alt vec)
;;; (det m)
;;; (inverse m)
;;; (+vv v w)
;;; (-vw v w)
;;; (+mm m n)
;;; (-mn m n)
;;; (zero dim)
;;; (vconst c dim)
;;; (e k dim)
;;; (ident dim)
;;; seed
;;; (vrand max-elt dim)
;;; (mrand max-elt dim)
;;; (vclean v epsilon)
;;; (mclean m epsilon)
;;; (printm m)

;;; For small matrices, we implement streams as ordinary lists.  For
;;; large matrices, use kstreams.  Some kstream operators are macros,
;;; and cannot be apply-ed or map-ped (e.g. kons), so some matrix
;;; operators in this file have no analogous forms in kstream represen-
;;; tation.

;;----------------------------------------------------------------------;;
;;                    s t r e a m   o p e r a t o r s                   ;;
;;----------------------------------------------------------------------;;
(define (accum op init vec)
  (cond
   (  (null? vec)  init  )
   (  t  (op (car vec) (accum op init (cdr vec)))  )))

(define (flatten stream)
  (accum append '() stream))

(define (filter-stream predicate? stream)
  (cond
   (  (null? stream)  '()  )
   (  (predicate? (car stream))
        (cons
         (car stream)
         (filter-stream predicate? (cdr stream)))  )
   (  t  (filter-stream predicate? (cdr stream))  )))

(define (accum-n op init streams)    ;; CROSS-SECTION ACROSS
  (cond                              ;; A STREAM OF STREAMS
   (  (null? (car streams))  '()  )
   (  t  (cons
          (accum   op init (map car streams))
          (accum-n op init (map cdr streams)))  )))

;;; page 252, Ex 3.40

;;; an arbitrary-dimensional matrix library.
;;; There is no error checking in this first draft.

;;----------------------------------------------------------------------;;
;;                           t r a n s p o s e                          ;;
;;----------------------------------------------------------------------;;
(define (%m m) (accum-n cons '() m))

;;----------------------------------------------------------------------;;
;;                      i n n e r   p r o d u c t s                     ;;
;;----------------------------------------------------------------------;;
(define (*vv v w) (accum + 0 (accum-n * 1 (list v w))))

(define (*mv m v) (map (lambda (row) (*vv row      v))     m))

(define (*vm v m) (map (lambda (row) (*vv row      v)) (%m m)))

(define (*mn m n) (map (lambda (vec) (*mv (%m n) vec))     m))

(define (*mn-f m n)
  (let (( n (%m n) ))
    (map (lambda (vec) (*mv n vec)) m)))
;;; NO NOTICEABLE PERFORMANCE IMPROVEMENT

(define (*vmv v m) (*vv (*vm v m) v))

;;----------------------------------------------------------------------;;
;;                       o u t e r   p r o d u c t                      ;;
;;----------------------------------------------------------------------;;
(define (@vv v w) (*mn (map list v) (list w)))

;;----------------------------------------------------------------------;;
;;                             s c a l i n g                            ;;
;;----------------------------------------------------------------------;;
(define (*vs v s) (map (lambda (x) (*   x s)) v))

(define (*ms m s) (map (lambda (v) (*vs v s)) m))

;;----------------------------------------------------------------------;;
;;        determinant by cofactor analysis (naturally recursive)        ;;
;;----------------------------------------------------------------------;;
;;; This procedure is unpleasantly slow by dim = 6.

(define (skip-elt stream k)
  (define (iter s count)
    (cond
     ( (null? s)  '() )
     ( (= k count) (cdr s) )
     ( else (cons (car s) (iter (cdr s) (1+ count))))))
  (iter stream 1))

(define (cofactor m i j)
  (define (iter k m)
    (cond
     ( (null? m)  '() )
     ( (= k i)    (iter (1+ k) (cdr m)) )
     ( else       (cons (skip-elt (car m) j)
                        (iter (1+ k) (cdr m))) )))
  (iter 1 m))

(define (sign i)
    (if (= 0 (remainder i 2)) 1 -1))

(define (det m)  
  (define (iter s j)
    (cond
     ( (null? s)  0 )
     ( else  (+ (* (car s) (det (cofactor m 1 j)) (sign (1+ j)))
                (iter (cdr s) (1+ j))) )))
  (cond
   ( (null? m)  1 )
   ( else (iter (car m) 1) )))

;;----------------------------------------------------------------------;;
;;                             i n v e r s e                            ;;
;;----------------------------------------------------------------------;;
(define (inverse m)
  (let (( d (det m) ))
    (define (do-row i j r)
      (cond
       ( (null? r)  '() )
       ( else (cons (/ (det (cofactor m i j)) (* (sign (+ i j)) d))
                    (do-row i (1+ j) (cdr r))) )))
    (define (iter i block)
      (cond
       ( (null? block) '() )
       ( else (cons (do-row i 1 (car block))
                    (iter (1+ i) (cdr block))) )))
    (%m (iter 1 m))))

;;----------------------------------------------------------------------;;
;;               v e c t o r   a d d   &   s u b t r a c t              ;;
;;----------------------------------------------------------------------;;
(define (+vv v w) (accum-n + 0 (list v w)))

(define (-vw v w) (accum-n - 0 (list v w)))

;;----------------------------------------------------------------------;;
;;               m a t r i x   a d d   &   s u b t r a c t              ;;
;;----------------------------------------------------------------------;;
(define (+mm m n) (accum-n +vv (zero (length (car m))) (list m n)))

(define (-mn m n) (accum-n -vw (zero (length (car m))) (list m n)))

;;----------------------------------------------------------------------;;
;;                        z e r o   v e c t o r s                       ;;
;;----------------------------------------------------------------------;;
(define (zero dim)
  (define (iter i)
    (cond
     ( (= i dim)  '() )
     ( else  (cons 0 (iter (1+ i))) )))
  (iter 0))

;;----------------------------------------------------------------------;;
;;                    c o n s t a n t   v e c t o r                     ;;
;;----------------------------------------------------------------------;;
(define (vconst c dim)
  (define (iter i)
    (cond
     ( (= i dim)  '() )
     ( else  (cons c (iter (1+ i))) )))
  (iter 0))

;;----------------------------------------------------------------------;;
;;                       b a s i s   v e c t o r s                      ;;
;;----------------------------------------------------------------------;;
(define (e k dim)
  (define (iter i)
    (cond
     ( (= i dim)  '() )
     ( (= i k)  (cons 1 (iter (1+ i))) )
     ( else     (cons 0 (iter (1+ i))) )))
  (iter 0))

;;----------------------------------------------------------------------;;
;;                     i d e n t i t y   m a t r i x                    ;;
;;----------------------------------------------------------------------;;
(define (ident dim)
  (define (iter i)
    (cond
     ( (= i dim)  '() )
     ( else  (cons (e i dim) (iter (1+ i))) )))
  (iter 0))

;;----------------------------------------------------------------------;;
;;                       r a n d o m   v e c t o r                      ;;
;;----------------------------------------------------------------------;;
(define seed
  (letrec (( iter (lambda (i max)
                    (cond
                     ( (>= i max)  (random 256) )
                     ( else  (random 256) (iter (1+ i) max) ))) ))
    (delay (iter 0 (remainder (time) 256)))))

(define (vrand max dim)
  (define (iter i)
    (cond
     ( (>= i dim)  '() )
     ( else  (cons (random max) (iter (1+ i))) )))
  (force seed)
  (iter 0))

;;----------------------------------------------------------------------;;
;;                       r a n d o m   m a t r i x                      ;;
;;----------------------------------------------------------------------;;
(define (mrand max dim)
  (define (iter i)
    (cond
     ( (>= i dim)  '() )
     ( else  (cons (vrand max dim) (iter (1+ i))) )))
  (iter 0))

;;----------------------------------------------------------------------;;
;;            f l o a t i n g   p o i n t   c l e a n u p               ;;
;;----------------------------------------------------------------------;;
(define (vclean v epsilon)
  (map (lambda (x) (if (< (abs x) epsilon) 0 x)) v))

(define (mclean m epsilon)
  (map (lambda (v) (vclean v epsilon)) m))

;;----------------------------------------------------------------------;;
;;                     p r i n t   r o u t i n e                        ;;
;;----------------------------------------------------------------------;;
(define (printm m)
  (display "(")
  (letrec (( iter
             (lambda (y)
               (cond
                ( (null? y) (display ")") (newline) )
                ( else  (display (car y)) (newline) 
                        (display " ") (iter (cdr y)) ))) ))
    (iter m)))
