/*
 *
 *  g f . c			-- STklos support (generic functions)
 *
 * Copyright (C) 1993 Erick Gallesio - I3S - CNRS / UNSA <eg@unice.fr>
 * 
 *
 * Permission to use, copy, and/or distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that both the above copyright notice and this permission notice appear in
 * all copies and derived works.  Fees for distribution or use of this
 * software or derived works may only be charged with express written
 * permission of the copyright holder.  
 * This software is provided ``as is'' without express or implied warranty.
 *
 * This software is a derivative work of other copyrighted softwares; the
 * copyright notices of these softwares are placed in the file COPYRIGHTS
 *
 *
 *            Author: Erick Gallesio [eg@unice.fr]
 *    Creation date:  9-Feb-1994 15:56
 * Last file update:  7-Apr-1994 23:20 
 */

#ifdef USE_STKLOS 

#include <stk.h>
#include "stklos.h"
#include "gf.h"

/* 
 * This file is a kind of "sub-module" of STklos implementation.
 * It implement what is necessary to the Clos-like generic functions
 * This implementation provides
 *	- generic functions (with class specializers)
 *	- multi-methods
 *	- next-method 
 *	- a simple MOP
 *
 * Next-methods use a new type of scheme object. This object memoizes the
 * original arguments list and the list of sorted next-methods. This is,
 * of course, a applyable object.
 *
 */

#define SPEC_OF(x)		THE_SLOT_OF(x, S_specializers)

static SCM apply_method(SCM m, SCM args, SCM other_methods);


/******************************************************************************
  *
  * next_method type definition 
  *
  ******************************************************************************/

static int tc_next_method;	/* The type field of a next method */

static SCM make_next_method(SCM methods, SCM args)
{
  register SCM z;

  NEWCELL(z, tc_next_method);
  CAR(z) = methods;
  CDR(z) = args;
  return z;
}

static void mark_next_method(SCM next)
{
  gc_mark(CAR(next));
  gc_mark(CDR(next));
}


static SCM apply_next_method(SCM next, SCM provided_args, SCM env)
{
  SCM methods = CAR(next);
  SCM args    = NULLP(provided_args)? CDR(next) : provided_args;

  if (NULLP(methods)) err("apply: no next method", NIL);
  return apply_method(CAR(methods), args, CDR(methods));
}

static extended_scheme_type next_method_type = {
  "next-method",	/* name */
  EXT_EVALPARAM,	/* flags */
  mark_next_method,	/* gc_mark_fct */
  NULL,			/* gc_sweep_fct */
  apply_next_method,	/* apply_fct */
  NULL			/* display_fct */
};


/******************************************************************************
 * 
 * Protocol for calling a generic fumction
 *
 * 	+ apply-generic (gf args env)
 *		+ compute-applicable-methods (gf args)
 *		+ apply-methods (methods args)
 *			+ apply-method (method args next-methods)
 *				
 * apply-method calls make-next-method to build the "continuation" of a a method
 * Calling a next-method will call apply-next-method which in turn will call 
 * apply-method again to call effectively the following method.
 *
 ******************************************************************************/

static SCM applicablep(SCM actual, SCM formal)
{
  return memq(formal, THE_SLOT_OF(actual, S_cpl));
}

static int more_specificp(SCM m1, SCM m2, SCM targs)
{
  register SCM s1, s2, a;

  /* 
   * Note: 
   *   m1 and m2 can have != length (i.e. one can be one element longer than the 
   * other when we have a dotted parameter list). For instance, with the call
   *   (M 1)
   * with
   *   (define-method M (a . l) ....)
   *   (define-method M (a) ....) 
   *
   * we consider that the second method is more specific.
   *
   */
  for (a=targs,s1=SPEC_OF(m1),s2=SPEC_OF(m2); ; s1=CDR(s1),s2=CDR(s2),a=CDR(a)) {
    if (NULLP(s1)) return 1;
    if (NULLP(s2)) return 0;
    if (CAR(s1) != CAR(s2)) {
      register SCM l, cs1 = CAR(s1), cs2 = CAR(s2);
      
      for (l = THE_SLOT_OF(CAR(a), S_cpl);   ; l = CDR(l)) {
	if (EQ(cs1, CAR(l))) return 1;
	if (EQ(cs2, CAR(l))) return 0;
      }
      return 0;/* should not occur! */
    }
  }
  return 0; /* should not occur! */
}


static SCM sort_applicable_methods(SCM method_list, SCM targs)
{
  SCM *v, vector;
  int i, j, incr, size;

  vector = lvector(method_list);
  size   = VECTSIZE(vector);
  v      = VECT(vector);

  /* Use a simple shell sort since it is generally faster than qsort on 
   * small vectors (which is probably mostly the case when we have to
   * sort a list of applicable methods).
   */

  for (incr = size / 2; incr; incr /= 2) {
    for (i = incr; i < size; i++) {
      for (j = i-incr ;j >= 0; j -= incr) {
	if (more_specificp(v[j], v[j+incr], targs)) break;
	else {
	  SCM tmp   = v[j+incr];
	  v[j+incr] = v[j];
	  v[j]	    = tmp;
	}
      }
    }
  }
  return vector2list(vector);
}

static SCM compute_applicable_methods(SCM gf, SCM args)
{
  SCM l, al, fl, applicable = NIL;
  SCM arg_types = NIL;
  SCM save = args;

  /* Build the list of arguments types */
  for ( ; NNULLP(args); args = CDR(args)) {
    arg_types = cons(class_of(CAR(args)), arg_types);
  }
  arg_types = reverse(arg_types);

  /* Build a list of all applicable methods */
  for (l= THE_SLOT_OF(gf, S_methods); NNULLP(l); l = CDR(l)) {

    for (al=arg_types, fl=SPEC_OF(CAR(l));    ;  al=CDR(al), fl=CDR(fl)) {
      if (INSTANCEP(fl) || 	  	/* We have a dotted argument list */
	  NULLP(al) && NULLP(fl)) {	/* both list exhausted */
	applicable = cons(CAR(l), applicable);
	break;
      }
      if (NULLP(al) || NULLP(fl) || applicablep(CAR(al), CAR(fl))==ntruth) break;
    }
  }

  if (NULLP(applicable)) {
    apply_generic(VCELL(intern("no-applicable-method")), LIST2(gf, save), NIL);
    /* if we are here, it's because no-applicable-method hasn't signaled an error */
    return NIL;
  }
  return (NULLP(CDR(applicable))) ? applicable :
    				    sort_applicable_methods(applicable, arg_types);
}

static SCM apply_method(SCM m, SCM args, SCM next_methods)
{
  return apply(THE_SLOT_OF(m, S_procedure),
	       cons(make_next_method(next_methods, args), args));
}

static SCM apply_methods(SCM methods, SCM args)
{
  if (NULLP(methods)) {
    /* 
     * methods can be NIL if we have a no-applicable-method handler which 
     * doesn't signal an error (or dont ends with a call to next-method)
     * In this case return an undefined value
     */
    return UNDEFINED;
  }

  return apply_method(CAR(methods), args, CDR(methods));
}

SCM apply_generic(SCM gf, SCM args, SCM env)
{
  if (NGENERICP(gf)) err("apply: bad generic function", gf);
  if (NULLP(THE_SLOT_OF(gf, S_methods))) 
    err("apply: no methods for generic", gf);
  
  return apply_methods(compute_applicable_methods(gf, args), args);
}


/******************************************************************************
  *
  * add-method
  *
  *******************************************************************************/

static SCM compute_new_list_of_methods(SCM gf, SCM new)
{
  SCM l1, l2, l;
  SCM new_spec = SPEC_OF(new);
  SCM methods  = THE_SLOT_OF(gf, S_methods);

  for (l = methods; NNULLP(l); l = CDR(l)) {
    for (l1=new_spec, l2=SPEC_OF(CAR(l));    ; l1=CDR(l1), l2=CDR(l2)) {
      if (NULLP(l1) && NULLP(l2)) {
	/* The spec. list of new method already exists in the gf mehods list */	
	CAR(l) = new;
	return methods;
      }
      if (NULLP(l1) || NULLP(l2) || NEQ(CAR(l1), CAR(l2))) break;
    }
  }
Out:
  /* If we are here, we have not encountered a method with same specializers */
  return cons(new, methods);
}


static PRIMITIVE add_method(SCM gf, SCM method)
{
  if (NGENERICP(gf))    err("add-method: bad generic function", gf); 
  if (NMETHODP(method)) err("add-method: bad method", method);
  
  THE_SLOT_OF(gf, S_methods) = compute_new_list_of_methods(gf, method); 
  return method;
}

/******************************************************************************
 *
 * A simple make (which will be redefined later in Scheme)
 * This version handles only creation of gf, methods and classes (no instances)
 *
 * Since this code will disappear when Stklos will be fully booted, 
 * no precaution is taken to be efficient.
 *
 ******************************************************************************/

static PRIMITIVE lmake(SCM args)
{
  SCM class, z;

  if (NULLP(args)) err("make: parameter list is null", NIL);

  class = CAR(args); args  = CDR(args); 
  
  if (class == VCELL(intern("<generic>"))) {
    z = make_instance(class, llength(THE_SLOT_OF(class, S_slots)), TYPE_GENERIC);

    THE_SLOT_OF(z, S_name)    = get_keyword(makekey(":name"), args, intern("???"));
    THE_SLOT_OF(z, S_methods) = NIL;
  }
  else {
    z = make_instance(class, llength(THE_SLOT_OF(class, S_slots)), TYPE_INSTANCE);

    if (class == VCELL(intern("<method>"))) {
      THE_SLOT_OF(z, S_name) =  
			get_keyword(makekey(":name"), args, intern("???"));
      THE_SLOT_OF(z, S_specializers) =  
			get_keyword(makekey(":specializers"), args, NIL);
      THE_SLOT_OF(z, S_procedure) =
			get_keyword(makekey(":procedure"), args, NIL);
    }
    else {
      /* In all the others case, make a new class .... No instance here */
      THE_SLOT_OF(z, S_name) = 
			get_keyword(makekey(":name"), args, intern("???"));
      THE_SLOT_OF(z, S_direct_supers) = 
			get_keyword(makekey(":dsupers"), args, NIL);
      THE_SLOT_OF(z, S_direct_slots)  = 
			get_keyword(makekey(":slots"), args, NIL);
    }
  }
  return z;
}


/******************************************************************************
 *
 * Initializations 
 *
 ******************************************************************************/

static void create_classes_for_gf(void)
{
  SCM proc, ent, meth, gen;
  SCM Class  = VCELL(intern("<class>"));
  SCM Object = VCELL(intern("<object>"));


  proc 	      = intern("<procedure-class>");
  VCELL(proc) = basic_make_class(Class, proc, LIST1(Class), NIL);

  ent	      = intern("<entity-class>");
  VCELL(ent)  = basic_make_class(Class, ent, LIST1(VCELL(proc)), NIL);

  meth        = intern("<method>");
  VCELL(meth) = basic_make_class(Class, meth, LIST1(Object),
				  LIST3(intern("name"), 
					intern("specializers"),
					intern("procedure")));
  
  gen 	      = intern("<generic>");
  VCELL(gen)  = basic_make_class(VCELL(ent), gen, LIST1(Object),
				  LIST2(intern("name"), intern("methods")));
}

static void create_generic(void)
{
  SCM Generic = VCELL(intern("<generic>"));

  VCELL(intern("no-applicable-method")) = lmake(LIST1(Generic));
}

void init_gf(void)
{
  tc_next_method = add_new_type(&next_method_type);
  create_classes_for_gf();
  create_generic();

  add_new_primitive("add-method", tc_subr_2,	 add_method);
  add_new_primitive("make",	  tc_lsubr,	 lmake);
}
#endif /* defined STKLOS */
