/*---------------------------------------------------------------------*/
/*    Copyright (c) 1994 by Manuel Serrano. All rights reserved.       */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \    /  '                              */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome Send them to                                           */
/*        <Manuel.Serrano@inria.fr>                                    */
/*        Manuel Serrano                                               */
/*        INRIA -- Rocquencourt                                        */
/*        Domaine de Voluceau, BP 105                                  */
/*        78153 Le Chesnay Cedex                                       */
/*        France                                                       */
/*---------------------------------------------------------------------*/


/*---------------------------------------------------------------------*/
/*    serrano/prgm/project/bigloo/runtime1.7/Clib/symbol.c ...         */
/*                                                                     */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Wed Feb 12 14:51:41 1992                          */
/*    Last change :  Sat Oct 22 13:32:47 1994 (serrano)                */
/*                                                                     */
/*    La gestion des symbol (leur creation et la table de hash)        */
/*    -------------------------------------------------------------    */
/*    Dans tout le code qui suit on est oblige de prendre certaines    */
/*    precautions:                                                     */
/*       - Le code ne peut pas etre ecrit en `Bigloo' car le code est  */
/*         commun au mode `scheme' et `xla'. On veut donc que le link  */
/*         avec la libairie `scheme' soit facultatif.                  */
/*       - On est obliger malgre C de respecter les structures `K2'    */
/*         ainsi que l'appel aux procedures d'allocations pour le `GC' */
/*         soit capable d'aller scruter la table des symboles.         */
/*---------------------------------------------------------------------*/
#include <bigloo.h>

/*---------------------------------------------------------------------*/
/*    Quelques prototypes                                              */
/*---------------------------------------------------------------------*/
extern obj_t c_string_to_symbol();
static obj_t make_symbol();
extern obj_t make_vector();
extern long  get_hash_number();

/*---------------------------------------------------------------------*/
/*    Quelques variables statics                                       */
/*---------------------------------------------------------------------*/
static obj_t symtab;
static obj_t c_symtab;

/*---------------------------------------------------------------------*/
/*    init_symbol_table ...                                            */
/*---------------------------------------------------------------------*/
init_symbol_table()
{
   symtab = make_vector( BINT( HASH_TABLE_SIZE ), BNIL );
   c_symtab = CREF( symtab );
}

/*---------------------------------------------------------------------*/
/*    make_symbol ...                                                  */
/*---------------------------------------------------------------------*/
static obj_t
make_symbol( name )
char *name;
{
   obj_t symbol, aux;
   
   symbol = MAKE_OBJECT( SYMBOL_SIZE, HEADER_SYMBOL, aux );
   symbol->symbol_t.name = (char *)ALLOCATE_ATOMIC( strlen( name ) + 1 );
   symbol->symbol_t.cval = BNIL;
   
   strcpy( symbol->symbol_t.name, name );

   return BREF( symbol );
}
   
/*---------------------------------------------------------------------*/
/*    c_string_to_symbol ...                                           */
/*    char * --> obj_t                                                 */
/*---------------------------------------------------------------------*/
obj_t
c_string_to_symbol( name )
char *name;
{
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( name, HASH_TABLE_SIZE_SHIFT );
   bucket = *(obj_t *)(((long)&c_symtab->vector_t) +
                        VECTOR_SIZE + (OBJ_SIZE * hash_number));
   
   if( NULLP( bucket ) )
   {
      obj_t symbol = make_symbol( name );
      
      *(obj_t *)(((long)&c_symtab->vector_t) +
                  VECTOR_SIZE +
                  (OBJ_SIZE * hash_number)) = MAKE_PAIR( symbol, BNIL );
      
      return symbol;
   }
   else
   {
      obj_t run = bucket, back = bucket;
      
      while( !NULLP( run ) &&
	    strcmp( SYMBOL( CAR( run ) ).name, name ) )
         back = run, run = CDR( run );
      
      if( !NULLP( run ) )
         return CAR( run );
      else
      {
         obj_t symbol = make_symbol( name );

         SET_CDR( back, MAKE_PAIR( symbol, BNIL ) );

         return symbol;
      }
   }
}

/*---------------------------------------------------------------------*/
/*    int                                                              */
/*    symbol_exists_p ...                                              */
/*    -------------------------------------------------------------    */
/*    Cette fonction retourne vrai s'il existe un symbol de nom        */
/*    `string' et faux sinon                                           */
/*---------------------------------------------------------------------*/
int
symbol_exists_p( name )
char *name;
{
   long hash_number;
   obj_t bucket;

   hash_number = get_hash_power_number( name, HASH_TABLE_SIZE_SHIFT );
   bucket = *(obj_t *)(((long)&c_symtab->vector_t) +
                        VECTOR_SIZE + (OBJ_SIZE * hash_number));
   
   if( NULLP( bucket ) )
   {
      return 0;
   }
   else
   {
      obj_t run = bucket;
      
      while( !NULLP( run ) &&
				strcmp( SYMBOL( CAR( run ) ).name, name ) )
		   run = CDR( run );

		return !NULLP( run );
   }
}

