;*---------------------------------------------------------------------*/
;*    Copyright (c) 1994 by Manuel Serrano. All rights reserved.       */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \    /  '                              */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome Send them to                                           */
;*        <Manuel.Serrano@inria.fr>                                    */
;*        Manuel Serrano                                               */
;*        INRIA -- Rocquencourt                                        */
;*        Domaine de Voluceau, BP 105                                  */
;*        78153 Le Chesnay Cedex                                       */
;*        France                                                       */
;*---------------------------------------------------------------------*/


;*=====================================================================*/
;*    .../expression.scm ...                                           */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Fri Jun 17 10:08:30 1994                          */
;*    Last change :  Mon Jul  4 16:42:15 1994 (serrano)                */
;*    -------------------------------------------------------------    */
;*    La transformation heap->stack des expressions.                   */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module stack_expression
   (include "Stack/property.sch"
            "Tools/trace.sch"
            "Var/variable.sch")
   (import  tools_shape
            stack_result
            stack_tail
            stack_alloc)
   (export  (stack-expression! body tail? depth)
            (reset-stack-statistics!)
            (default-s-property)
            (trace-stack-statistics)))

;*---------------------------------------------------------------------*/
;*    Des variables pour faire des statistiques                        */
;*---------------------------------------------------------------------*/
(define *nb-of-heap-allocs*  '())
(define *nb-of-stack-allocs* '())

;*---------------------------------------------------------------------*/
;*    reset-stack-statistics! ...                                      */
;*---------------------------------------------------------------------*/
(define (reset-stack-statistics!)
   (set! *nb-of-heap-allocs*  '())
   (set! *nb-of-stack-allocs* '()))

;*---------------------------------------------------------------------*/
;*    default-s-property ...                                           */
;*---------------------------------------------------------------------*/
(define (default-s-property)
   (let ((prop (make-s-property)))
      (s-property-depth-set!         prop  (unspecified))
      (s-property-returned?-set!     prop  '())
      (s-property-captured-set!      prop  (unspecified))
      (s-property-related-set!       prop  '())
      (s-property-stack-alloc?-set!  prop  #f)
      (s-property-reference-set!     prop  (unspecified))
      (s-property-let-binding-set!   prop  (unspecified))
      (s-property-tail-binding?-set! prop (unspecified))
      prop))

;*---------------------------------------------------------------------*/
;*    stack-expression! ...                                            */
;*    -------------------------------------------------------------    */
;*    exp         : l'expression a traiter                             */
;*    tail?       : est-on en position recursive terminale ?           */
;*    depth       : la profondeur dans la pile                         */
;*---------------------------------------------------------------------*/
(define (stack-expression! exp tail? depth)
   (trace (stack loop) "stack-expression!: " (shape exp) #\Newline)
   (if (not (pair? exp))
       (if (and (local? exp)
		(not (eq? (local-class exp) 'function))
                (s-property-stack-alloc? (local-info exp)))
           (stack-reference exp)
           exp)
       (case (car exp)
          ((quote)
           exp)
          ((begin)
           (stack-expression*! (cdr exp) tail? depth)
           exp)
          ((cif)
           (set-car! (cdr exp) (stack-expression! (cadr exp) #f depth))
           (set-car! (cddr exp) (stack-expression! (caddr exp) tail? depth))
           (set-car! (cdddr exp) (stack-expression! (cadddr exp) tail? depth))
           exp)
          ((set!)
           (set-car! (cddr exp) (stack-expression! (caddr exp) #f depth))
           exp)
          ((function function-extra-light function-light)
           exp)
          ((pragma)
           (stack-expression*! (cdr exp) #f depth)
           exp)
	  ((cast)
	   (set-car! (cddr exp) (stack-expression! (caddr exp) tail? depth))
	   exp)
          ((failure)    
           (stack-expression*! (cdr exp) #f depth)
           exp)
          ((typed-case)
           (set-car! (cddr exp) (stack-expression! (caddr exp) #f depth))
           (for-each (lambda (clause)
                        (set-car! (cdr clause)
                                  (stack-expression! (cadr clause)
                                                     tail?
                                                     depth)))
                     (cdddr exp))
           exp)
          ((let)
           (let ((body (caddr exp)))
              (let loop ((bindings (cadr exp)))
                 (if (null? bindings)
                     (begin
                        (set-car! (cddr exp)
                                  (stack-expression! body tail? (+fx depth 1)))
                        exp)
                     (let* ((binding (car bindings))
                            (local   (car binding))
                            (value   (cadr binding)))
                        (if (not (s-property? (local-info local)))
                            (begin
                               (local-info-set! local
						(default-s-property))
                               ;; on met a jour les champs de `s-property'
                               (s-property-depth-set! (local-info local) depth)
                               (s-property-let-binding-set!
                                (local-info local) exp)
                               (s-property-tail-binding?-set!
                                (local-info local) tail?)))
                        ;; de toutes facons, il faut parcourir value ...
                        (set-car! (cdr binding)
                                  (stack-expression! value #f (+fx depth 1)))
                        ;; ... avant meme de se poser la question de savoir
                        ;; si on tranforme l'allocation
                        (if (stackable? local value exp body tail? depth)
                            (let* ((fun (car value))
                                   (cell (assq fun *nb-of-stack-allocs*)))
                               (if (not (pair? cell))
                                   (set! *nb-of-stack-allocs*
                                         (cons (cons fun 1)
                                               *nb-of-stack-allocs*))
                                   (set-cdr! cell (+fx 1 (cdr cell))))
                               (heap->stack! local value)))
                        (loop (cdr bindings)))))))
          ((labels)
           (for-each (lambda (binding)
                        (let ((function (local-value (car binding))))
                           (for-each
                            (lambda (arg)
                               (if (not (s-property? (local-info arg)))
                                   (local-info-set! arg
                                                    (default-s-property)))
                               (s-property-depth-set! (local-info arg) depth))
                            (function-args function))
                           (let ((new-body (stack-expression!
                                            (function-body function)
                                            #t
                                            (+fx depth 1))))
                              (set-car! (cddr binding) new-body)
                              (function-body-set! function new-body))))
                     (cadr exp))
           (set-car! (cddr exp) (stack-expression! (caddr exp)
                                                   tail?
                                                   (+fx depth 1)))
           exp)
          ((block return-from)
           (set-car! (cddr exp) (stack-expression! (caddr exp) #f depth))
           exp)
          ((apply funcall funcall-light funcall-medium)
           (stack-expression*! (cdr exp) tail? depth)
           exp)
          (else
           (stack-expression*! (cdr exp) tail? depth)
           exp))))

;*---------------------------------------------------------------------*/
;*    stack-expression*! ...                                           */
;*---------------------------------------------------------------------*/
(define (stack-expression*! exp tail? depth)
   (let loop ((exps exp))
      (if (null? exps)
          exp
          (begin
             (set-car! exps (stack-expression! (car exps)
                                               (if tail?
                                                   (null? (cdr exps))
                                                   #f)
                                               depth))
             (loop (cdr exps))))))

;*---------------------------------------------------------------------*/
;*    stackable? ...                                                   */
;*    -------------------------------------------------------------    */
;*    Peut-on faire une allocation (si c'en est bien une) en pile ?    */
;*---------------------------------------------------------------------*/
(define (stackable? local value exp body tail? depth)
   (if (not (heap-allocation? value))
       #f
       (begin
          (let* ((fun (car value))
                 (cell (assq fun *nb-of-heap-allocs*)))
             (if (not (pair? cell))
                 (set! *nb-of-heap-allocs* (cons (cons fun 1)
                                                 *nb-of-heap-allocs*))
                 (set-cdr! cell (+fx 1 (cdr cell)))))
          (trace-stackable local value exp body tail? depth)
          (and (not (eq? (local-access local) 'write))
               (not (is-the-temporary-a-result? local exp #t depth))))))

;*---------------------------------------------------------------------*/
;*    trace-stackable ...                                              */
;*---------------------------------------------------------------------*/
(define (trace-stackable local value exp body tail? depth)
   (trace (stack loop)
          "--------------------------------"
          #\Newline
          "stackable?  : " (shape exp)
          #\Newline
          "--------------------------------"
          #\Newline
          "read-only?  : " (not (eq? (local-access local) 'write))
          #\Newline
          "result ?    : " (is-the-temporary-a-result? local exp #t depth)
          #\Newline
          "--------------------------------"
          #\Newline))

;*---------------------------------------------------------------------*/
;*    trace-stack-statistics ...                                       */
;*---------------------------------------------------------------------*/
(define (trace-stack-statistics)
   (trace stack
          "==========================================================="
          #\Newline
          "                 *** stack statistics ***"
          #\Newline
          "-----------------------------------------------------------"
          #\Newline
          "Heap Allocations (before the optimization) : "
          (shape *nb-of-heap-allocs*)
          #\newline
          "stack allocations (after the optimization) : "
          (shape *nb-of-stack-allocs*)
          #\Newline
          "-----------------------------------------------------------"
          #\Newline))
