;;; This code is from Structure and Interpretation of Computer Programs,
;;; by Harold Abelson and Gerald Jay Sussman with Julie Sussman, MIT Press,
;;; 1985.  Copyright 1985 by The Massachusetts Institute of Technology.

;;; Section 2.3.2

;;; Manifest types

(define (attach-type type contents)
  (cons type contents))

(define (type datum)
  (if (not (atom? datum))
      (car datum)
      (error "Bad typed datum -- TYPE" datum)))

(define (contents datum)
  (if (not (atom? datum)) 
      (cdr datum)
      (error "Bad typed datum -- CONTENTS" datum)))

;;; Section 2.3.3

;;; Here is a sample of how an operation is installed in the table:
;(put 'rectangular 'real-part real-part-rectangular)

;;; Here is an example of a default operation; note that it is defined
;;; in terms of generic selectors (real-part, imag-part):
;(put 'default 'magnitude
;     (lambda (z)
;       (sqrt (+ (square (real-part z)) (square (imag-part z))))))

(define (operate op obj)
  (let ((proc (get (type obj) op)))
    (if (not (null? proc))
        (proc (contents obj))
        (let ((proc (get 'default op)))  ; default stuff isn't in the book
          (if (not (null? proc))
              (proc obj)  ; note that obj is passed in still as a typed datum
              (error "Operator undefined for this type -- OPERATE"
                     (list op obj)))))))

;;; The below is magic as far as MC27 is concerned; ignore it for now.

;;; Section 3.3.3 -- Tables
;;; Local tables

(define (make-table)
  (let ((local-table (list '*table*)))
    (define (lookup key-1 key-2)
      (let ((subtable (assq key-1 (cdr local-table))))
        (if (null? subtable)
            nil
            (let ((record (assq key-2 (cdr subtable))))
              (if (null? record)
                  nil
                  (cdr record))))))
    
    (define (insert! key-1 key-2 value)
      (let ((subtable (assq key-1 (cdr local-table))))
        (if (null? subtable)
            (set-cdr! local-table
                      (cons (list key-1
                                  (cons key-2 value))
                            (cdr local-table)))
            (let ((record (assq key-2 (cdr subtable))))
              (if (null? record)
                  (set-cdr! subtable
                            (cons (cons key-2 value)
                                  (cdr subtable)))
                  (set-cdr! record value)))))
      'ok)    
    (define (dispatch m)
      (cond ((eq? m 'lookup-proc) lookup)
            ((eq? m 'insert-proc!) insert!)
            (else (error "Unknown operation -- TABLE" m))))
    
    dispatch))

;;; The PUT and GET operations used in chapter 2

(define operation-table (make-table))
(define get (operation-table 'lookup-proc))
(define put (operation-table 'insert-proc!))