;;; Language generation and parsing

;;; First, we define some example words from a variety of interesting grammatical 
;;; categories.

(define (noun)
  (amb 'student 'professor 'dog 'cat 'class))

(define (verb)
  (amb 'studies 'lectures 'eats 'sleeps))

(define (adj)
  (amb 'brilliant 'tired))

(define (adverb)
  (amb 'quickly 'delightedly 'slothfully))

(define (article)
  (amb 'the 'a))

(define (prep)
  (amb 'for 'to 'in 'by))


;;;
;;; We parse a sentence (list of words) by recursively looking for constituents
;;; of the current grammatical phrase, starting with sentence.  Alternate language
;;; structures are searched for using AMB.  Words are matched using CHECK-WORD,
;;; and the output is built up by BUILD.  Note the use of SET! in check-word,
;;; where it is used to advance down the sentence in an un-doable way.
;;;

(define *s* '())
                         
(define (parse s)
  (set! *s* s)
  (let ((sent (parse-sentence)))
    (if (null? *s*)
        sent
        (fail))))

(define (parse-sentence)
  (build 'sentence
         (parse-noun-phrase)
         (parse-verb-phrase)))

;; Cannot write this because it is left-recursive.  I.e., to parse a noun-phrase,
;; we have to start by parsing a noun-phrase, and this leads to infinite descent.
;; We use the usual parsing technique of look-ahead; see if the prepositional phrase
;; is there before we try to combine it.
;;(define (parse-noun-phrase)
;;  (amb (parse-np3)
;;       (build 'noun-phrase (parse-noun-phrase) (parse-prepositional-phrase))))
;;
;; Instead, we parse the np3 first, and then check to see if there is a pp to follow:

(define (parse-noun-phrase)
  (define (add-pps np)
    (amb np
         (let ((pp (parse-prepositional-phrase)))
           (add-pps (build 'noun-phrase np pp)))))
  (add-pps (parse-np3)))

(define (parse-np3)
  (build 'np3 (parse-article) (parse-np2)))

(define (parse-np2)
  (amb (parse-np)
       (build 'np2 (parse-adjective) (parse-np2))))

(define (parse-adjective)
  (check-word 'adjective adj))

(define (parse-np)
  (amb (check-word 'noun noun)
       (build 'np 
              (check-word 'noun noun)
              (parse-np))))

(define (parse-article)
  (check-word 'article article))

(define (parse-prepositional-phrase)
  (build 'pp (check-word 'prep prep) (parse-np3)))

(define (parse-verb-phrase)
  (define (add-pps vp)
    (amb vp
         (let ((pp (parse-prepositional-phrase)))
           (add-pps (build 'verb-phrase vp pp)))))
  (add-pps
   (amb (check-word 'verb verb)
        (build 'vp (check-word 'verb verb) (check-word 'adverb adverb)))))

(define (check-word part-of-speech generator)
  (cond ((null? *s*) (fail))
	((equal? (car *s*) (generator))
         (let ((it (car *s*)))
           (set! *s* (cdr *s*))
           (build-prim part-of-speech it)))
        (else (fail))))

(define build list)
(define build-prim list)
