;;; -*- Mode:Scheme; Package:S&ICP; Base:10 -*-

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                           ;;
;; PS5-CODE.SCM -- Code from S&ICP for Digital Circuit Simulator and Agenda. ;;
;;                                                                           ;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This is all nearly verbatim from S&ICP... included here for convenience and
;;; cultural enrichment.  

;;; We define our signal levels here:

(define high 1)
(define low  0)
(define same-signal? =)

(define (high? s)
  (same-signal? s high))

(define (low? s)
  (same-signal? s low))


;;; Section 3.3.4 -- Digital-circuit simulator

;;; A couple compound function boxes to play with

(define (half-adder a b s c)
  (let ((d (make-wire))
	(e (make-wire)))
    (or-gate  a b d)
    (and-gate a b c)
    (inverter c e)
    (and-gate d e s)))

(define (full-adder a b c-in sum c-out)
  (let ((s  (make-wire))
        (c1 (make-wire))
        (c2 (make-wire)))
    (half-adder b c-in s   c1)
    (half-adder a s    sum c2)
    (or-gate c1 c2 c-out)))

;;; Primitive function boxes

(define (inverter input output)
  (define (invert-input)
    (let ((new-value (logical-not (get-signal input))))
      (after-delay inverter-delay
                   (lambda ()
                     (set-signal! output
                                  new-value)))))
  (add-action! input invert-input))

(define (logical-not s)
  (cond ((low? s) high)
        ((high? s) low)
        (else (error "Invalid signal -- LOGICAL-NOT" s))))

(define (and-gate a1 a2 output)
  (define (and-action-procedure)
    (let ((new-value (logical-and (get-signal a1)
				  (get-signal a2))))
      (after-delay and-gate-delay
                   (lambda ()
                     (set-signal! output new-value)))))
  (add-action! a1 and-action-procedure)
  (add-action! a2 and-action-procedure))

(define (logical-and s1 s2)
  (cond ((low? s1) low)
	((low? s2) low)
	((and (high? s1) (high? s2)) high)
	(else (error "Invalid signal -- LOGICAL-AND" s1 s2))))

;;; Wires

(define (make-wire)
  (let ((signal-value       low)
	(action-procedures '()))

    (define (set-my-signal! new-value)
      (if (not (same-signal? signal-value new-value))
          (sequence (set! signal-value new-value)
		    (call-each action-procedures))
          'done))

    (define (accept-action-procedure proc)
      (set! action-procedures (cons proc action-procedures))
      (proc))

    (define (dispatch m)
      (cond ((eq? m 'get-signal ) signal-value)
            ((eq? m 'set-signal!) set-my-signal!)
            ((eq? m 'add-action!) accept-action-procedure)
            (else (error "Unknown operation -- WIRE" m))))

    dispatch))

(define (call-each procedures)
  (if (null? procedures)
      'done
      (sequence
       ((car procedures))
       (call-each (cdr procedures)))))

(define (get-signal wire)
  (wire 'get-signal))

(define (set-signal! wire new-value)
  ((wire 'set-signal!)    new-value))

(define (add-action! wire action-procedure)
  ((wire 'add-action!)    action-procedure))

;;; Agenda use

(define (after-delay delay-time action)
  (add-to-agenda! (+ delay-time (current-time the-agenda))
                  action
                  the-agenda))

;;; Top level of simulation

(define (propagate)
  (if (empty-agenda? the-agenda)
      'done
      (let ((first-item (first-agenda-item the-agenda)))
        (first-item)
        (remove-first-agenda-item! the-agenda)
        (propagate))))

;;; Probing a wire

(define (probe name wire)
  (add-action! wire
              (lambda ()        
                (print name) (princ " ")
                (princ (current-time the-agenda))
                (princ "  New-value = ")
                (princ (get-signal wire)))))

;;; Implementation of the agenda

(define (make-time-segment time queue)
  (cons time queue))

(define (segment-time  s) (car s))
(define (segment-queue s) (cdr s))

(define (make-agenda)
  (list '*agenda*
        (make-time-segment 0 (make-queue))))

(define (segments agenda) (cdr agenda))

(define (first-segment agenda) (car (segments agenda)))
(define (rest-segments agenda) (cdr (segments agenda)))

(define (set-segments! agenda segments)
  (      set-cdr!      agenda segments))

(define (current-time          agenda)
  (segment-time (first-segment agenda)))

(define (empty-agenda? agenda)
  (and (empty-queue? (segment-queue (first-segment agenda)))
       (null? (rest-segments agenda))))

(define (add-to-agenda! time action agenda)
  (define (add-to-segments! segments)
    (if (= (segment-time (car segments)) time)
        (insert-queue! (segment-queue (car segments))
                       action)
        (let ((rest (cdr segments)))
          (cond ((null? rest)
                 (insert-new-time! time action segments))
                ((> (segment-time (car rest)) time)
                 (insert-new-time! time action segments))
                (else (add-to-segments! rest))))))
  (add-to-segments! (segments agenda)))

(define (insert-new-time! time action segments)
  (let ((q (make-queue)))
    (insert-queue! q action)
    (set-cdr! segments
              (cons (make-time-segment time q)
                    (cdr segments)))))

(define (remove-first-agenda-item!             agenda)
  (delete-queue! (segment-queue (first-segment agenda))))

(define (first-agenda-item agenda)
  (let ((q (segment-queue (first-segment agenda))))
    (if (empty-queue? q)
        (sequence (set-segments! agenda
				 (rest-segments agenda))
		  (first-agenda-item agenda))
        (front q))))

;;; Section 3.3.2 -- Queues

;;; Representaton of queues

(define (front-ptr queue) (car queue))
(define (rear-ptr  queue) (cdr queue))

(define (set-front-ptr! queue item) (set-car! queue item))
(define (set-rear-ptr!  queue item) (set-cdr! queue item))

;;; Operations on queues

(define (empty-queue? queue) (null? (front-ptr queue)))

(define (make-queue) (cons '() '()))

(define (front queue)
  (if (empty-queue? queue)
      (error "FRONT called with an empty queue" queue)
      (car (front-ptr queue))))

(define (insert-queue! queue item)
  (let ((new-pair (cons item '())))
    (cond ((empty-queue? queue)
           (set-front-ptr! queue new-pair)
           (set-rear-ptr!  queue new-pair)
           queue)
          (else
           (set-cdr! (rear-ptr queue) new-pair)
           (set-rear-ptr!      queue  new-pair)
           queue)))) 

(define (delete-queue! queue)
  (cond ((empty-queue? queue)
         (error "Delete called with an empty queue" queue))
        (else
         (set-front-ptr! queue (cdr (front-ptr queue)))
         queue))) 

;;; A sample simulator

(define the-agenda (make-agenda))

(define inverter-delay 2)
(define and-gate-delay 3)
(define  or-gate-delay 5)

;;;
;;; A useful utility for hacking the problem set
;;;

(define (reset-agenda!)
  (set!    the-agenda (make-agenda)))
