;;; The following utility matcher will compare a pattern, PAT
;;;  with an expression, EXP, relative to a given dictionary
;;;  mapping pattern variables to values.

;;; A pattern matches an expression if every subexpression of the 
;;;  pattern matches the corresponding subexpression of the 
;;;  expression.

;;; If a match succeeds, a dictionary is returned, associating the 
;;;  pattern variables with the subexpressions they matched.
;;; If a match fails, the symbol FAILED is returned.

;;; Patterns contain constant parts and "arbitrary expression" 
;;;  variables.  A match succeeds if all of the constant parts 
;;;  of the pattern are equal to the corresponding parts of the
;;;  expression and all of the variables in the pattern can be
;;;  consistently associated with subexpressions of the given
;;;  expression.

;;; Such an arbitrary expression variable will be associated 
;;;  with the matching expression in the dictionary.  Multiple
;;;  occurrences of an arbitrary expression variable must all
;;;  match the same expression.  A match will fail if an
;;;  arbitrary expression variable must assume two different values.

(define (match pat exp dict)
  (cond ((eq? dict 'failed) 'failed)
        ((atom? pat)
         (if (atom? exp)
             (if (eq? pat exp)
                 dict
                 'failed)
             'failed))
        ((arbitrary-expression? pat)
         (extend-dictionary pat exp dict))
        ((atom? exp) 'failed)
        (else
         (match (cdr pat)
                (cdr exp)
                (match (car pat)
                       (car exp)
                       dict)))))


;;; Given a dictionary, we can instantiate a skeleton
;;;  expression that contains constants and "skeleton 
;;;  evaluation" variables.  We walk the expression tree,
;;;  substituting values for the evaluation variables.
;;;  These values are the ones associated with the 
;;;  variable in the dictionary.

(define (instantiate skel dict)
  (cond ((atom? skel) skel)
        ((skeleton-evaluation? skel)
         (lookup (evaluation-expression skel) dict))
        (else 
         (cons (instantiate (car skel) dict)
               (instantiate (cdr skel) dict)))))

;;; Dictionaries are implemented as association lists:
;;;  An association list is a list of elements, each of 
;;;  which is a pair whose CAR is the key symbol and
;;;  whose CDR holds the value associated with that 
;;;  key.  In this program we actually have the value 
;;;  in the CAR of a pair pointed at by the CDR of the
;;;  pair whose CAR contains the key.


(define (make-empty-dictionary) '())

(define (extend-dictionary pat dat dictionary)
  (let ((vname (variable-name pat)))
    (let ((v (assq vname dictionary)))
      (cond ((not v)
             (cons (list vname dat) dictionary))
            ((equal? (cadr v) dat) dictionary)
            (else 'failed)))))

(define (lookup var dictionary)
  (let ((v (assq var dictionary)))
    (if (not v)
        var
        (cadr v))))



;;;    Pattern syntax

(define (arbitrary-expression? pat)
  (if (pair? pat)
      (eq? (car pat) '?)
      false))

(define (variable-name pat)
  (cadr pat))


;;;    Skeleton syntax

(define (skeleton-evaluation? pat)
  (if (pair? pat)
      (eq? (car pat) '?)
      false))

(define (evaluation-expression evaluation)
  (cadr evaluation))

