;;;; Procedure-tree interpreter for Scheme, extended with AMB.

(declare (usual-integrations))

(define (driver-loop)
  (define (internal-loop fail)
    (newline)
    (display "SEARCH==>> ")
    (let ((exp (read)))
      (if (eq? exp 'try-again)
	  (fail)
	  (begin
	    (newline)
	    (display "Starting a new problem ")
	    ((analyze exp)
	     the-global-environment
	     (lambda (val fail)
	       (newline)
	       (write val)
	       (internal-loop fail))
	     (lambda ()
	       (newline)
	       (display "There are no more values of ")
	       (write exp)
	       (driver-loop)))))))
  (internal-loop
   (lambda ()
     (newline)
     (display "There is no current problem ")
     (driver-loop))))
     


(define the-global-environment)

(define (init)
  (set! the-global-environment
	(extend-environment '() '() the-empty-environment))
  (set! scheme-variable-cache '())
  (driver-loop))

;;; AMBEVAL is used to execute individual AMBScheme expressions 
;;;  from Scheme.  The first successful value is returned, and no 
;;;  provision is made for reentry on failure.  This is mostly 
;;;  to allow AMBLOAD to load files of definitions into the AMBScheme
;;;  global environment.

(define (ambeval exp)
  ((analyze exp) the-global-environment
		 (lambda (val fail) val)
		 (lambda () 'failed)))

(define (ambload filename)
  (with-input-from-file filename
    (lambda ()
      (let lp ((exp (read)))
	(if (eof-object? exp)
	    'done
	    (begin (ambeval exp)
		   (lp (read))))))))



;;; Produces (lambda (env succeed fail) ...)

(define (analyze exp)
  (cond ((self-evaluating? exp) (analyze-self-evaluating exp))
	((quoted? exp) (analyze-quoted exp))
	((variable? exp) (analyze-variable exp))
	((assignment? exp) (analyze-assignment exp))
	((definition? exp) (analyze-definition exp))
	((if? exp) (analyze-if exp))
	((lambda? exp) (analyze-lambda exp))
	((begin? exp) (analyze-sequence (begin-actions exp)))
	((amb? exp) (analyze-amb exp))
	((cond? exp) (analyze (COND->IF exp)))
	((let? exp) (analyze (LET->combination exp)))
	((application? exp) (analyze-application exp))
	(else
	 (error "Unknown expression type -- ANALYZE" exp))))

(define (analyze-self-evaluating exp)
  (lambda (env succeed fail)
    (succeed exp fail)))

(define (analyze-quoted exp)
  (let ((qval (text-of-quotation exp)))
    (lambda (env succeed fail)
      (succeed qval fail))))

(define (analyze-variable exp)
  (lambda (env succeed fail)
    (succeed (lookup-variable-value exp env)
	     fail)))

(define (analyze-assignment exp)
  (let ((var (assignment-variable exp))
	(vproc (analyze (assignment-value exp))))
    (lambda (env succeed fail)
      (vproc env
	     (lambda (val fail)
	       (let ((oval (lookup-variable-value var env)))
		 (set-variable-value! var val env)
		 (succeed 'done
			  (lambda ()
			    (set-variable-value! var oval env)
			    (fail)))))
	     fail))))

(define (analyze-definition exp)
  (let ((var (definition-variable exp))
	(vproc (analyze (definition-value exp))))
    (lambda (env succeed fail)
      (vproc env			
	     (lambda (val fail)
	       (define-variable! var val env)	;doesn't undefine on failure.
	       (succeed 'done fail))
	     fail))))


(define (analyze-lambda exp)
  (let ((vars (lambda-parameters exp))
	(bproc (analyze-sequence (lambda-body exp))))
    (lambda (env succeed fail)
      (succeed (make-procedure vars bproc env)
	       fail))))

(define (analyze-if exp)
  (let ((pproc (analyze (if-predicate exp)))
	(cproc (analyze (if-consequent exp)))
	(aproc (analyze (if-alternative exp))))
    (lambda (env succeed fail)
      (pproc env
	     (lambda (pval fail)
	       (if pval
		   (cproc env succeed fail)
		   (aproc env succeed fail)))
	     fail))))

(define (analyze-sequence exps)
  (define (sequentially a b)
    (lambda (env succeed fail)
      (a env
	 (lambda (va fail)
	   (b env
	      succeed
	      fail))
	 fail)))
  (let ((procs (map analyze exps)))
    (if (null? procs)
	(error
	 "BEGIN requires subexpressions -- ANALYZE" exps))
    (define (loop first rest)
      (if (null? rest)
	  first
	  (loop (sequentially first (car rest))
		(cdr rest))))
    (loop (car procs) (cdr procs))))

(define (analyze-amb exp)
  (let ((alternatives (map analyze (operands exp))))
    (lambda (env succeed fail)
      (let try-next
	  ((alternatives alternatives))
	(if (null? alternatives)
	    (fail)
	    ((car alternatives)
	     env
	     succeed
	     (lambda ()
	       (try-next (cdr alternatives)))))))))

(define (analyze-application exp)
  (let ((fproc (analyze (operator exp)))
	(aprocs (map analyze (operands exp))))
    (lambda (env succeed fail)
      (fproc env
	     (lambda (proc fail)
	       (evlist aprocs
		       env
		       (lambda (args fail)
			 (exapply proc args succeed fail))
		       fail))
	     fail))))

(define (exapply proc args succeed fail)
  (cond ((primitive-procedure? proc)
	 (succeed (apply-primitive-procedure proc args)
		  fail))
	((compound-procedure? proc)
	 ((procedure-body proc)
	  (extend-environment (procedure-parameters proc)
			      args
			      (procedure-environment proc))
	  succeed
	  fail))
	(else
	 (error "Unknown procedure type -- RTAPPLY"
		proc))))

(define (evlist operands env succeed fail)
  (if (null? operands)
      (succeed '() fail)
      ((car operands)
       env
       (lambda (arg fail)
	 (evlist (cdr operands)
		 env
		 (lambda (args fail)
		   (succeed (cons arg args)
			    fail))
		 fail))
       fail)))
