;;;
;;;   T E S T   T O O L B O X
;;;   = = = =   = = = = = = =
;;;

; Routines to assist in testing.
;
; Invoke using the expressions
;      (set! TestDir <TESTDIR>)
;      (TestAFile <PACKAGENAME>)
;
; The file <TESTDIR><PACKAGENAME>.tst should contain a
; sequence of expressions in one of the following forms:
;     1)   ! expression
;     2)   = expression expected-val
;     3)   ? expression expected-exp
;     4)   * expression
; Form 1) is used to perform an action without any test.
; Form 2) evaluates expression and checks whether the result
;         is equal? to expected-val. A special form of equal?
;         is used to handle system dependant values of
;         INFINITY and NEG-INFINITY.
; Form 3) is like form 2) but evaluates the expected-exp
;         as well as the expression.
; Form 4) evaluates expression and checks that it invokes
;         Fatal-Error.
;
; If TestVerbose is #f then only FAILED expressions are
; reported.

(define TestDir "")

(define TestAFile
  (lambda (aName)
    (let ((fullName
           (string-append TestDir
                          (string-append aName ".tst"))))
      (display "Testing ") (display aName) (display " from ")
      (display fullName) (display " ...") (newline)
      (call-with-input-file fullName GetAnAction))))

(define TestVerbose #f)

(define TestDisplay
  (lambda (anExpr)
    ; Only display if TestVerbose requires it
    (if TestVerbose (display anExpr) #f)))
  
(define TestPrettyPrint
  (lambda (anExpr)
    ; Only pretty-print if TestVerbose requires it
    (if TestVerbose (MyPrettyPrint anExpr) #f)))

(define TestNewline
  (lambda ()
    ; Only newline if TestVerbose requires it
    (if TestVerbose (newline) #f)))
  
(define GetAnAction
  (lambda (aPort)
    (let ((actionExp (read aPort)))
      (if (eof-object? actionExp)
          'done
          (begin
           (cond ((eq? actionExp '!) (DoForEffect
                                      (read aPort)))
                 ((eq? actionExp '=) (TestAPair aPort))
                 ((eq? actionExp '?) (TestEvalPair aPort))
                 ((eq? actionExp '*) (TestForError
                                      (read aPort)))
                 (else (Fatal-Error
                        "Unexpected action in tests")))
           (GetAnAction aPort))))))

(define DoForEffect
  (lambda (anExp)
    (TestDisplay "DOING ") (TestPrettyPrint anExp)
    (EvalExp anExp) (TestNewline)))

(define MyEqual?
  (lambda (a b)
    ; Like equal? but treats NEG-INFINITY and INFINITY
    ; real numbers and procedures specially.
    (cond ((procedure? a) (procedure? b))
          ((integer? a) (cond ((= a NEG-INFINITY)
                              (eqv? b 'NEG-INFINITY))
                             ((= a INFINITY)
                              (eqv? b 'INFINITY))
                             ((integer? b)
                              (= a b))
                             (else #f)))
          ((real? a) (and (real? b)
                          (or (and (zero? a)
                                   (zero? b))
                              (< (abs (/ (- a b) a)) 0.0001))))
          ((pair? a) (and (pair? b)
                          (MyEqual? (car a) (car b))
                          (MyEqual? (cdr a) (cdr b))))
          ((vector? a)
           (and (vector? b)
                (= (vector-length a)
                   (vector-length b))
                (do ((ok #t)
                     (i (vector-length a)))
                    ((and ok
                          (positive? i)) ok)
                    (set! i (- i 1))
                    (if (MyEqual? (vector-ref a i)
                                  (vector-ref b i))
                        #f
                        (set! ok #f)))))
          ((string? a) (and (string? b)
                            (string=? a b)))
          (else (eqv? a b)))))

(define TestAPair
  (lambda (aPort)
    (let* ((exp (read aPort))
           (expected (read aPort))
           (result (EvalExp exp)))
      (if (MyEqual? result expected)
          (begin (TestDisplay "TEST ")
                 (TestPrettyPrint exp)
                 (TestDisplay "...PASSED")
                 (TestNewline))
          (begin (display "TEST ")
                 (MyPrettyPrint exp)
                 (display "...FAILED, should give ")
                 (MyPrettyPrint expected)
                 (display " but gives ")
                 (MyPrettyPrint result)                                    
                 (newline))))))

(define TestEvalPair
  (lambda (aPort)
    (let* ((exp (read aPort))
           (expected (read aPort))
           (result (EvalExp exp))
           (exp-result (EvalExp expected)))
      (if (MyEqual? result exp-result)
          (begin (TestDisplay "TEST ")
                 (TestPrettyPrint exp)
                 (TestDisplay "...PASSED")
                 (TestNewline))
          (begin (display "TEST ")
                 (MyPrettyPrint exp)
                 (display "...FAILED, should give ")
                 (MyPrettyPrint exp-result)
                 (display " but gives ")
                 (MyPrettyPrint result)                                    
                 (newline))))))

(define TestForError
  ; note that there is an error in TauScheme that causes the
  ; input port to be closed after the Error-Continuation is
  ; called so error testing is skipped for TauScheme
  (lambda (anExp)
    (let ((saveCont Error-Continuation)
          (result #f))
      (set! result
            (if (string=? *SchemeVersion* "TauScheme")
                "ERROR-VALUE"
                (call-with-current-continuation
                 (lambda (aContinuation)
                   (set! Error-Continuation aContinuation)
                   (EvalExp anExp)))))
      ; restore Error-Continuation
      (set! Error-Continuation saveCont)
      ; when gets here result should be ERROR-VALUE
      (if (equal? result "ERROR-VALUE")
          (begin (TestDisplay "TEST ")
                 (TestPrettyPrint anExp)
                 (TestDisplay "...PASSED")
                 (TestNewline))
          (begin (display "TEST ")
                 (MyPrettyPrint anExp)
                 (display "...FAILED  should give")
                 (display " ERROR-VALUE but gave ")
                 (MyPrettyPrint result)
                 (newline))))))
