       
; ********** The "Lineland" microworld **********
; This is a problem of reverse discrimination involving a 5 seat room containing
; 2 students and 2 lecturers. Initially all the lecturers are in front of the
; students and the empty seat is at the front of the room. The goal is to
; reverse the situation so all students are in front of all lecturers
; (with no regard to where the empty seat occurs). Valid moves for any
; lecturer or student are:
;
; (1) Move one position in either direction into the adjacent empty seat
; (2) Jump over one person in either direction into the empty seat

(define LL-SearchProblem #f)
    
(define SetUpLineLand
  (lambda ()
    
    ; represent room state as in (e l l s s) where 'e', 'l'
    ; and 's' are the empty seat, a lecturer or student
    ; respectively. Thus the inital state is:
    
    (define initialState '(e l l s s))
    
    ; the goal is to reach the state (s s l l) when the
    ; position of the empty seat is ignored
    
    (define removeEmptySeat
      (lambda (aState)
        (if (EmptyState? aState)
            ()
            (if (eq? 'e (FirstElement aState))
                (removeEmptySeat (RestOfElements aState))
                (cons (FirstElement aState)
                      (removeEmptySeat
                       (RestOfElements aState)))))))
    
    (define goalFN
      (lambda (aState) (equal? '(s s l l) (removeEmptySeat aState))))
    
    ; the actions are easier to deal with by letting the
    ; empty seat move 1 or 2 places in either direction.
    
    (define moveEmpty
      (lambda (n)
        ; return a function that will move the empty seat n
        ; to left if n is negative, and n to right if positive
        (lambda (aState)
          (let* ((origin (FindFirstSlotOfSymbol 'e aState))
                 (destination (+ origin n)))
           (cond ((or (<= origin 0)
                      (> origin (length aState))
                      (<= destination 0)
                      (> destination (length aState))) #f)
                 (else
                  (let ((destSym
                         (FindSymbolInSlot destination aState)))
                    (FillSlot destination
                              'e
                              (FillSlot origin destSym aState)))))))))
    
    ; the measure of closeness to the solution is obtained by accumulating
    ; a penalty score over all lecturers for each student that is behind the
    ; lecturer. This is improved by weighting each students contribution
    ; by 1 for the first lecturer and 2 for the second lecturer and also
    ; including the number of seats the student is behind the lecturer.
    ; This gives a penalty of zero for the goal states and a negative value
    ; for others. The initial state gives a value of -11 with a contribution
    ; of -5 for the first lecturer (students 2 and 3 seats behind) and a
    ; contribution of -6 (-2*3) for the second lecturer (students 1 and 2
    ; seats behind).
    
    (define computePenalty
      (lambda (aState)
        (do ((score 0)
             (lecturerNumber 1 (+ lecturerNumber 1))
             (section (member 'l (removeEmptySeat aState))
                      (member 'l section)))
            ((not section) score)
            (do ((relPos 0 (+ relPos 1))
                 (subSection section (cdr subSection))
                 (lecturerScore 0))
                ((null? subSection)
                 (set! score (+ score lecturerScore)))
                (if (eq? 's (car subSection))
                    (set! lecturerScore
                          (- lecturerScore (* lecturerNumber relPos)))
                    #f))
            (set! section (cdr section)))))
        
    ; Now set up the Search Problem

    (set! LL-SearchProblem
          (MakeSearchProblem initialState
                             goalFN
                             computePenalty
                             (MakeActionList
                               (moveEmpty -2)
			       (moveEmpty -1)
			       (moveEmpty +1)
			       (moveEmpty +2))))))

(begin (SetUpLineLand)
       #t)
