;;;
;;;      R E S T A U R A N T   S Y S T E M  - section 3.8.1
;;;      = = = = = = = = = =   = = = = = =

; This system illustrates the use of three distinct toolboxes
; interacting together to provide a system that can be used
; to maintain information about restaurants. The ATN toolbox
; is used to provide a natural language interface to the system
; so the user can ask questions and supply new information to
; the system. A frame network from the Frames toolbox is used
; to store the information about restaurants and the relationships
; among them. For example it hold the information that a Licenced
; Restaurant is a subclass of a Restaurant and hence inherits a
; number of properties that are in common with all restaurants.
; Finally the Production Systems toolbox is used to reason about
; some of the properties of restaurants. For example if it is
; known that the restaurant has ashtrays or has candles then
; it will have a smoky atmosphere.
; It should be stressed that this is only a simple example and
; some of the properties about restaurants are greatly simplified
; (or are even incorrect) but it should give the flavour of a
; larger, more realistic example.

; User Interface
; --------------

; The form of sentences accepted by the system are either statements
; or questions about restaurants in the form of a list.

; The form of valid statements are either statements about something
; being a subclass of something else such as:
;    a) The KulahKhan is a ChineseRestaurant
; or a statement about a property held by a class or specific
; restaurant such as:
;    b) A TakeAWay serves fast food

; The form of valid questions are either statements about whether
; something is a subclass or example of something else such as:
;    c) Is the KulahKhan is a ChineseRestaurant
; or a question about whether a class of or specific restaurant
; has a particular property such as:
;    d) Does the KulahKhan have fast food
; or a request to ask for all the information available about a
; given restaurant as in:
;    e) Describe the KulahKhan
;
; Finally the sentence (bye) is used to terminate the conversation

; The properties are limited to adjective noun pairs such as:
;     "fast food" "orange counter" "4 chairs"
; If the adjective is missing it defaults to "Yes" when stored.

; The following ATN's are used to parse these sentences:

; CLASS is used to represent a resturant or class of them etc and is
; just a single word (that is not one of the other keywords allowed
; in sentences namely: the a an is has does have)

(define notKeyWord?
  (lambda (aRegisterBank)
    ; check that the current value of the register '**' is not a valid keyword
    (not (member (GetRegister aRegisterBank '**)
		 '(the a an is has does have own owns ?))) ))

(define CLASS
  (MakeATN (STATE 'C1 (ANY notKeyWord? (SETR 'class **)
                                       (TO 'C2)))
	   (STATE 'C2 (POP (GETR 'class) #t))))

; taCLASS represents one of:
;
;      CLASS          - same as next line
;      the CLASS
;      article CLASS

(define article (MakeCategory 'a 'an 'all 'most))

(define taCLASS
  (MakeATN
   (STATE 'taC1 
          (CAT  article #t (SETR 'specific #f)
                           (TO 'taC2))
          (CAT  (MakeCategory 'the) #t (SETR 'specific #t)
                                       (TO 'taC2))
          (PUSH CLASS   #t (SETR 'specific #t)
                           (TO 'taC3)))		          	    
   (STATE 'taC2 (PUSH CLASS   #t (TO 'taC3)))
   (STATE 'taC3 (POP  (GETR 'class) #t))))

; aCLASS represents article CLASS

(define aCLASS
  (MakeATN (STATE 'aC1 (CAT  article #t (TO 'aC2)))
	   (STATE 'aC2 (PUSH CLASS   #t (TO 'aC3)))
	   (STATE 'aC3 (POP  (GETR 'class) #t))))

; Prop is an adjective noun pair. Allow anything that is not a
; keyword for these. Also allow the adjective to be left out.

(define Prop
  (MakeATN
   (STATE 'P1 (ANY notKeyWord? (SETR 'adjective **)
                               (TO 'P2))
              (ANY notKeyWord? (SETR 'adjective #f)
                               (SETR 'noun **)
			       (TO 'P3)))
   (STATE 'P2 (ANY notKeyWord? (SETR 'noun **)
                               (TO 'P3)))
   (STATE 'P3 (POP (GETR 'noun) #t))))

; form of statements

(define hasCategory
  (MakeCategory 'has 'serves 'owns 'provides 'offers))

(define haveCategory
  (MakeCategory 'have 'serve 'own 'provide 'offer))

(define statement
  (MakeATN
   (STATE 'S1 (PUSH taCLASS #t (TO 'S2)))
   (STATE 'S2 (CAT (MakeCategory 'is) #t 
		   (SETR 'prop #f)
		   (SETR 'subclass (GETR 'class))
		   (TO 'S3))
              (CAT hasCategory #t 
		   (SETR 'prop #t)
		   (SETR 'propclass (GETR 'class))
		   (TO 'S5)))
   (STATE 'S3 (PUSH aCLASS  #t 
		    (SETR 'superclass (GETR 'class))
		    (TO 'S4)))
   (STATE 'S4 (POP #t #t))
   (STATE 'S5 (PUSH Prop #t (TO 'S6)))
   (STATE 'S6 (POP #t #t))))

; form of questions

(define question
  (MakeATN
   (STATE 'Q1 (CAT (MakeCategory 'is) #t (SETR 'prop #f)
                                         (TO 'Q2))
              (CAT (MakeCategory 'does 'has) #t (SETR 'prop #t)
                                                (TO 'Q5)))
   (STATE 'Q2 (PUSH taCLASS #t 
                            (SETR 'subclass (GETR 'class))
                            (TO 'Q3)))
   (STATE 'Q3 (PUSH aCLASS #t 
                           (SETR 'superclass (GETR 'class))
                           (TO 'Q4)))
   (STATE 'Q4 (CAT (MakeCategory '?) #t (TO 'Q4))
	      (POP #t #t)) 
   (STATE 'Q5 (PUSH taCLASS #t 
		    (SETR 'propclass (GETR 'class))
		    (TO 'Q6)))
   (STATE 'Q6 (CAT haveCategory #t (TO 'Q7)))
   (STATE 'Q7 (PUSH PROP        #t (TO 'Q4)))))

(define describe
  (MakeATN
   (STATE 'D1 (CAT (MakeCategory 'Describe) #t (SETR 'prop #f)
                                               (TO 'D2)))
   (STATE 'D2 (PUSH taCLASS #t (TO 'D3)))
   (STATE 'D3 (POP #t #t))))

; form of interface sentences

(define InterfaceATN
  (MakeATN
   (STATE 'I1 
          (PUSH statement #t 
                          (SETR 'type (lambda (aRB) 'stmt))
                          (TO 'I2))
          (PUSH question #t 
                         (SETR 'type (lambda (aRB)'quest))
                         (TO 'I2))
          (PUSH describe #t 
                         (SETR 'type (lambda (aRB)'desc))
                         (TO 'I2)))
   (STATE 'I2 (POP (GETR 'type) #t))))

(define ReadAndProcessASentence
  (lambda ()
    
    (define invalidSentence?
      ; check that it is a list first...
      (lambda (aSentence) (not (pair? aSentence))))
    
    (newline)
    (display "...> ")
    (let ((aSentence (read)))
      (newline)
      (if (invalidSentence? aSentence)
          (begin 
            (newline)
	    (DisplayLine "Please enter your sentence as a list")
	    (ReadAndProcessASentence))
          (if (equal? aSentence '(bye))
              #t
              (let ((registers (Parse InterfaceATN aSentence #f)))
                (if (not registers)
                    (DisplayLine "Sentence not understood")
                    (ProcessSentence aSentence registers))
                #f))))))

; After a successful parse the state of the ATN registers are as follows:
;   Register     Values
;   --------     ------
;   type         'stmt' if statement,
;                'quest' if question,
;                'desc' if describe 
;   prop         #t if property involved - e.g. "large tables"
;   specific     #t if specific "the" rather than general
;                "a" or "an" is used - e.g. "large" tables
;   adjective    the adjective used for the property (if prop=#t)
;                will be #f if none was given
;   noun         the noun used for the property (if prop=#t)
;               - e.g. large "tables"
;   propowner    the identifier used for the class involved
;                in a property question or statement (if prop=#t)
;   subclass     the identifier used for the subclass object
;                in a containment question or statement
;                (if type not desc and prop=#f)
;   superclass   the identifier used for the superclass object
;                in a containment question or statement
;                (if type not desc and prop=#f)
;   class        the identifier used for the class object
;                in a describe question (type=desc)


(define ProcessSentence
  (lambda (aSentence aRegisterBank)
    (let ((type (GetAssociationValue 'type aRegisterBank))
          (prop (GetAssociationValue 'prop aRegisterBank))
          (specific (GetAssociationValue 'specific aRegisterBank)))
      (if prop
          ; question or statement about properties
          (let ((adjective (GetAssociationValue 'adjective aRegisterBank))
                (noun (GetAssociationValue 'noun aRegisterBank))
                (propowner (GetAssociationValue 'propclass
						aRegisterBank)))
            (if (eq? type 'stmt)
                (ProcessPropertyStatement specific
					  propowner
					  adjective
					  noun)
                (ProcessPropertyQuestion specific
					 propowner
					 adjective
					 noun)))
          (if (eq? type 'desc)
              ; request for a frame description
              (let ((class (GetAssociationValue 'class aRegisterBank)))
                (ProcessDescribe specific class))
              (let ((subclass (GetAssociationValue 'subclass aRegisterBank))
                    (superclass (GetAssociationValue 'superclass aRegisterBank)))
                (if (eq? type 'stmt)
                    ; "containment" (isa) question or statement
                    (ProcessInclusionStatement 
                        specific  subclass  superclass)
                    (ProcessInclusionQuestion 
                        specific  subclass superclass))))))))

; Initial Frame Database and definitions
; ------- ----- -------- --- -----------

(define CountMenu
  ; IF-NEEDED demon: decides on variety by counting the number of items
  ; on the menu
  (lambda (aFN aFrame aSlot)
    (let ((menu (GetAValue aFN aFrame 'Menu)))
      (if (and (pair? menu) (> (length menu) 2))
          'yes
          'no))))

(define RefuseToAdd
  ; IF-ADDED demon: return #f to stop addition of some item
  (lambda (aFN aFrame aSlot aValue)
    (if (eq? aValue 'yes)
        (begin (DisplayLine "Sorry, but the" aFrame
                            "is not allowed to provide" aSlot)
               #f)
	aFN)))

(define RiotIfRemoved
  ; IF-NEEDED demon: return #f to stop removal of ashtrays
  (lambda (aFN aFrame aSlot aValue)
    (DisplayLine "I can't remove" aSlot "from a" aFrame)
    (DisplayLine "This might start a riot among the clientele")                
    #f))

(define RestaurantWorld
  (MakeFN
   (FRAME 'Restaurant 
          (SLOT 'Food     (DEFAULT 'edible))
          (SLOT 'Tables   (DEFAULT 'yes))
          (SLOT 'Toilets  (DEFAULT 'yes))
          (SLOT 'Portions (DEFAULT 'reasonable))
          (SLOT 'Variety  (IF-NEEDED CountMenu)))
   (FRAME 'BYO  (AKO 'Restaurant)
	  (SLOT 'Portions (DEFAULT 'large))
	  (SLOT 'WineList (VALUE   'no) 
		          (IF-ADDED RefuseToAdd)))
   (FRAME 'Licenced (AKO 'Restaurant)
	  (SLOT 'WineList (DEFAULT 'yes))
	  (SLOT 'Candles  (DEFAULT 'yes))
	  (SLOT 'Food     (DEFAULT 'good)))
   (FRAME 'TakeAWay (AKO 'Restaurant)
	  (SLOT 'WineList (DEFAULT 'no))
	  (SLOT 'Toilets  (DEFAULT 'no))
	  (SLOT 'Tables   (DEFAULT 'no)))
   (FRAME 'PanchosPandemonium (AKO 'BYO)
	  (SLOT 'Menu     (VALUE '(nachos tostadas tachos)))
	  (SLOT 'Food     (VALUE 'good)))
   (FRAME 'Vegetarian (AKO 'BYO)
	  (SLOT 'Menu     (VALUE '(greens spouts)))
	  (SLOT 'Meat     (DEFAULT 'no)
		          (IF-ADDED RefuseToAdd))
	  (SLOT 'Ashtrays (VALUE   'no)
		          (IF-ADDED  RefuseToAdd)))
   (FRAME 'PinkPlatypus (AKO 'Vegetarian)
	  (SLOT 'Food     (VALUE 'good)))
   (FRAME 'French (AKO 'Licenced)
	  (SLOT 'Portions (VALUE 'small))
	  (SLOT 'Menu     (VALUE '(snails truffles))))
   (FRAME 'ChezChristophe (AKO 'French)
	  (SLOT 'Ashtrays (VALUE 'yes))
	  (SLOT 'Food     (VALUE 'mediocre)))
   (FRAME 'Italian (AKO 'Licenced)
	  (SLOT 'Portions (VALUE 'large))
	  (SLOT 'Menu     (VALUE '(pasta pizza))))
   (FRAME 'LuigisLusciousLinguini (AKO 'Italian)
	  (SLOT 'Ashtrays (VALUE 'no)))
   (FRAME 'DirtyDicks (AKO 'TakeAWay)
	  (SLOT 'Ashtrays (VALUE 'yes)
		          (IF-REMOVED RiotIfRemoved))
	  (SLOT 'Menu     (VALUE '(burger fish fries))))))

(define ProcessPropertyStatement
  ; process valid assertions: (the ... has ...)
  (lambda (specific propowner adjective noun)
    (set! RestaurantWorld
          (AddAValue RestaurantWorld
                     propowner
                     noun
                     (if (not adjective) 'yes adjective)))))

; Routines to process the statements and questions
; -------- -- ------- --- ---------- --- ---------

(define ProcessPropertyQuestion
  ; process valid queries: (does the ... have ... ?)
  (lambda (specific propowner adjective noun)
    
    (define testForItemOnMenu
      (lambda (item menu) 
        (if (pair? menu)
            (member item menu)
            (equal? item menu))))
    
    (let ((value 
           (GetAValue RestaurantWorld 
                      propowner 
                      noun)))
      (cond 
       ; test if a particular menu item is asked for
       ((equal? adjective 'any)
        (let ((menu (GetAValue RestaurantWorld
			       propowner
			       'menu)))
          (if (testForItemOnMenu noun menu)
              (DisplayLine "Of course - quite a wide range in fact")
              (DisplayLine "Unfortunately not, but they do offer"
			   menu))) )
       ; test if property is asked for about which no information is 		
       ; stored. If yes, invoke the forward chainer.
       ((equal? value "NOTKNOWN")
        (TryByInference propowner 
                        adjective 
                        noun))
       ; if information is given, then it must either be a yes/no question 
       ; (e.g. toilets = yes), or a question asking for a specific property
       ; value (e.g. toilets = clean).  
       ; Test for and process yes/no queries first.  
       ((not adjective)
        (cond ((eq? value 'yes)
               (DisplayLine "Yes, it seems very likely that it has"
			    noun))
              ((eq? value 'no)
               (DisplayLine "No, it's unlikely to have much" 
			    noun))
              (else 
               (DisplayLine "Yes it has" value noun))))
       ; - followed by processing of more specific queries 
       (else
        (cond ((and (eq? adjective 'no) (eq? value 'yes))
               (DisplayLine "No, it does have" noun))
              ((eq? value 'yes)
               (DisplayLine "Sorry, I know that it will have" noun
			    "but not if they are" adjective))
              ((equal? value adjective)
               (DisplayLine "Yes it does"))
              ((eq? value 'no)
               (DisplayLine "No, it does not have any" noun))
              (else 
               (DisplayLine "No, but it does have" value noun))))))))

(define ProcessInclusionStatement
  ; process valid class/subclass assertions: (the ... is ...)
  (lambda (specific subclass superclass)
    (set! RestaurantWorld
          (AddAKOLink RestaurantWorld 
                      subclass 
                      superclass))))

(define ProcessInclusionQuestion
  ; process valid class/subclass queries (is the ... a ... ?)
  (lambda (specific subclass superclass)
    (let ((chain (GetAKOChain RestaurantWorld subclass)))
      (if (member superclass chain)
          (DisplayLine "Yes it is")
          (DisplayLine "Not that I know of")))))

(define ProcessDescribe
  ; respond to requests for more information with a "frame dump"
  (lambda (specific class)
    (DisplayLine "Here is all I know about"
                 (if specific "the" "") class)
    (let ((frame (FindFrame RestaurantWorld class)))
      (if frame
          (PrintFrame frame)
          (DisplayLine "nothing known")))))

; Routines to reason about properties using Production Systems
; -------- -- ------ ----- ---------- ----- ---------- -------

(define HasSnails?
  ; check if list bound to x contains the symbol "snails" ? 
  (lambda (anAList)
    (let ((list (GetAssociationValue 'x anAList)))
      (and (pair? list) (member 'snails list)))))

(define HasFries?
  ; check if list bound to x contains the symbol "fries" ? 
  (lambda (anAList)
    (let ((list (GetAssociationValue 'x anAList)))
      (and (pair? list) (member 'fries list)))))

(define RestaurantPM
  (MakePM
   (RULE 'smokeyness? 
         (CONDITIONS  (PATTERN 'yes 'ashtrays))
         (CONCLUSIONS (ASSERT '(smokey atmosphere))
                      (ASSERT '(unhealthy atmosphere))))
   (RULE 'intimacy? 
         (CONDITIONS  (PATTERN 'yes 'Candles))
         (CONCLUSIONS (ASSERT '(smokey   atmosphere))
                      (ASSERT '(intimate atmosphere))))
   (RULE 'classyness?  
         (CONDITIONS  (PATTERN 'yes 'tables)
                      (PATTERN 'yes 'wineList))
         (CONCLUSIONS (ASSERT '(yes class))))
   (RULE 'romanticity? 
         (CONDITIONS  (PATTERN 'intimate 'atmosphere)
                      (PATTERN 'yes 'class)
                      (PATTERN 'expensive 'bills))
         (CONCLUSIONS (ASSERT '(yes romantic))))
   (RULE 'Portions->Expense? 
         (CONDITIONS  (PATTERN 'small 'portions))
         (CONCLUSIONS (ASSERT '(expensive bills))))
   (RULE 'Snails->Expense?  
         (CONDITIONS  (PATTERN (? 'x HasSnails?) 'menu))
         (CONCLUSIONS (ASSERT '(expensive bills))))
   (RULE 'cheapness?  
         (CONDITIONS  (PATTERN (? 'x HasFries?) 'menu))
         (CONCLUSIONS (ASSERT '(cheap bills))))
   (RULE 'familyClientele?  
         (CONDITIONS  (PATTERN (? 'x HasFries?) 'menu)
                      (PATTERN 'yes 'Toilet))
         (CONCLUSIONS (ASSERT '(family clientele))))
   (RULE 'businessClientele? 
         (CONDITIONS (PATTERN 'expensive 'bills))
         (CONCLUSIONS (ASSERT '(business clientele)))    
         ) ))

(define BuildMemory
  ; construct a working memory containing all facts known about 
  ; a relevant class of properties 
  (lambda (propowner)
    (let ((chain (GetAKOChain RestaurantWorld propowner))
          (nameList ())
          (aWM (MakeWM)))
      ; go through each slot in each frame to build a list of all slot names,
      ; and add them as facts to the working memory
      (for-each 
       (lambda (aFrameName)
         (set! aWM 
               (AddFact aWM (list 'CLASS aFrameName)))
         (let ((slots
                (GetSlots (FindFrame RestaurantWorld
				     aFrameName))))
           (for-each 
            (lambda (aSlot)
              (let ((aSlotName (GetSlotName aSlot)))
                (if (or (AKOSlot? aSlot)
                        (member aSlotName nameList))
                    #f
                    (set! nameList (cons aSlotName nameList)))))
            slots)))
       chain)
      ; go through names, getting their value, 
      ; and add them as facts to aWM
      (for-each 
       (lambda (aName)
         (let ((value (GetAValue RestaurantWorld
				 propowner
				 aName)))
           (set! aWM (AddFact aWM (list value aName)))))
       nameList)
      aWM)))

(define TryByInference
  (lambda (propowner adjective noun)
    (DisplayLine "Sorry, I can't remember if"
                 propowner
                 "has"
                 adjective
                 noun)
    (DisplayLine "let me think for a while !")
    ; We begin by constructing a working memory containing all the facts
    ; known about "propowner" (recursing through all its ancestors).  
    ; We then find all the facts that may be concluded from this,
    ; followed  by a check on whether the requested property is present
    ; in the newly constructed working memory
    (let ((newWM 
           (ForwardChainer RestaurantPM
                           (BuildMemory propowner)
                           #t))) ; turn the "verbose" flag on
      (if (eq? adjective #f)
          (set! adjective 'yes)
          #f)
      (if (member (list adjective noun) newWM)
          (DisplayLine "Heureka ! I suspect that it"
		       "probably does have"
		       (if (eq? adjective 'yes) "" adjective)
		       noun)
          (DisplayLine "Sorry, probably not, but I am not sure!")))))

; Top Level function which enters a loop to read in a sentence
; and call top level ATN to parse it then process the sentence

(define Rodney
  ; a polite version of Rodney,  the restaurant advisory program
  (lambda ()
    (DisplayLine "Hello, my name is Rodney")
    (DisplayLine "and it is my pleasure to be"
                 "your culinary advisor today")
    (DisplayLine "How can I be of service ?")
    (DisplayLine "(please give your requests or" 		
                 "assertions as a list !)")
    (do ((finished? (ReadAndProcessASentence)
                    (ReadAndProcessASentence)))
        (finished?
         (DisplayLine "Goodbye, thank you for coming")))))

