;  J A N E S G A R D E N  - section 3.7.3
;  = = = = = = = = = = =
;
; Natural language interface to a simple GardenWorld
; where a person can be directed to move about the
; world and perform simple actions using sentence
; in a subset of english

; LANGUAGE  SECTION
; ========  =======


; Define the categories

(define article    (MakeCategory 'a 'the))
(define objectNoun (MakeCategory 'tree 'flower 'pond 'gnome
				 'flowerbed 'wateringcan 'water))
(define dirNoun    (MakeCategory 'north 'south 'east 'west))
(define dirPrep    (MakeCategory 'near 'beside 'in 'from 'at 'by 'to))
(define dirAdverb  (MakeCategory 'up 'down 'left 'right))
(define toVerb     (MakeCategory 'to 'towards))
(define goVerb     (MakeCategory 'walk 'go 'head))
(define pickVerb   (MakeCategory 'pick 'grab 'get 'fill))
(define putVerb    (MakeCategory 'put 'drop))

; The registers used are as follows:
;
; Register:     Possible values:
; --------      ---------------
;  action         go put pick show bye
;  object         an 'objectnoun'
;  where          a list of locations each being either
;                 a 'dirAdverb' or a 'dirPrp' followed by
;                 an 'objectNoun'
;
; Thus after the sentence:
;   (pick up the flower near the tree by the pond)
; the state of the registers will be:
;   action = pick
;   object = flower
;   where  = (near tree by pond)
;
; There is also a global variable LastObject that contains the
; last object manipulated so it can be referred to by "it"

(define LastObject #f)
(define saveLast!
  (lambda (aRegisterBank)
    (set! LastObject (GetRegister aRegisterBank '**))
    LastObject))
(define validLast?
  (lambda (aRegisterBank) (if (not LastObject) #f #t)))
(define getLast
  (lambda (aRegisterBank) LastObject))

(define addWhere
  (lambda (aRegisterBank)
    ; prepend the current value of register '**' to
    ; the register 'where'
    (cons (GetRegister aRegisterBank '**)
          (GetRegister aRegisterBank 'where))))

; Define the ATN's

; Noun phrases:
; ============

(define NP
  (MakeATN
    (STATE 'N1 (CAT article #t (TO 'N2)))
    (STATE 'N2 (CAT objectNoun #t (SETR 'object saveLast!)
                                  (TO 'N3)))
    (STATE 'N3 (POP (GETR 'object) #t))))

(define DNP
  (MakeATN
    (STATE 'D1 (CAT article #t (TO 'D2)))
    (STATE 'D2 (CAT objectNoun #t (SETR 'where addWhere)
                                  (TO 'D3))
               (CAT dirNoun    #t (SETR 'where addWhere)
                                  (TO 'D3)))
    (STATE 'D3 (POP ** #t))))

; Picking and putting
; ===================

(define PickPut
  (MakeATN
    (STATE 'P1 (CAT pickVerb #t (SETR 'action **)
                                (SETR 'where  ())
                                (TO 'P2))
               (CAT putVerb  #t (SETR 'action **)
                                (SETR 'where  ())
                                (TO 'P6)))
    (STATE 'P2 (CAT (MakeCategory 'up) #t (TO 'P3))
               (PUSH NP      #t (SETR 'where addWhere)
                                (TO 'P4)))
    (STATE 'P3 (PUSH NP      #t (SETR 'where addWhere)
                                (TO 'P4)))
    (STATE 'P4 (CAT dirPrep  #t (SETR 'where addWhere)
                                (TO 'P5))
               (POP **       #t))
    (STATE 'P5 (PUSH DNP     #t (TO 'P4)))
    (STATE 'P6 (CAT (MakeCategory 'down) #t (TO 'P8))
               (CAT (MakeCategory 'it) validLast?
		                (SETR 'object getLast)
                                (TO 'P7))
               (CAT (MakeCategory 'it) validLast?
		                (SETR 'object getLast)
                                (TO 'P9))
               (PUSH NP      #t (TO 'P7)))
    (STATE 'P7 (CAT (MakeCategory 'down) #t (TO 'P9)))
    (STATE 'P8 (PUSH NP      #t (TO 'P9)))
    (STATE 'P9 (POP **       #t))))

; Going
; =====

(define Go
  (MakeATN
    (STATE 'G1 (CAT goVerb    #t (SETR 'action **)
                                 (SETR 'where  ())
                                 (TO 'G2)))
    (STATE 'G2 (CAT toVerb    #t (TO 'G3))
               (CAT dirAdverb #t (SETR 'where addWhere)
                                 (TO 'G4)))
    (STATE 'G3 (PUSH DNP      #t (TO 'G4)))
    (STATE 'G4 (CAT dirPrep   #t (SETR 'where addWhere)
                                 (TO 'G3))
               (POP **        #t))))

; Full sentence structure
; =======================

(define GardenWorldSentence
  (MakeATN
    (STATE 'W1 (CAT (MakeCategory 'show) #t (SETR 'action **)
                                            (TO 'W2))
               (CAT (MakeCategory 'bye)  #t (SETR 'action **)
                                            (TO 'W2))
               (PUSH PickPut #t (TO 'W2))
               (PUSH Go      #t (TO 'W2)))
    (STATE 'W2
               (POP **       #t))))

; WORLD  SECTION
; =====  =======

; Symbols used to represent objects: use "W" for full wateringcan
; and "w" for an empty one. Its state is stored in a global
; variable 'FullWateringCan?'

(define FullWateringCan? #f)

(define Symbols '((tree        "T")
                  (flower      "F")
                  (flowerbed   "B")
                  (wateringcan "W")
                  (gnome       "G")
                  (pond        "P")
                  (empty       ".")))
                  
(define MakePos (lambda (x y) (list x y)))
(define GetX    (lambda (pos) (car pos)))
(define GetY    (lambda (pos) (cadr pos)))

(define GardenWorldGrid #f)
(define MaxX 8)
(define MaxY 8)

(define GetXY
  (lambda (pos)
    (vector-ref (vector-ref GardenWorldGrid (GetY pos))
                (GetX pos))))

(define SetXY!
  (lambda (pos object)
    (vector-set! (vector-ref GardenWorldGrid (GetY pos))
                 (GetX pos)
                 object)))

(define InitGardenWorld
  (lambda ()
    (set! GardenWorldGrid (make-vector MaxY))
    (do ((y 0 (+ y 1))
         (newrow #f))
        ((>= y MaxY) #f)
        (set! newrow (make-vector MaxX))
        (do ((x 0 (+ x 1)))
            ((>= x MaxX) #f)
            (vector-set! newrow x 'empty))
        (vector-set! GardenWorldGrid y newrow))
    (SetXY! '(3 1) 'flower)
    (SetXY! '(5 1) 'gnome)
    (SetXY! '(6 2) 'pond)
    (SetXY! '(4 3) 'wateringcan)
    (SetXY! '(7 7) 'flowerbed)
    (SetXY! '(4 5) 'tree)
    (set! FullWateringCan? #f) ; initially empty
    #f))

(define JanesPos (MakePos 1 1))
(define JanesBag ()) ; initially empty

(define DisplayWorld
  (lambda ()
    (newline)
    (do ((y 0 (+ y 1)))
        ((>= y MaxY) #f)
        (do ((x 0 (+ x 1))
             (pos #f)
             (obj #f))
            ((>= x MaxX) #f)
            (set! pos (MakePos x y))
            (set! obj (GetXY pos))
            (cond ((equal? pos JanesPos) (display "J"))
                  ((equal? obj 'wateringcan)
                   (if FullWateringCan?
                       (display "W")
                       (display "w")))
                  (else (display (cadr (assoc obj Symbols))))))
        (newline))))

; Moving around the world - it "wraps" around at the ends

(define mod
  (lambda (x y)
    (let ((r (remainder x y)))
      (if (negative? r) (+ r y) r))))
(define +xy
  (lambda (pos nx ny)
    ; add nx to x and ny to y of pos
    (MakePos (mod (+ nx (GetX pos)) MaxX)
             (mod (+ ny (GetY pos)) MaxY))))

(define FindObject
  (lambda (object currPos range)
    ; find nearest example of object to currPos that is in range
    (do ((r 0 (+ r 1))
         (newpos #f)
         (found #f))
        ((or found (> r range)) found)
        (do ((y (- 0 r) (+ y 1)))
            ((or found (> y r)) #f)
            (do ((x (- 0 r) (+ x 1)))
                ((or found (> x r)) #f)
                (set! newpos (+xy currPos x y))
                (if (or (eq? object (GetXY newpos))
                        (and (eq? object 'flower)
                             (eq? (GetXY newpos) 'flowerbed))
                        (and (eq? object 'water)
                             (eq? (GetXY newpos) 'pond)))
                    (set! found newpos)
                    #f))))))

; definitions of the ranges

(define eyesight 3)
(define ranges '((near     3)
                 (by       2)
                 (beside   1)
                 (in       0)
                 (at       0)
                 (from     0)
                 (to       0)))

(define MoveJane
  (lambda (where)
    ; try and move Jane according to the instructions in the "where" register
    
    (define moveJane
      (lambda (newPos)
        ; move from JanesPos to newPos
        (set!  JanesPos newPos)))

    (define moveAux
      (lambda (whereList)
        ; recursively move Jane
        (if (null? whereList)
            #f
            (let ((head (car whereList))
                  (tail (cdr whereList))
                  (newPos #f)
                  (range eyesight))
              (if (member head dirPrep)
                  (begin
                   (set! range (cadr (assoc head ranges)))
                   (set! head (car tail))
                   (set! tail (cdr tail)))
                  #f)
	      ; as some versions do not have a ``case'' the following code
	      ; from page 367 of the book has been rewtitten
              ; (case head
              ;   ((north up)   (moveJane (+xy JanesPos 0 -1))
              ;                 (moveAux tail))
              ;   ((south down) (moveJane (+xy JanesPos 0 +1))
              ;                 (moveAux tail))
              ;   ((west left)  (moveJane (+xy JanesPos -1 0))
              ;                 (moveAux tail))
              ;   ((east right) (moveJane (+xy JanesPos +1 0))
              ;                 (moveAux tail))
              ;   ((tree flower pond gnome flowerbed
              ;     wateringcan water)
              ;                      (set! newPos
              ;                            (FindObject head JanesPos range))
              ;                      (if (null? newPos)
              ;                          (begin (display "Can't find ")
              ;                                 (display head))
              ;                          (begin (moveJane newPos)
              ;                                 (moveAux tail))))
              ;   (else (display "Error in where list")))))))
              (cond ((member head '(north up))
		     (moveJane (+xy JanesPos 0 -1)) (moveAux tail))
		    ((member head '(south down))
		     (moveJane (+xy JanesPos 0 +1)) (moveAux tail))
		    ((member head '(west left))
		     (moveJane (+xy JanesPos -1 0)) (moveAux tail))
		    ((member head '(east right))
		     (moveJane (+xy JanesPos +1 0)) (moveAux tail))
		    ((member head '(tree flower pond gnome flowerbed
				    wateringcan water))
		     (set! newPos (FindObject head JanesPos range))
		     (if (null? newPos) 
			 (begin (display "Can't find ") (display head))
			 (begin (moveJane newPos) (moveAux tail))))
		    (else (display "Error in where list")))))))
    
    (let ((oldPos JanesPos))
      (moveAux where)
      (if (equal? oldPos JanesPos)
          #f ; hasn't moved so don't redisplay
          (DisplayWorld)))))

(define UpdateWorld
  (lambda (aRegisterBank)
    ; using the information in the registers - update the world
    
    (define remove
      (lambda (object bag)
        ; remove object from bag and return lighter bag
        (cond ((null? bag) #f)
              ((equal? object (car bag)) (cdr bag))
              (else (cons (car bag) (remove object (cdr bag)))))))
    
    (define doPut
      (lambda (object)
        ; put object at current position but only if empty or
        ; the object there is a pond or try to put a flower
        ; in a flowerbed or pouring water
        (cond ((not (member object JanesBag))
               (display "Jane doesn't have it"))
              ((equal? 'pond (GetXY JanesPos))
               (display "Splash!")
               (set! JanesBag (remove object JanesBag)))
              ((and (equal? 'flowerbed (GetXY JanesPos))
                    (equal? object 'flower))
               (display "Planted flower")
               (set! JanesBag (remove object JanesBag)))
              ((equal? object 'water)
               (display "Watered it")
               (set! JanesBag (remove object JanesBag))
               (set! FullWateringCan? #f))
              ((not (equal? 'empty (GetXY JanesPos)))
               (display "There is already something there"))
              (else
               (SetXY! JanesPos object)
               (set! JanesBag (remove object JanesBag))
               (if (and (equal? object 'wateringcan)
                        FullWateringCan?)
                   (set! JanesBag (remove 'water JanesBag))
                   #f)))))
        
    (define doGet
      (lambda (object)
        ; get object from current position but not immovable
        ; objects and only flowers from flowerbeds or fill
        ; watering can if have it and at pond
        (let ((currObject (GetXY JanesPos)))
          (cond ((member object '(tree pond flowerbed))
                 (display "Can't take that object"))
                ((and (equal? object 'flower)
                      (equal? currObject 'flowerbed))
                 (set! JanesBag (cons object JanesBag)))
                ((and (equal? object 'water)
                     (member 'wateringcan JanesBag)
                     (equal? currObject 'pond))
                 (set! FullWateringCan? #t)
                 (set! JanesBag (cons object JanesBag)))
                ((equal? currObject object)
                 (set! JanesBag (cons object JanesBag))
                 (if (and (equal? object 'wateringcan) FullWateringCan?)
                     (set! JanesBag (cons 'water JanesBag))
                     #f)
                 (SetXY! JanesPos 'empty))
                (else
                 (display "There is no such object here"))))))
    
    (let ((action #f)
          (object #f)
          (where  #f))
      (set! action (GetRegister aRegisterBank 'action))
      ; (case action
      ;   ((show)
      ;    (DisplayLine "Contents of Jane's bag: " JanesBag))
      ;   ((bye)
      ;    (DisplayLine "Thanks for visiting"))
      ;   ((walk go head)
      ;    (set! where (GetRegister aRegisterBank 'where))
      ;    (MoveJane where))
      ;   ((put drop)
      ;    (set! where (GetRegister aRegisterBank 'where))
      ;    (MoveJane where)
      ;    (doPut (GetRegister aRegisterBank 'object)))
      ;   ((pick grab get fill)
      ;    (set! where (GetRegister aRegisterBank 'where))
      ;    (MoveJane where)
      ;    (doGet (GetRegister aRegisterBank 'object)))
      ;   (else (display "illegal action"))))))
      (cond ((eq? action 'show)
	     (DisplayLine "Contents of Jane's bag: " JanesBag))
	    ((eq? action 'bye)
	     (DisplayLine "Thanks for visiting"))
	    ((member action '(walk go head))
	     (set! where (GetRegister aRegisterBank 'where))
	     (MoveJane where))
	    ((member action '(put drop))
	     (set! where (GetRegister aRegisterBank 'where))
	     (MoveJane where)
	     (doPut (GetRegister aRegisterBank 'object)))
	    ((member action '(pick grab get fill))
	     (set! where (GetRegister aRegisterBank 'where))
	     (MoveJane where)
	     (doGet (GetRegister aRegisterBank 'object)))
	    (else (display "illegal action"))))))

(define Converse
  (lambda ()
    ; hold a conversation updating world
    (InitGardenWorld)
    (DisplayWorld)
    (do ((sentence #f)
         (result   #f)
         (done     #f))
        ((equal? sentence '(bye)) #f)
        (newline)
        (display "..> ")
        (set! sentence (read))
        (newline)
        (if (not (pair? sentence))
            (display "Please give your command as a list")
            (begin
             (set! result (Parse GardenWorldSentence sentence #f))
             (if (not result)
                 (display "Sorry, I didn't understand that")
                 (UpdateWorld result)))))))

(DisplayLine "Type (Converse) to manipulate Garden World")
