
% OL(P): Object Layer for Prolog -- runtime system
% Version 1.1 for SICStus Prolog and QUINTUS Prolog
% (C) 1993 Markus P.J. Fromherz.  All Rights Reserved.
% (C) 1993 Xerox Corporation.     All Rights Reserved.

% This file together with the ol_resource and project ol_project is
% needed to run OL(P) programs independently.

% operators for the object-oriented statements

:- op(1002,  fx, [(object),(end_object),(publish),(override),(unfold)]).
:- op(1001, xfy, (is_a)).
:- op(999,  xfy, (in)).
:- op(550,  xfy, '::').
:- op(550,   fy, '::').
:- op(650,   fx, '`').
:- op(700,  xfx, ':=').
:- op(180,  xfy, '.').
:- op(180,   fy, '.').

O::M :-
   ol_goal(O::M, Goal),
   Goal.

ol_goal(O::M, Goal) :-
   nonvar(O), nonvar(M),
   ( atom(O), ON = O ; O =.. [ON,_] ),
   ol_object(ON, _, _),
   !,
   ol_compile_body(O::M, pl, prolog, prolog, _, G),
   ( ol_state(ON, autonomous) ->
        ol_autonomous_trafo(G, ON, Goal)
      ;
        Goal = G
   ).
ol_goal(O::M, _, _) :-
   ( var(O) -> raise_exception(instantiation_error(O::M,1)) ;
     var(M) -> raise_exception(instantiation_error(O::M,2)) ;
     \+ (( atom(O), ON = O ; O =.. [ON,_] ),ol_object(ON,_,_)) ->
               raise_exception(existence_error(O::M,1,(object),O,_))
   ).

ol_call(V, This, Self, Inst, BL) :-
   nonvar(V), V=O::M, nonvar(O), nonvar(M), ol_object(O, _, _),
   !,
   ol_replace_pv(V, This, Self, RV),
   ol_compile_body(RV, BL, This, Self, Inst, G),
   ( ol_state(O, autonomous) ->
        ol_autonomous_trafo(G, O, Goal),
        Goal
      ;
        G
   ).
ol_call(V, _, _, _, _) :-
   ( var(V) -> raise_exception(instantiation_error(V,0)) ;
     V=O::M ->
        ( var(O) -> raise_exception(instantiation_error(O::M,1)) ;
          var(M) -> raise_exception(instantiation_error(O::M,2)) ;
          \+ ol_object(O,_,_) -> raise_exception(existence_error(O::M,1,(object),O,_))
        ) ;
        raise_exception(domain_error(V,0,(object)::goal,V))
   ).


% ol_compile_body(+Goals, +BL, +This, ?Self, ?Instance, -CompiledGoals) -- CompiledGoals is
%    the compiled version of Goals in object This and context variable Self
%    for base language BL; needs ol_resource
ol_compile_body(V, BL, This, Self, Inst, ol_call(V,This,Self,Inst,BL)) :-
   var(V),
   !.
ol_compile_body(`Goals, _, _, _, _, Goals) :-
   !.
ol_compile_body(O::V, BL, This, Self, Inst, CG) :-
   var(V),
   ol_compile_goal(O::V, BL, This, Self, Inst, CG1),
   ol_replace(CG1, true, true, CG),
   !.
ol_compile_body(::V, BL, This, Self, Inst, CG) :-
   var(V),
   ol_compile_goal(::V, BL, This, Self, Inst, CG1),
   ol_replace(CG1, true, true, CG),
   !.
ol_compile_body(Goals, BL, This, Self, Inst, CompiledGoals) :-
   % ol_compile_goals/6 deals with goal terms (_,_), ::(_,_), O::(_,_) etc.
   ol_compile_goals(Goals, BL, This, Self, Inst, CompiledGoals),
   !.
ol_compile_body(G, BL, This, Self, Inst, CG) :-
   ol_compile_goal(G, BL, This, Self, Inst, CG1),
   ol_replace(CG1, true, true, CG),
   !.

% ol_compile_goal(+OLGoal, +BaseLanguage, +This, ?Self, ?Instance, -Goal)
% *** var(O) ; var(M)
ol_compile_goal(O::M, BL, This, Self, Inst, G) :-
   O==Self,
   ol_compile_goal(::M, BL, This, Self, Inst, G).
ol_compile_goal(O::M, _, This, _, _, ol_call(O,M,This)) :-
   var(O) ; var(M).
 % ol_interface_head(O, This, loc, _, M, InterfaceGoal).
ol_compile_goal(::M, _, This, Self, Inst, ol_send(M,rsd(Self,This,loc,Inst))) :-
   var(M).
 % ol_interface_head(O, This, loc, Inst, M, InterfaceGoal).
% *** pseudo-vars prolog, super
ol_compile_goal(prolog::M, _, _, _, _, M).
ol_compile_goal(super::M, _, This, _, Inst, ol_send(M,rsd(This,This,inh,Inst))).
 % ol_interface_head(This, This, inh, Inst, M, InterfaceGoal).
% *** new/[1,2]
ol_compile_goal(O::new(I), _, _, _, _, ol_new(O,I)).
ol_compile_goal(::new(I), _, _, Self, _, ol_new(Self,I)).
ol_compile_goal(O::new(As,I), _, _, _, _, ol_new(O,As,I)).
ol_compile_goal(::new(As,I), _, _, Self, _, ol_new(Self,As,I)).
% *** rest of O::M and ::M
ol_compile_goal(O::M, _, This, _, _, ol_send(M,rsd(ON,This,loc,Inst))) :-
   atom(O), ON = O ; O =.. [ON,Inst].
 % ol_interface_head(ON, This, loc, Inst, M, InterfaceGoal).
ol_compile_goal(::M, _, This, Self, Inst, ol_send(M,rsd(Self,This,loc,Inst))).
 % ol_interface_head(Self, This, loc, Inst, M, InterfaceGoal).
ol_compile_goal(call(O::M), BL, This, Self, Inst, ol_call(O::M,This,Self,Inst,BL)).
ol_compile_goal(call(::M), BL, This, Self, Inst, ol_call(::M,This,Self,Inst,BL)).
% *** I := J.As
ol_compile_goal(A := B, _, _, _, _, A:=B) :-
   var(B).
ol_compile_goal(I := J.As, _, _, _, _, ol_assign(J,As,I)).
ol_compile_goal(I := .As, _, _, _, Inst, ol_assign(Inst,As,I)).
% *** I.A = V, I.A == V
ol_compile_goal(A = B, _, _, _, _, A=B) :-
   var(A).
ol_compile_goal(A == B, _, _, _, _, A==B) :-
   var(A).
ol_compile_goal(A \== B, _, _, _, _, A\==B) :-
   var(A).
ol_compile_goal(I.A = V, _, _, _, _, ol_val_e(I,A,V)).
ol_compile_goal(.A = V, _, _, _, Inst, ol_val_e(Inst,A,V)).
ol_compile_goal(I.A == V, _, _, _, _, ol_val_eq(I,A,V)).
ol_compile_goal(.A == V, _, _, _, Inst, ol_val_eq(Inst,A,V)).
ol_compile_goal(I.A \== V, _, _, _, _, ol_val_neq(I,A,V)).
ol_compile_goal(.A \== V, _, _, _, Inst, ol_val_neq(Inst,A,V)).

ol_compile_goal(G, _, _, _, _, G).

% ol_interface_head(Receiver, Sender, Def, Instance, MethodGoal, InterfaceHead)
%    this goal is unfolded in ol_compile_goal/5 and portray/1 for efficiency
ol_interface_head(Receiver, Sender, Def, Inst, MethodGoal,
                  ol_send(MethodGoal, rsd(Receiver,Sender,Def,Inst))).


% terms

% ol_replace_pv(Term, This, Self, RTerm) -- RTerm is Term with atoms this and
%    self replaced by This and Self, respectively
ol_replace_pv(V, _, _, V) :-
   var(V),
   !.
ol_replace_pv(`T, _, _, `T) :-
   !.
ol_replace_pv(T, This, Self, TT) :-
   atomic(T),
   !,
   ( T == this -> TT = This ;
     T == self -> TT = Self ;
                  TT = T
   ).
ol_replace_pv(T, This, Self, TT) :-
   functor(T, N, A),
   ol_replace_pv(N, This, Self, TN),
   ( var(TN) ->
        T =.. [N|Ps],
        ol_replace_pv_args(2, Ps, This, Self, TPs),
        TT =.. [TN|TPs]
      ;
        functor(TT, TN, A),
        ol_replace_pv_args(A, T, This, Self, TT)
   ).

ol_replace_pv_args(0, _, _, _, _) :-
   !.
ol_replace_pv_args(A, T, This, Self, TT) :-
   arg(A, T, Arg),
   ol_replace_pv(Arg, This, Self, TArg),
   arg(A, TT, TArg),
   A1 is A-1,
   ol_replace_pv_args(A1, T, This, Self, TT).

% ol_replace(Term, Template, Replacement, RTerm) -- RTerm is Term with Template
%    replaced by Replacement
ol_replace(V, _, _, V) :-
   var(V),
   !.
ol_replace(`T, _, _, T) :-
   !.
ol_replace(T, Template, Replacement, TT) :-
   atomic(T),
   !,
   ( T == Template -> TT = Replacement ; TT = T ).
ol_replace(T, Template, Replacement, TT) :-
   functor(T, N, A),
   ol_replace(N, Template, Replacement, TN),
   ( var(TN) ->
        T =.. [N|Ps],
        ol_replace_args(2, Ps, Template, Replacement, TPs),
        TT =.. [TN|TPs]
      ; functor(TT, TN, A),
        ol_replace_args(A, T, Template, Replacement, TT)
   ).

ol_replace_args(0, _, _, _, _) :-
   !.
ol_replace_args(A, T, Template, Replacement, TT) :-
   arg(A, T, Arg),
   ol_replace(Arg, Template, Replacement, TArg),
   arg(A, TT, TArg),
   A1 is A-1,
   ol_replace_args(A1, T, Template, Replacement, TT).


ol_list(L) :-
   L == [].
ol_list(L) :-
   nonvar(L), L = [_|Xs],
   nonvar(Xs), ol_list(Xs).

ol_append([], Ys, Ys).
ol_append([X|Xs], Ys, [X|Zs]) :-
   ol_append(Xs, Ys, Zs).

% instances

ol_attribute(I, A) :-
   ol_attribute(I, [], A).

ol_attribute([A=_|_], As, A) :-
   \+ ol_member(A, As).   % replaced value, name already returned
ol_attribute([B=_|I], As, A) :-
   ol_attribute(I, [B|As], A).

% ol_val(I, A, V)
ol_val([A=W|_], A, V) :-
   !,
   W = V.
ol_val([_|As], A, V) :-
   ol_val(As, A, V).

/* with position instead of list:
ol_val(Inst, A, V) :-
   functor(Inst, _, N),
   arg(N, Inst, Ps),
   ol_member((A,P), Ps),
   !,
   arg(P, Inst, V).

ol_replace_attributes([(A,P)|Ps], As, J, I) :-
   ol_member(A=V, As),
   !,
   arg(P, I, V),
   ol_replace_attributes(Ps, As, J, I).
ol_replace_attributes([(_,P)|Ps], _, J, I) :-
   arg(P, J, V),
   arg(P, I, V),
   ol_replace_attributes(Ps, As, J, I).
*/
