/*
 * PCN System -- PCN Object file access routines
 * Author:      Steve Tuecke
 *              Argonne National Laboratory
 *
 * Please see the DISCLAIMER file in the top level directory of the
 * distribution regarding the provisions under which this software
 * is distributed.
 *
 * pcno.h
 *
 * This contains header information for the PCN Object (PCN-O) file
 * format.
 *
 * The internal PCN-O format is described in pcno.pcn.
 *
 * This header can be included in either C or PCN programs.
 *
 * It contains the C structure declarations of parts of the PCN-O file,
 * as well as all PCN-O related #defines used by both C and PCN code.
 *
 * See the pcno.pcn modules for more information on the accessing
 * PCN-O files from PCN.
 *
 * A few general comments regarding the external PCN-O file format:
 *   - All integer values in the various structures are always stored
 *	in big endian format on disk.  If a machine uses little
 *	endian, then it needs to do conversion as it reads the
 * 	PCN-O file in.
 *   - The "u_int_t" type is assumed to be 32 bits.
 *   - Unless otherwised stated, all sizes and offsets are
 *	in bytes, and must be padded out to word (4 byte)
 *	bounderies.
 *   - All sizes in the various block headers described below
 *	include the size of the header.
 *
 * A few general comments regarding the internal PCN-O structure:
 *   - The internal PCN-O structure is just nested tuples, where
 *	the first tuple argument gives enough information to
 *	uniquely identify that tuple level.  For example, if
 *	the first tuple argument is PCNO_LAZY_DATA, this signifies
 *	this tuple level must be retrieved explicitly.  And all
 *	other possible identifiers must be unique with respect
 *	to PCNO_LAZY_DATA and to other identifiers.
 */

#ifndef _PCN_INCLUDE_PCNO_H
#define _PCN_INCLUDE_PCNO_H

#include "pcn_types.h"
#ifndef PCN
#include "pcn_structs.h"
#endif

/*
 * It is assumed that all words in the PCN-O files (u_int_t) are
 * 4 bytes, and that all integers are big endian.  If this is
 * not true, then you need to do conversions when reading words
 * from the PCN-O file into the C structures defined below.
 */
#define PCNO_WORD_SIZE	4

#ifndef TRUE
#define TRUE 1
#define FALSE 0
#endif

/*
 * A PCN-O file contains a header followed by zero or more segments.
 * The header has the following format.
 */
#ifndef PCN
typedef struct pcno_header_struct
{
    u_int_t		magic;		/* PCN-O magic number */
    u_int_t		size;		/* The file's size (with header) */
    u_int_t		n_segments;	/* The number of segments */
    u_int_t		version;	/* PCN-O version number */
} pcno_header_t;
#endif

#define PCNO_HEADER_SIZE		4

#define PCNO_MAGIC	42424242	/* The PCN-O magic number */
#define PCNO_VERSION	3		/* The PCN-O version number */

/*
 * All segments, sections and subsections described below (we'll call
 * them all "blocks") have a standard header format.  All block
 * specific header information comes after this basic header.
 */
#ifndef PCN
typedef struct pcno_block_header_struct
{
    u_int_t		type;		/* This block's type */
    u_int_t		size;		/* This block's size, including
					 * this header */
} pcno_block_header_t;
#endif

#define PCNO_BLOCK_HDR_SIZE		2


/*
 * A block with a list of strings is used in several places in
 * the PCN-O format -- PCN call table and foreign call table, for example.
 * Sometimes these strings are meant to be used in groups.  The
 * 'group_size' fields tells the intended grouping size.
 *
 * These string list blocks have the following format:
 */
#ifndef PCN
typedef struct pcno_string_list_struct
{
    u_int_t		type;		/* The section type */
    u_int_t		size;		/* The section size */
    u_int_t		n_entries;	/* Number of strings */
    u_int_t		group_size;	/* Grouping size of strings */
} pcno_string_list_t;
#endif

#define PCNO_STRING_LIST_HDR_SIZE	4

/*
 * After the header comes a series of 'n_entries' entries, each
 * with the following form.
 *
 * Note: The 'size' fields does NOT include itself.  It is only
 * the size (in bytes) of the entry itself.
 */
#ifndef PCN    
typedef struct pcno_string_struct
{
    u_int_t		size;		/* Size of this entry */
    char_t		*entry;		/* The exported procedure name */
} pcno_string_t;
#endif



/*
 * Each segment must be one of the following types.
 */
#define PCNO_SEG_MODULE	1		/* A module segment */
#define PCNO_SEG_NAMED	2		/* An arbitrary named segment */

/*
 * A named segment (PCNO_SEG_NAMED) has the following header.
 */
#ifndef PCN
typedef struct pcno_named_segment_struct
{
    u_int_t		type;		/* The segment type */
    u_int_t		size;		/* The segment size */
    u_int_t		data_offset;	/* Offset to data */
    u_int_t		data_size;	/* Size of the data */
    char_t		*name;		/* Name of segment */
} pcno_named_segment_t;
#endif

#define PCNO_SEG_NAMED_HDR_SIZE		4

/*
 * A module segment (PCNO_SEG_MODULE) consists of the a header
 * followed by zero or more module sections (each beginning with the
 * usual block header, pcno_block_header_t).
 *
 * A module segment has the following header.
 */
#ifndef PCN
typedef struct pcno_module_segment_struct
{
    u_int_t		type;		/* The segment type */
    u_int_t		size;		/* The segment size */
    u_int_t		n_sections;	/* Number of sections in module */
    u_int_t		section_offset;	/* Offset (from start of header)
					 * to first section. */
    char_t		*name;		/* The module's name */
} pcno_module_segment_t;
#endif

#define PCNO_SEG_MODULE_HDR_SIZE	4


/*    
 * Valid module section types are:
 */
#define PCNO_MODSECT_PROCEDURE	101	/* A procedure section */
#define PCNO_MODSECT_METACALLS	201	/* A metacalls table section */


/*
 * A procedure section (PCNO_MODSECT_PROCEDURE) contains a header
 * followed by 1 or more subsections.  The procedure header has the
 * structure:
 */
#ifndef PCN
typedef struct pcno_procedure_struct
{
    u_int_t		type;		/* The segment type */
    u_int_t		size;		/* The segment size */
    u_int_t		n_subsects;	/* Number of subsections */
    u_int_t		subsect_offset;	/* Offset (from start of header)
					 * to first subsection. */
    u_int_t		exported;	/* Is the procedure exported? */
    u_int_t		n_counters;	/* Number of counter */
    u_int_t		n_timers;	/* Number of timers */
    u_int_t		arity;		/* The procedure's arity */
    char_t		*name;		/* Procedure name */
} pcno_procedure_t;
#endif

#define PCNO_MODSECT_PROCEDURE_HDR_SIZE	8

/*
 * The valid values for 'exported' is 0 (for FALSE) and 1 (for TRUE).
 */


/*
 * The valid procedure subsections types are:
 */
#define PCNO_PROC_CODE			10101	/* Code */
#define PCNO_PROC_PCALL_TABLE		20101	/* PCN call table */
#define PCNO_PROC_METACALL_TABLE	30101	/* Metacall table */
#define PCNO_PROC_FCALL_TABLE		40101	/* Foreign call table */
#define	PCNO_PROC_DOUBLE_TABLE		50101	/* Double constant table */
#define	PCNO_PROC_STRING_TABLE		60101	/* String constant table */
#define	PCNO_PROC_INT_TABLE		70101	/* Integer constant table */
#define	PCNO_PROC_MODEL			80101	/* Gauge model */


/*
 * The procedure's code subsection (PCNO_PROC_CODE)
 * contains a header followed by an array of u_int_t's which
 * represent the PCN Abstract Machine code.
 */
#ifndef PCN
typedef struct pcno_code_struct
{
    u_int_t		type;		/* The section type */
    u_int_t		size;		/* The section size */
    u_int_t    		code_size;	/* The code's size (in chars) */
    code_t		*code_array;	/* The procedure's code */
} pcno_code_t;
#endif

#define PCNO_PROC_CODE_HDR_SIZE		3

/*
 * The procedure's PCN call table subsection (PCNO_PROC_PCALL_TABLE)
 * is a string list block.  It is assumed that the strings
 * will always occur in pairs (group_size = 2), and
 * represent <module_name,procedure_name>
 * pairs.  A module_name of "" (the empty string) means that it
 * is a call within the module.
 */
#ifndef PCN
typedef pcno_string_list_t	pcno_pcall_table_t;
typedef pcno_string_t		pcno_pcall_t;
#endif


/*
 * The procedure's foreign call table subsection (PCNO_PROC_FCALL_TABLE)
 * is a string list block, where the strings are the names of
 * foreign procedures.  (group_size = 1)
 */
#ifndef PCN
typedef pcno_string_list_t	pcno_fcall_table_t;
typedef pcno_string_t		pcno_fcall_t;
#endif


/*
 * The procedure's double constant table subsection (PCNO_PROC_DOUBLE_TABLE)
 * is a string list block, where the strings are the ascii
 * representation of the double constants. (group_size = 1)
 */
#ifndef PCN
typedef pcno_string_list_t	pcno_double_table_t;
typedef pcno_string_t		pcno_double_t;
#endif


/*
 * The procedure's string constant table subsection (PCNO_PROC_STRING_TABLE)
 * is a string list block, where the strings are the ascii
 * representation of the string constants. (group_size = 1)
 */
#ifndef PCN
typedef pcno_string_list_t	pcno_string_table_t;
#endif


/*
 * The procedure's integer constant table subsection (PCNO_PROC_INT_TABLE)
 * is a string list block, where the strings are the ascii
 * representation of the integer constants. (group_size = 1)
 */
#ifndef PCN
typedef pcno_string_list_t	pcno_int_table_t;
typedef pcno_string_t		pcno_int_t;
#endif


/*
 * The procedure's Gauge model subsection (PCNO_PROC_MODEL)
 * is a string list block, where the strings are the ascii
 * representation of parts of the model. (group_size = 1)
 */
#ifndef PCN
typedef pcno_string_list_t	pcno_model_t;
typedef pcno_string_t		pcno_model_entry_t;
#endif


/*
 * The procedure's metacall list (PCNO_PROC_METACALLS), or
 * a modules metacall list (PCNO_MODSECT_METACALLS),
 * is a string list block.  The meta call list is a list
 * of all procedures that might be the target of metacalls
 * from this procedure, and therefore must be linked into
 * the final executable if this procedure is linked in.
 * It is assumed that the strings
 * will always occur in pairs (group_size = 2), and
 * represent <module_name,procedure_name>
 * pairs.  A module_name of "" (the empty string) means that
 * the meta call target is in the module.
 */
#ifndef PCN
typedef pcno_string_list_t	pcno_metacall_list_t;
typedef pcno_string_t		pcno_metacall_t;
#endif



/*
 * PCNO_COMPUTE
 *
 * When writing a block that contains a size or number
 * of entries field, PCNO_COMPUTE can be put into this
 * field to tell the system to compute the correct value
 * for this field and to fill it in automatically.
 */
#define PCNO_COMPUTE		-1


/*
 * PCNO_STRUCTURE
 *
 * This is the identifier for the entire internal
 * PCN-O structure.  The sub-tuples contain the
 * PCN-O file information.
 */
#define PCNO_STRUCTURE			11


/*
 * PCNO_BLOCK_HEADER
 *
 * Each tuple of the internal PCN-O structure contains
 * a block type (identifier) and a block header.  The
 * block header is a tuple that has an identifier
 * of PCNO_BLOCK_HEADER.
 */
#define PCNO_BLOCK_HEADER		12


/*
 * PCNO_LAZY_DATA
 *
 * The pcno access module retrieves data from the PCN-O file
 * in a lazy manner.  Only enough data is read in to differentiate
 * the block types.  The rest of the blocks are just "pointed to",
 * and must be explicitly retrieved.  PCNO_LAZY_DATA is used
 * to differentiate these "pointers" from the real data.  So it
 * must be unique with respect to all of the PCN-O block types.
 */
#define PCNO_LAZY_DATA			21

/*
 * Other defines used to identify lazy blocks
 */
#define PCNO_LAZY_SEGMENTS		121
#define PCNO_LAZY_MODULE_SECTIONS	221
#define PCNO_LAZY_PROC_SUBSECTS		321
#define PCNO_LAZY_CODE_ARRAY		421
#define PCNO_LAZY_LIST_ENTRIES		521
#define PCNO_LAZY_NAMED_DATA		621

#define PCNO_DATA			22

/*
 * PCNO_RETRIEVE_DATA
 * PCNO_DISCARD_DATA
 *
 * These definitions are used to communicate the type
 * of operation from retrieve_lazy_data() or discard_lazy_data()
 * to the lazy_data_handler.
 */
#define PCNO_RETRIEVE_DATA		31
#define PCNO_DISCARD_DATA		32

/*
 * PCNO_OK
 *
 * This is used when a status needs to be returned by a procedure
 * that does not return a pcno data structure.  For example,
 * discard_lazy_data() returns this if everything goes ok.
 */
#define PCNO_OK				0


/*
 * PCNO_ERROR
 *
 * When a read or retrieve error occurs in the pcno access routines,
 * this is used to differentiate the error from the real data.  So it
 * must be unique with respect to all of the PCN-O block types.
 */
#define PCNO_ERROR			99


/*
 * PCNO_ERROR_*
 *
 * Error codes and strings used by the pcno access routines.
 */
#define PCNO_ERROR_READ_OPEN		1
#define PCNO_ERROR_READ_OPEN_STRING	"Cannot open file for reading"
#define PCNO_ERROR_READ			2
#define PCNO_ERROR_READ_STRING		"Error while reading PCN-O file"
#define PCNO_ERROR_READ_STRUCTURE	3
#define PCNO_ERROR_READ_STRUCTURE_STRING	"Error in external PCN-O file format encountered during reading"

#define PCNO_ERROR_LAZY_TYPE		51
#define PCNO_ERROR_LAZY_TYPE_STRING	"Illegal lazy data block type passed to pcno:do_lazy_retrieve()"

#define PCNO_ERROR_WRITE_OPEN		101
#define PCNO_ERROR_WRITE_OPEN_STRING	"Cannot open file for writing"
#define PCNO_ERROR_WRITE_STRUCTURE	102
#define PCNO_ERROR_WRITE_STRUCTURE_STRING	"Error in internal PCN-O structure encountered during writing"
#define PCNO_ERROR_WRITE		103
#define PCNO_ERROR_WRITE_STRING		"Error while writing PCN-O file"


/*
 * Some debugging stuff
 */
#ifndef PCNO_DEBUG
#define PCNO_DEBUG 1
#endif 


#endif /* _PCN_INCLUDE_PCNO_H */
