/*
 * PCN Abstract Machine Emulator
 * Authors:     Steve Tuecke and Ian Foster
 *              Argonne National Laboratory
 *
 * Please see the DISCLAIMER file in the top level directory of the
 * distribution regarding the provisions under which this software
 * is distributed.
 *
 * runpcn.c	- A wrapper program for running PCN on an iPSC/860.
 *
 * This program is meant to be run on an iPSC/860 host.  It (optionally)
 * does a getcube, loads the nodes with an emulator, and then shuts it all
 * down cleanly.  This program allows simple use of a node only emulator
 * on the iPSC/860.
 *
 * There are two variables, 'default_name' and 'default_exe'.  If an
 * executable is not specified, then runpcn will try to use 'default_name'
 * first, and if that doesn't exist then it will use 'default_exe'.
 *
 * To use the program, do a 'runpcn -h', which will yield something like:
 
Usage: runpcn [-t <cubetype>] [<name>] [<args> ...]
        [-t <cubetype>] specifies the iPSC/860 cube type.  If this
                argument is used, runpcn will do a getcube before loading
                the cube with the PCN emulator.  <cubetype> is the same
                for the getcube program.  If this argument is left off
                it is assumed that a getcube has already been done.
        [<name>] is the name of the iPSC/860 node executable to use.
        [<args> ...] is any PCN arguments to use.

 *
 */

#include <stdio.h>
#include <signal.h>

#include "config.h"

char *default_name	= "pcn.ipsc860";
char *default_exe	= DEFAULT_EXE;

#ifndef TRUE
#define TRUE  1
#define FALSE 0
#endif

char run_buf[8192];
int did_getcube = FALSE;

void print_help();
void parse_rest();
void run_pcn();

mysystem(buf)
char *buf;
{
    int rc;
#ifdef TEST
    printf("Systeming off \"%s\"\n", buf);
#endif /* TEST */

    rc = system(buf);

#ifdef TEST
    printf("Return code for \"%s\" is %d\n", buf, rc);
#endif /* TEST */

    if (rc != 0)
    {
	printf("Error: \"%s\" failed (system() return code of %d)\n", buf, rc);
	if (did_getcube)
	{
	    printf("Killing cube\n");
	    system("killcube");
	    system("relcube");
	}
	exit(1);
    }
}


void int_handler(i)
int i;
{
#ifdef TEST    
    printf("In int_handler(%d)\n", i);
#endif /* TEST */
    printf("Killing cube\n");
    mysystem("killcube");
    if (did_getcube)
	mysystem("relcube");
    exit(1);
}

int main(argc, argv)
int argc;
char *argv[];
{
    signal(SIGINT, int_handler);

    if (argc == 1)
    {
	parse_rest(argc, argv);
    }

#ifdef RUNPCN    
    else if (strcmp(argv[1], "-h") == 0)
    {
	print_help();
    }
    
    else if (strncmp(argv[1], "-t", 2) == 0)	/* argument to getcube */
    {
	if (strlen(argv[1]) == 2)		/* -t <stuff> */
	{
	    if (argc < 3)
	    {
		print_help();
		exit(1);
	    }
	    did_getcube = TRUE;
	    sprintf(run_buf, "getcube -t %s", argv[2]);
	    mysystem(run_buf);
	    parse_rest(argc - 2, &argv[2]);
	    mysystem("relcube");
	}
	else					/* -t<stuff> */
	{
	    did_getcube = TRUE;
	    sprintf(run_buf, "getcube -t %s", argv[1] + 2);
	    mysystem(run_buf);
	    parse_rest(argc - 1, &argv[1]);
	    mysystem("relcube");
	}
    }
#endif /* RUNPCN */    

    else	/* No -t argument: getcube was done outside of program */
    {
	parse_rest(argc, argv);
    }
    return(0);
}

void parse_rest(argc, argv)
int argc;
char *argv[];
{
#ifdef RUNPCN    
    if (argc == 1 || argv[1][0] == '-')		/* use default name */
    {
#endif /* RUNPCN */
	if (access(default_name, 4) == 0)
	{
	    /* Use the node executable in this directory */
	    argv[0] = default_name;
	}
	else if (access(default_exe, 4) == 0)
	{
	    /* Use the default node executable */
	    argv[0] = default_exe;
	}
	else
	{
	    fprintf(stderr, "Cannot find: %s\n", default_name);
	    fprintf(stderr, "         or: %s\n", default_exe);
	    exit(1);
	}
	run_pcn(argc, argv);
#ifdef RUNPCN	
    }
    else		/* executable name supplied as an argument */
    {
	run_pcn(argc - 1, &argv[1]);
    }
#endif /* RUNPCN */    
}

void run_pcn(argc, argv)
int argc;
char *argv[];
{
    int i;

    run_buf[0] = '\0';

    strcat(run_buf, "load ");

    for (i = 0; i < argc - 1; i++)
    {
	strcat(run_buf, argv[i]);
	strcat(run_buf, " ");
    }
    strcat(run_buf, argv[argc-1]);
    strcat(run_buf, "; waitcube");

    printf("Doing load: \"%s\"\n", run_buf);
    mysystem(run_buf);
    
#ifdef TEST
    printf("Sleeping... ");
    fflush(stdout);
    sleep(5);
    printf("done\n");
    fflush(stdout);
#endif /* TEST */

    /*
    mysystem("waitcube");
    */
    printf("Killing cube\n");
    mysystem("killcube");
}


void print_help()
{
    printf("Usage: runpcn [-t <cubetype>] [<name>] [<args> ...]\n");
    printf("\t[-t <cubetype>] specifies the iPSC/860 cube type.  If this\n");
    printf("\t\targument is used, runpcn will do a getcube before loading\n");
    printf("\t\tthe cube with the PCN emulator.  <cubetype> is the same\n");
    printf("\t\tfor the getcube program.  If this argument is left off\n");
    printf("\t\tit is assumed that a getcube has already been done.\n");
    printf("\t[<name>] is the name of the iPSC/860 node executable to use.\n");
    printf("\t[<args> ...] is any PCN arguments to use.\n");
}

