%
% Copyright (C) 1990 by Ted Kim.
%
% This software work was developed at UCLA with support in part from
% DARPA Contract F29601-87-C-0072.
%

%
% util.pl
% X Window Interface for Prolog (XWIP)
% Utility Routines
%

%
% pxSelectValue(+Most, +Least, -Value)
%

pxSelectValue(MS, LS, xSplit(MS, LS)) :- MS > 0, !.
pxSelectValue(_, LS, LS).

%
% pxUnsignedPut1(+Value)
% pxUnsignedPut2(+Value)
%

% Type must match PX_SplitVector, PX_Integer
pxUnsignedPut1(xSplit(MS, LS)) :- !, pxUnsignedPut1(4, MS, LS, 0).
pxUnsignedPut1(X) :- pxUnsignedPut1(1, X, 0, 0).

pxUnsignedPut2(xSplit(MS, LS)) :- !, pxUnsignedPut2(4, MS, LS, 0).
pxUnsignedPut2(X) :- pxUnsignedPut2(1, X, 0, 0).

%
% pxSplitGetOne(-Value)
% pxSplitPutOne(+Value)
%

pxSplitGetOne(V) :- pxSplitGetOne(MS, LS, 0), pxSelectValue(MS, LS, V).

pxSplitPutOne(xSplit(MS, LS)) :- !, pxSplitPut(MS, LS, 0).
pxSplitPutOne(V) :- pxVectorPut(V, 0).

%
% pxVectorGet(+Next, -List)
% pxVectorGetN(+Number, -List)
% pxVectorPut(+List)
%

% Next must match PX_End, PX_SplitVector, PX_Cont
pxVectorGet(1, []).
pxVectorGet(4, [xSplit(MS, LS) | L]) :-
	pxSplitGet(MS, LS, N, 0), pxVectorGet(N, L).
pxVectorGet(0, [E | L]) :- pxVectorGet(E, N, 0), pxVectorGet(N, L).

pxVectorGetN(0, []) :- !.
pxVectorGetN(N, [E | L]) :-
	pxSplitGetOne(E), NewN is N - 1, pxVectorGetN(NewN, L).

pxVectorPut([]).
pxVectorPut([xSplit(MS, LS) | L]) :- pxSplitPut(MS, LS, 0), pxVectorPut(L).
pxVectorPut([ E | L]) :- pxVectorPut(E, 0), pxVectorPut(L).
pxVectorPut(_) :- !, pxVectorXFree, fail.

%
% pxStructGetN(+Name, +Length, -Struct)
% pxStructPutN(+Number, +Struct)
%

pxStructGetN(N, L, S) :-
	functor(S, N, L), E is L + 1, pxStructGetOne(1, E, S).

pxStructGetOne(E, E, _) :- !.
pxStructGetOne(C, L, S) :- 
	pxSplitGetOne(E), arg(C, S, E), NewC is C + 1,
	pxStructGetOne(NewC, L, S).

pxStructPutN(L, S) :-
	E is L + 1, pxStructPutOne(1, E, S).

pxStructPutOne(E, E, _) :- !.
pxStructPutOne(C, L, S) :-
	arg(C, S, E), pxSplitPutOne(E), NewC is C + 1,
	pxStructPutOne(NewC, L, S).

%
% pxStringGet(+Next, -List)
% pxStringPut(+List)
% pxStringLength(+StringList, -TotalElements, -TotalLength)
%

% Next must match PX_End, PX_Cont
pxStringGet(1, []) :- pxStringDealloc(0).
pxStringGet(0, [E | L]) :- pxStringGet(E, N, 0), pxStringGet(N, L).

pxStringPut([]).

pxStringPut([E | L]) :- pxStringPutOne(E), pxStringPut(L).
pxStringPut(_) :- !, pxVectorXFree, fail.

pxStringLength(SL, TE, TL) :- pxStringLength(SL, 0, TE, 0, TL).

pxStringLength([], TE, TE, TL, TL).
pxStringLength([E | L], TEA, TE, TLA, TL) :-
	NewTEA is TEA + 1,
	name(E, CharList), pxListLength(CharList, ELen), NewTLA is TLA + ELen, 
	pxStringLength(L, NewTEA, TE, NewTLA, TL).

%
% pxListGet(+Next, -List)
% pxListLength(+List, -Length)
%

% Next must match PX_End, PX_Cont 
pxListGet(1, []).
pxListGet(0, [E | L]) :- pxListGet(E, N, 0), pxListGet(N, L).

pxListLength(L, Len) :- pxListLength(L, 0, Len).
pxListLength([], Len, Len).
pxListLength([_ | L], LA, Len) :- NewLA is LA + 1, pxListLength(L, NewLA, Len).

%
% pxAttributeList(+AttributeList, +Predicate)
% pxQueryList(+AttributeList, +Data, +Predicate)
%

pxAttributeList([], _).
pxAttributeList([AV | AL], T) :-
	functor(AV, A, 1), arg(1, AV, V),
	functor(G, T, 2), arg(1, G, A), arg(2, G, V),
	call(G), pxAttributeList(AL, T).

pxQueryList([], _, _).
pxQueryList([AV | AL], D, T) :-
	functor(AV, A, 1), arg(1, AV, V),
	functor(G, T, 3), arg(1, G, A), arg(2, G, D), arg(3, G, V),
	call(G), pxQueryList(AL, D, T).

%
% pxPropertyCompose(+Format, +Atom, +Next, -Property)
% pxPropertyDecompose(+Property, -Format, -List, -Length)
%

% Format must match PX_String in ff.h
pxPropertyCompose(0, A, _, A) :- !, pxVectorXFree.
pxPropertyCompose(F, _, N, xProperty(F, L)) :- pxVectorGet(N, L).

pxPropertyDecompose(xProperty(F, L), F, L, Len) :- pxListLength(L, Len).

%
% pxRedCheck(+Value, -ModifiedValue, -Flag)
% pxGreenCheck(+Value, -ModifiedValue, -Flag)
% pxBlueCheck(+Value, -ModifiedValue, -Flag)
%
% pxColorGetN(N, -CellList)
% pxColorPut(+AssignmentList)
% pxPixelPut(+CellList)
%

pxRedCheck(xChange, 0, 1) :- !.
pxRedCheck(xNoChange, 0, 0) :- !.
pxRedCheck(R, R, 1).

pxGreenCheck(xChange, 0, 2) :- !.
pxGreenCheck(xNoChange, 0, 0) :- !.
pxGreenCheck(G, G, 2).

pxBlueCheck(xChange, 0, 4) :- !.
pxBlueCheck(xNoChange, 0, 0) :- !.
pxBlueCheck(B, B, 4).

pxColorGetN(0, []):- !.
pxColorGetN(N, [xCell(P, xColor(R, G, B)) | CL]) :-
	pxColorGet(P, R, G, B, 0), NewN is N - 1, pxColorGetN(NewN, CL).

pxColorPut([]).
pxColorPut([xCell(P, xColor(R, G, B)) | CL]) :-
	pxRedCheck(R, NewR, RBit),
	pxGreenCheck(G, NewG, GBit),
	pxBlueCheck(B, NewB, BBit),
	FM is RBit \/ GBit \/ BBit,
	pxColorPut(P, NewR, NewG, NewB, FM, 0),
	pxColorPut(CL).
pxColorPut(_) :- !, pxVectorXFree, fail.

pxPixelPut([]).
pxPixelPut([xCell(P, _) | CL]) :- pxSplitPutOne(P), pxPixelPut(CL).
pxPixelPut(_) :- !, pxVectorXFree, fail.

%
% pxPointPut(+PointList)
% pxSegmentPut(+SegmentList)
% pxRectanglePut(+RectangleList)
% pxArcPut(+ArcList)
%

pxPointPut(PL) :-
	pxListLength(PL, L), TL is L * 2, pxVectorAlloc(16, TL, 0),
	pxPointPut1(PL).

pxPointPut1([]).
pxPointPut1([xPoint(X, Y) | L]) :- pxVectorPut([X, Y]), pxPointPut1(L).
pxPointPut1(_) :- !, pxVectorXFree, fail.

pxSegmentPut(SL) :- 
	pxListLength(SL, L), TL is L * 4, pxVectorAlloc(16, TL, 0),
	pxSegmentPut1(SL).

pxSegmentPut1([]).
pxSegmentPut1([xSegment(X, Y, W, H) | L]) :-
	pxVectorPut([X, Y, W, H]), pxSegmentPut1(L).
pxSegmentPut1(_) :- !, pxVectorXFree, fail.

pxRectanglePut(RL) :-
	pxListLength(RL, L), TL is L * 4, pxVectorAlloc(16, TL, 0),
	pxRectanglePut1(RL).

pxRectanglePut1([]).
pxRectanglePut1([xRectangle(X, Y, W, H) | L]) :-
	pxVectorPut([X, Y, W, H]), pxRectanglePut1(L).
pxRectanglePut1(_) :- !, pxVectorXFree, fail.

pxArcPut(RL) :-
	pxListLength(RL, L), TL is L * 6, pxVectorAlloc(16, TL, 0),
	pxArcPut1(RL).

pxArcPut1([]).
pxArcPut1([ xArc(X, Y, W, H, A1, A2) | L]) :-
	pxVectorPut([X, Y, W, H, A1, A2]), pxArcPut1(L).
pxArcPut1(_) :- !, pxVectorXFree, fail.

%
% pxTextType(+Text, -Type)
% pxTextTypeOk(+CurrentType, +Text, -NewType)
% pxTextDecompose(+Text, -String, -Length)
% pxTextFormat(+TextList, -ItemList, -Type, -Items, -Chars)
% pxTextFormat(+TextList, -ItemList, +PendingFont, +PendingDelta,
%	+CType, -RType, +CItems, -RItems, +CChars, -RChars)
% pxTextPut(+ItemList)
%

% Type must match PX_TextItem8, PX_TextItem16
pxTextType(xText(8, _), 11) :- !.
pxTextType(xText(16, _), 12) :- !.
pxTextType(S, 11) :- atom(S).

pxTextTypeOk(0, E, NT) :- !, pxTextType(E, NT).
pxTextTypeOk(CT, E, CT) :- pxTextType(E, CT).

pxTextDecompose(xText(8, L), L, Len) :- !, pxListLength(L, Len).
pxTextDecompose(xText(16, L), L, Len) :- !, pxListLength(L, Len).
pxTextDecompose(E, E, Len) :- name(E, CL), pxListLength(CL, Len).

pxTextFormat(TL, IL, RT, RI, RC) :-
	pxTextFormat(TL, IL, 0, 0, 0, RT, 0, RI, 0, RC).

pxTextFormat([], [], 0, _, RT, RT, RI, RI, RC, RC) :- !.
pxTextFormat([], [pxText([], 0, 0, F)], F, _, RT, RT, CI, RI, RC, RC) :-
	RI is CI + 1.
pxTextFormat([xFont(F) | TL], IL, _, PD, CT, RT, CI, RI, CC, RC) :-
	pxTextFormat(TL, IL, F, PD, CT, RT, CI, RI, CC, RC).
pxTextFormat([xDelta(D) | TL], IL, F, PD, CT, RT, CI, RI, CC, RC) :-
	NewD is D + PD,
	pxTextFormat(TL, IL, F, NewD, CT, RT, CI, RI, CC, RC).
pxTextFormat([Text | TL], [pxText(S, SL, D, F) | IL], F, D,
  CT, RT, CI, RI, CC, RC) :- !,
	pxTextTypeOk(CT, Text, NewCT),
	NewCI is CI + 1,
	pxTextDecompose(Text, S, SL),
	NewCC is CC + SL,
	pxTextFormat(TL, IL, 0, 0, NewCT, RT, NewCI, RI, NewCC, RC).

% Type must match PX_TextAtom, PX_TextVector
pxTextPut([]).
pxTextPut([pxText(S, L, D, F) | IL]) :-
	atom(S), !, pxTextPut(0, S, L, D, F, 0), pxTextPut(IL).
pxTextPut([pxText(S, L, D, F) | IL]) :-
	pxTextPut(13, [], L, D, F, 0), pxVectorPut(S), pxTextPut(IL).
pxTextPut(_) :- !, pxVectorXFree, fail.

%
% pxKeymapToCodes(+MapList, -CodeList)
% pxKeymapFromCodes(-MapList, +CodeList)
%

pxKeymapToCodes([], []) :- !.
pxKeymapToCodes([E | ML], CL) :- pxKeymapToCodes(E, ML, 0, 0, CL).
pxKeymapToCodes(0, [], _, _, []) :- !.
pxKeymapToCodes(0, [E | ML], _, B, CL) :- !,
	NewB is B + 8,
	pxKeymapToCodes(E, ML, 0, NewB, CL).
pxKeymapToCodes(E, ML, I, B, [C | CL]) :- 
	Number is 1 << I,
	Result is Number /\ E,
	Result > 0, !,
	C is I + B,
	NewE is \(Number) /\ E,
	NewI is I + 1,
	pxKeymapToCodes(NewE, ML, NewI, B, CL).
pxKeymapToCodes(E, ML, I, B, CL) :-
	NewI is I + 1,
	pxKeymapToCodes(E, ML, NewI, B, CL).

pxKeymapFromCodes([0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,
	           0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0], []) :-  !.
pxKeymapFromCodes(ML, [C | CL]) :- 
	sort([C | CL], [NewC | NewCL]),
	pxKeymapFromCodes(NewC, NewCL, 0, 0, ML).

pxKeymapFromCodes(_, _, 256, _, []) :- !.
pxKeymapFromCodes(C, [], B, E, ML) :- !, pxKFC(C, B, E, ML).
pxKeymapFromCodes(C, [NewC | CL], B, E, ML) :-
	B is C /\ \(7), !, 
	I is C /\ 7, R is 1 << I, NewE is E \/ R,
	pxKeymapFromCodes(NewC, CL, B, NewE, ML).
pxKeymapFromCodes(C, CL, B, E, [E | ML]) :-
	NewB is B + 8, pxKeymapFromCodes(C, CL, NewB, 0, ML).

pxKFC(_, 256, E, [E]) :- !.
pxKFC(C, B, E, [NewE | ML]) :-
	B is C /\ \(7), !,
	I is C /\ 7, R is 1 << I, NewE is E \/ R,
	NewB is B + 8,
	pxMakeZeros(NewB, ML).
pxKFC(C, B, E, [E | ML]) :-
	NewB is B + 8,
	pxKFC(C, NewB, 0, ML).

pxMakeZeros(256, []) :- !.
pxMakeZeros(B, [0 | ML]) :-
	NewB is B + 8, pxMakeZeros(NewB, ML).
	
%
% pxMapCompose(+Next, +Count, +Keysyms, +Functor1, +Functor2, -Map)
% pxMapGet(+Count, +Limit, +Keysyms, +Functor, -Map)
% pxMapDecompose(+Map, +Functor1, +Functor2, -Count, -KeySyms)
% pxMapCheck(+Count, +Limit, +Functor, +Keysyms, +Map)
% pxMapPut(+Count, +Limit, +Keysyms, +Map)
%

pxMapCompose(1, _, _, F1, _, T) :- functor(T, F1, 0).
pxMapCompose(0, C, K, F1, F2, Map) :-
	functor(Map, F1, C), E is C + 1, pxMapGet(1, E, K, F2, Map).

pxMapGet(E, E, _, _, _) :- !.
pxMapGet(C, L, K, F2, Map) :-
	pxMapGetN(F2, K, E), arg(C, Map, E), NewC is C + 1,
	pxMapGet(NewC, L, K, F2, Map).

pxMapGetN(N, L, S) :-
	functor(S, N, L), E is L + 1, pxMapGetOne(1, E, S).

pxMapGetOne(E, E, _) :- !.
pxMapGetOne(C, L, S) :- 
	pxVectorGet(R, _, 0), pxcNoSymbol(R, E), arg(C, S, E), NewC is C + 1,
	pxMapGetOne(NewC, L, S).

pxMapDecompose(Map, F1, _, 0, 0) :- functor(Map, F1, 0), !.
pxMapDecompose(Map, F1, F2, C, K) :-
	functor(Map, F1, C), arg(1, Map, Arg1), functor(Arg1, F2, K),
	E is C + 1,
	pxMapCheck(2, E, F2, K, Map).

pxMapCheck(E, E, _, _, _) :- !.
pxMapCheck(C, L, F2, K, Map) :-
	arg(C, Map, E), functor(E, F2, K), NewC is C + 1,
	pxMapCheck(NewC, L, F2, K, Map).

pxMapPut(E, E, _, _) :- !.
pxMapPut(C, L, K, Map) :-
	arg(C, Map, E), pxMapPutN(K, E), NewC is C + 1,
	pxMapPut(NewC, L, K, Map).

pxMapPutN(L, S) :-
	E is L + 1, pxMapPutOne(1, E, S).

pxMapPutOne(E, E, _) :- !.
pxMapPutOne(C, L, S) :-
	arg(C, S, E), pxcNoSymbol(R, E), pxVectorPut(R,0), NewC is C + 1,
	pxMapPutOne(NewC, L, S).

%
% pxEventGet(+Length, -Event)
% pxEventPut(+Event)
%
pxEventGet(37, E) :- !, pxEventGet(6, E). % kludge for keymap notify
pxEventGet(L, E) :-
	functor(E, xEvent, L),
	pxSplitGetOne(T), pxcEventType(Type, T, Struct), arg(1, E, Type),
	pxSplitGetOne(S), arg(2, E, S),
	pxSplitGetOne(SE), pxcBoolean(SE, SendEvent), arg(3, E, SendEvent),
	pxSplitGetOne(C), arg(4, E, C),
	pxSplitGetOne(W), arg(5, E, W),
	pxcGetEvent(Struct, E, L).
pxEventGet(_, _) :- !, pxVectorXFree, fail.

pxEventPut(E) :-
	functor(E, xEvent, L), 
	arg(1, E, Type),
	(Type = xKeymapNotify ->
	    pxVectorAlloc(32, 37, 0) ;		% keymap notify length
	    pxVectorAlloc(32, L, 0) ),
	pxcEventType(Type, T, Struct), pxSplitPutOne(T),
	arg(2, E, Serial), pxSplitPutOne(Serial),
	arg(3, E, SendEvent), pxcBoolean(SE, SendEvent), pxSplitPutOne(SE),
	arg(4, E, Connection), pxSplitPutOne(Connection),
	arg(5, E, Window), pxSplitPutOne(Window),
	pxcPutEvent(Struct, E, L).
pxEventPut(_) :- !, pxVectorXFree, fail.

%
% pxMotionEventList(+Next, -TimeEventList)
%

% must match PX_Cont, PX_SplitVector, PX_End 
pxMotionEventList(1, []).
pxMotionEventList(4, [xMotion(xSplit(MS, LS), X, Y) | L]) :-
	pxSplitMotion(MS, LS, X, Y, N, 0), pxMotionEventList(N, L).
pxMotionEventList(0, [xMotion(T, X, Y) | L]) :- 
	pxNormalMotion(T, X, Y, N, 0), pxMotionEventList(N, L).

%
% pxMaskFromBits(-Mask, +BitList, +Table)
% pxMaskToBits(+Mask, -BitList, +Table)
%

pxMaskFromBits(M, BL, T) :- pxMaskFromBits(BL, 0, M, T).

pxMaskFromBits([], M, M, _) :- !.
pxMaskFromBits([B | BL], A, M, T) :-
	functor(G, T, 2), arg(1, G, B), arg(2, G, P), call(G),
	NewA is (1 << P) \/ A,
	pxMaskFromBits(BL, NewA, M, T).

pxMaskToBits(M, BL, T) :- 
	functor(G, T, 2), arg(1, G, maxindex), arg(2, G, MI), call(G),
	pxMaskToBits(M, BL, T, 0, MI).

pxMaskToBits(0, [], _, _, _) :- !.
pxMaskToBits(_, [], _, MI, MI) :- !.
pxMaskToBits(M, [B | BL], T, P, MI) :-
	functor(G, T, 2), arg(1, G, B), arg(2, G, P), call(G),
	Number is 1 << P,
	Result is Number /\ M,
	Result > 0,
	!,
	NewM is \(Number) /\ M,
	NewP is P + 1,
	pxMaskToBits(NewM, BL, T, NewP, MI).
pxMaskToBits(M, BL, T, P, MI) :-
	NewP is P + 1,
	pxMaskToBits(M, BL, T, NewP, MI).

%
% pxAppend(L1, L2, L3)
%	traditional append
% pxMember(List, Item)
%	traditional member
% pxFMember(List, Functor, Arg1)
%	functor member
%
pxAppend([], L, L).
pxAppend([X | Xs], Ys, [X | Zs]) :- pxAppend(Xs, Ys, Zs).

pxMember([], _) :- fail.
pxMember([E | _], E) :- !.
pxMember([_ | L], I) :- pxMember(L, I).

pxFMember([], _, _) :- fail.
pxFMember([E | _], F, A1) :- functor(E, F, _), !, arg(1, E, A1).
pxFMember([_ | L], F, A1) :- pxFMember(L, F, A1).

%
% pxTermToString(Term, String)
% pxTermToString(Term, Current, String)
%	convert term to string representation
% pxArgsToString(ArgList, String)
%	convert arg list to string representation
%
pxTermToString(T, S) :- pxTermToString(T, [], S).

pxTermToString(T, C, S) :-
	T =.. [F | A],
	name(F, FCs),
	pxAppend(C, FCs, NewC),
	pxArgsToString(A, NewC, S, 1).		% start flag

pxArgsToString([], C, S, 0) :- pxAppend(C, [41], S). % close paren
pxArgsToString([], C, C, 1).
pxArgsToString([E | L], C, S, 0) :-
	pxAppend(C, [44], C1),			% comma
	pxTermToString(E, C1, C2),
	pxArgsToString(L, C2, S, 0).
pxArgsToString([E | L], C, S, 1) :-
	pxAppend(C, [40], C1),			% open paren
	pxTermToString(E, C1, C2),
	pxArgsToString(L, C2, S, 0).

%
% pxPropertyToText(+Property, +Terminated, -TextList)
% pxPropertyToText(+PropertyList, +Terminated, +NameLength, -NameHead,
%   -NameList, -TextList)
%
% pxTextToProperty(+TextList, +Terminated, -Property)
% pxTextToProperty1(+TextList, +Terminated, -PropertyList)
% pxTextToProperty2(+NameList, -PropertyList, -Tail)
%
%	TextList = single atom or mixed list of atoms and (format 8)
%		xText structures, empty strings are represented by
%		xText(8, []) 
%	Terminated = null terminated, rather than null-separated
%	Length is in bytes
%

pxPropertyToText(xProperty(8, []), _, []) :- !.
pxPropertyToText(xProperty(8, PL), Term, TL) :-
	pxPropertyToText(PL, Term, 0, NH, NH, TL).

pxPropertyToText([], xFalse, 0, _, _, [xText(8, [])]) :- !.
pxPropertyToText([], xFalse, _, NH, [], [A]) :- name(A, NH).
pxPropertyToText([], xTrue, 0, _, _, []) :- !.
pxPropertyToText([], xTrue, _, _, _, _) :- fail. % no null
pxPropertyToText([0 | PL], Term, 0, NH, NL, [xText(8, []) | TL]) :- !,
	pxPropertyToText(PL, Term, 0, NH, NL, TL).
pxPropertyToText([0 | PL], Term, _, NH, [], [A | TL]) :- !,
	name(A, NH), pxPropertyToText(PL, Term, 0, NewNL, NewNL, TL).
pxPropertyToText([C | PL], Term, NLen, NH, [C | T], TL) :-
	NewNLen is NLen + 1, pxPropertyToText(PL, Term, NewNLen, NH, T, TL).

pxTextToProperty(A, Term, xProperty(8, PL)) :-
	atom(A), !, pxTextToProperty1([A], Term, PL).
pxTextToProperty(T, Term, xProperty(8, PL)) :-
	T = xText(8, _), !, pxTextToProperty1([T], Term, PL).
pxTextToProperty(TL, Term, xProperty(8, PL)) :- pxTextToProperty1(TL, Term, PL).

pxTextToProperty1([[]], xFalse, []) :- !.
pxTextToProperty1([[]], xTrue, [0]) :- !.
pxTextToProperty1([A], xFalse, PL) :- atom(A), !,
	name(A, NL), pxTextToProperty2(NL, PL, []).
pxTextToProperty1([A], xTrue, PL) :- atom(A), !,
	name(A, NL), pxTextToProperty2(NL, PL, [0]).
pxTextToProperty1([xText(8, C)], xFalse, PL) :- pxTextToProperty2(C, PL, []).
pxTextToProperty1([xText(8, C)], xTrue, PL) :- pxTextToProperty2(C, PL, [0]).
pxTextToProperty1([[] | TL], Term, [0 | NewPL]) :- !,
	pxTextToProperty1(TL, Term, NewPL).
pxTextToProperty1([A | TL], Term, PL) :- atom(A), !,
	name(A, NL),
	pxTextToProperty2(NL, PL, [0 | NewPL]),
	pxTextToProperty1(TL, Term, NewPL).
pxTextToProperty1([xText(8, C) | TL], Term, PL) :- 
	pxTextToProperty2(C, PL, [0 | NewPL]),
	pxTextToProperty1(TL, Term, NewPL).

pxTextToProperty2([], T, T).
pxTextToProperty2([C | L], [C | PL], T) :- pxTextToProperty2(L, PL, T).

% These functions deal with possibly incomplete structure properties
% with flags to determine present members. (eg WM_HINTS)
%
% pxPropertySet(Name, Arity, List, Property)
% pxPropertySet1(List, Struct, CurrentFlags, Flags)
%	makes a structure property where only some elements are present
% pxPropertyArgsToList(Structure, Default, Head, Tail)
% pxPropertyArgsToList(Index, Arity, Structure, Default, Head, Tail)
%	convert property structure to list
%
%	form of property set table:
%		PropertyName(type, Format).
%		PropertyName(Attribute, Value, Index, Flag, RawValue)
%
%	Length is not stored in the table, since some structs may be
%	of variable length. (ie pre and post-IC3M WM_SIZE_HINTS)

pxPropertySet(N, A, L, xProperty(F, P)) :-
	functor(Q, N, 2),
	arg(1, Q, type),
	call(Q),
	arg(2, Q, F),
	functor(S, N, A),
	pxPropertySet1(L, S, 0, Flags),
	arg(1, S, Flags),
	pxPropertyArgsToList(S, 0, P, []).

pxPropertySet1([], _, CFlags, CFlags).
pxPropertySet1([E | L], S, CFlags, Flags) :-
	functor(S, SetName, _), functor(Goal, SetName, 5),
	functor(E, Attribute, 1), arg(1, Goal, Attribute),
	arg(1, E, Value), arg(2, Goal, Value),
	call(Goal),
	arg(5, Goal, Raw), arg(3, Goal, Index), arg(Index, S, Raw),
	arg(4, Goal, Flag), NewCFlags is CFlags \/ Flag,
	pxPropertySet1(L, S, NewCFlags, Flags).

pxPropertyArgsToList(S, D, H, T) :-
	functor(S, _, A), 
	pxPropertyArgsToList(0, A, S, D, H, T).

pxPropertyArgsToList(A, A, _, _, T, T).
pxPropertyArgsToList(I, A, S, D, [D | L], T) :-
	NewI is I + 1, arg(NewI, S, E), var(E), !,
	pxPropertyArgsToList(NewI, A, S, D, L, T).
pxPropertyArgsToList(I, A, S, D, [E | L], T) :-
	NewI is I + 1, arg(NewI, S, E),
	pxPropertyArgsToList(NewI, A, S, D, L, T).

% These functions deal with possibly repeating structure properties
% whose missing elements are filled in with some default value. 
%
% pxPropertyStruct(Name, Arity, Repeating, List, Property)
% pxPropertyStruct(ListList, Name, Arity, Default, Head, Tail)
% pxPropertyStruct(List, Struct)
%
%	form of property struct table:
%		PropertyName(type, Format, DefaultValue).
%		PropertyName(Attribute, Value, Index, RawValue).
%
%	Length and whether the structure can be repeating is not
%	stored in the table for future flexibility.

pxPropertyStruct(N, A, xFalse, L, xProperty(F, P)) :-
	functor(Q, N, 3),
	arg(1, Q, type),
	call(Q),
	arg(2, Q, F),
	arg(3, Q, D),
	functor(S, N, A),
	pxPropertyStruct(L, S),
	pxPropertyArgsToList(S, D, P, []).
pxPropertyStruct(N, A, xTrue, LL, xProperty(F, P)) :-
	functor(Q, N, 3),
	arg(1, Q, type),
	call(Q),
	arg(2, Q, F),
	arg(3, Q, D),
	pxPropertyStruct(LL, N, A, D, P, []).

pxPropertyStruct([], _, _, _, T, T).
pxPropertyStruct(L, N, A, D, H, T) :-
	functor(S, N, A),
	pxPropertyStruct(L, S),
	pxPropertyArgsToList(S, D, H, PL),
	pxPropertyStruct(L, N, A, D, PL, T).

pxPropertyStruct([], _).
pxPropertyStruct([E | L], S) :-
	functor(S, StructName, _), functor(Goal, StructName, 4),
	functor(E, Attribute, 1), arg(1, Goal, Attribute),
	arg(1, E, Value), arg(2, Goal, Value),
	call(Goal),
	arg(4, Goal, Raw), arg(3, Goal, Index), arg(Index, S, Raw),
	pxPropertyStruct(L, S).

%
% pxOneHint(Hints, HintName, Default, Value)
% pxTwoHints(Hints, FirstHint, SecondHints, Default, Value)
%
pxOneHint(Hints, Hint, _, Value) :- pxFMember(Hints, Hint, Value), !.
pxOneHint(_, _, Default, Default).

pxTwoHints(Hints, First, _, _, Value) :- pxFMember(Hints, First, Value), !.
pxTwoHints(Hints, _, Second, _, Value) :- pxFMember(Hints, Second, Value), !.
pxTwoHints(_, _, _, Default, Default).

%
% pxComputeX(UserX, UserXNegative, ProgramX, ProgramXNegative, Screen, Width, 
%	BorderWidth, X, UserFlag, XNegative)
% pxComputeY(UserY, UserYNegative, ProgramY, ProgramYNegative, Screen, Width, 
%	BorderWidth, Y, UserFlag, YNegative)
% pxComputeWH(Hints, UserSize, ProgramDefaultSize, IncrementHint,
%	BaseHint, MinHint, MaxHint, Value)
% pxSelectWH(User, Program, Default, Value, UserFlag)
%

pxComputeX(xNone, _, xNone, _, _, _, _, 0, xFalse, xFalse) :- !.
pxComputeX(xNone, _, PX, xFalse, _, _, _, PX, xFalse, xFalse) :- !.
pxComputeX(xNone, _, PX, xTrue, S, W, BW, X, xFalse, xTrue) :- !,
	xQueryScreen(S, [xWidth(SW)]), X is SW + PX - W - 2 * BW.
pxComputeX(UX, xFalse, _, _, _, _, _, UX, xTrue, xFalse).
pxComputeX(UX, xTrue, _, _, S, W, BW, X, xTrue, xTrue) :-	
	xQueryScreen(S, [xWidth(SW)]), X is SW + UX - W - 2 * BW.

pxComputeY(xNone, _, xNone, _, _, _, _, 0, xFalse, xFalse) :- !.
pxComputeY(xNone, _, PY, xFalse, _, _, _, PY, xFalse, xFalse) :- !.
pxComputeY(xNone, _, PY, xTrue, S, H, BW, Y, xFalse, xTrue) :- !,
	xQueryScreen(S, [xHeight(SH)]), Y is SH + PY - H - 2 * BW.
pxComputeY(UY, xFalse, _, _, _, _, _, UY, xTrue, xFalse).
pxComputeY(UY, xTrue, _, _, S, H, BW, Y, xTrue, xTrue) :-	
	xQueryScreen(S, [xHeight(SH)]), Y is SH + UY - H - 2 * BW.

pxComputeWH(Hints, USize, PSize, IncH, BaseH, MinH, MaxH, Value, UserFlag) :-
	pxTwoHints(Hints, BaseH, MinH, 0, Base),
	pxOneHint(Hints, MinH, Base, Min),
	pxOneHint(Hints, IncH, 1, Inc),
	pxSelectWH(USize, PSize, 1, Size, UserFlag),
	Initial is Size * Inc + Base,
	pxOneHint(Hints, MaxH, Initial, Max),
	(Initial < Min -> New = Min; New = Initial),
	(New > Max -> Value = Max; Value = New).

pxSelectWH(xNone, xNone, Default, Default, xFalse) :- !.
pxSelectWH(xNone, Program, _, Program, xFalse) :- !.
pxSelectWH(User, _, _, User, xTrue).

%
% pxPositionHints(XUserFlag, YUserFlag, X, Y, XHint, YHint)
%	parse position hints
% pxSizeHints(WUserFlag, HUserFlag, W, H, WHint, HHint)
%	parse size hints
% pxComputeGravity(XNegative, YNegative, Gravity)
%

pxPositionHints(xFalse, xFalse, X, Y, xProgramX(X), xProgramY(Y)) :- !.
pxPositionHints(_, _, X, Y, xUserX(X), xUserY(Y)).

pxSizeHints(xFalse, xFalse, W, H, xProgramWidth(W), xProgramHeight(H)) :- !.
pxSizeHints(_, _, W, H, xUserWidth(W), xUserHeight(H)).

pxComputeGravity(xFalse, xFalse, xNorthWest).
pxComputeGravity(xFalse, xTrue,  xSouthWest).
pxComputeGravity(xTrue,  xFalse, xNorthEast).
pxComputeGravity(xTrue,  xTrue,  xSouthEast).

%
% pxPropertyToSize(+Property, -Size, -Tail)
%
pxPropertyToSize([], T, T).
pxPropertyToSize([MinW, MinH, MaxW, MaxH, WInc, HInc | PL],
  [xIconSize(MinW, MinH, MaxW, MaxH, WInc, HInc) | SL], T) :-
	pxPropertyToSize(PL, SL, T).

%
% pxProtocol(+ProtocolList, +Connection, -AtomList)
%
pxProtocol([], _, []).
pxProtocol([E | PL], C, [A | AL]) :-
	xAtom(C, E, A), pxProtocol(PL, C, AL).

%
% eof
%

