/*
 * Copyright (C) 1990 by Ted Kim.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/*
 * ff.c - XWIP foreign function support
 *
 * PXError
 *
 * PXVectorFree
 * pxVectorXFree
 *
 * pxPrecisionGet
 * pxPrecisionSet
 *
 * pxUnsignedPut1
 * pxUnsignedPut2
 *
 * pxVectorAlloc
 * pxVectorGet
 * pxVectorPut
 *
 * pxSplitGet
 * pxSplitGetOne
 * pxSplitPut
 *
 * pxStringAlloc
 * pxStringDealloc
 * pxStringGet
 * pxStringPutOne
 *
 * pxTextAlloc
 * pxTextPut
 *
 * pxColorGet
 * pxColorPut
 *
 * pxListGet
 */

#include <stdio.h>
#include <string.h>
#include <netdb.h>
#include <X11/Xlib.h>

#include "dl.h"
#include "ff.h"
#include "attrib.h"

extern char *malloc();

char *pxErrorString = NULL;
bool pxServerError = false;
unsigned long pxErrorSerial = 0;

int pxType =  PX_Invalid;
int pxElements = 0;
DLHeader *pxList = NULL;
PxVector8 *pxVector8 = NULL;
PxVector16 *pxVector16 = NULL;
PxVector32 *pxVector32 = NULL;
XFontProp *pxFontProperty = NULL;
XChar2b *pxText16 = NULL;
char **pxStringPointers = NULL;
XTextItem *pxTextItem8 = NULL;
XTextItem16 *pxTextItem16 = NULL;
XColor *pxColor = NULL;
XHostAddress *pxHostAddress = NULL;

/* use Xlib Bool since these are fed directly into Xlib functions */
Bool pxAttribChanged1 = False;
Bool pxAttribChanged2 = False;
int pxAttribMask1 = 0;
int pxAttribMask2 = 0;
PxAttributes pxAttribBuf;

unsigned pxUnsignedArg1 = 0;
unsigned pxUnsignedArg2 = 0;

int pxPrecision = 31;
int pxPrecisionMask = (1 << 31);

char *pxMallocBlock = NULL;
char *pxStringArea = NULL;

char *pxNil = "[]";
char *bad_event_format = "bad event format";
char *bad_request = "bad request";
char *cant_initialize = "cannot initialize";
char *coordinate_out_of_range = "coordinate out of range";
char *no_matching_screen = "no matching screen";
char *no_memory = "can't get memory";
char *no_such_character = "no such character";
char *no_such_connection = "no such connection";
char *no_such_depth = "no such depth";
char *no_such_fil = "no such font information loaded";
char *no_such_format = "no such format";
char *no_such_gc = "no such GC";
char *no_such_image = "no such image";
char *no_such_screen = "no such screen";
char *no_such_visual = "no such visual";
char *no_such_window = "no such window";
char *not_text = "not text";
char *unknown_attribute = "unknown attribute";
char *unknown_event_type = "unknown event type";

/*
 * PXError(msg)
 *
 * use with ErrorInit macro
 * always returns fail
 */
PXError(msg)
     char *msg;			
{
  (void)fprintf(stderr, "[ERROR %s: %s]\n", pxErrorString, msg);
  return FF_Fail;
}

/*
 * PXVectorFree()
 *
 * call to release pxMallocBlock storage, returns FF_Fail
 */
PXVectorFree()
{
  pxType = PX_Invalid;
  if (pxMallocBlock != NULL) {
    free(pxMallocBlock);
    pxMallocBlock = NULL;
  }
  return FF_Fail;
}

/*
 * pxVectorXFree()
 *
 * same as above but use XFree to free X objects
 * also can be called from Prolog, though it is rarely necessary to do so
 * void return
 */
void
pxVectorXFree()
{
  pxType = PX_Invalid;
  if (pxMallocBlock != NULL) {
    XFree(pxMallocBlock);
    pxMallocBlock = NULL;
  }
}

/*
 * pxPrecisionGet(-Bits)
 *	integer: Bits
 */
void
pxPrecisionGet(pb)
     FFInteger *pb;
{
  *pb = pxPrecision + 1;
}

/*
 * pxPrecisionSet(+Bits, 0)
 *	integer: Bits
 */
FFInteger
pxPrecisionSet(pb)
     FFInteger pb;
{
  if ((pb < 26) || (pb > 33))
    return FF_Fail;
  
  pxPrecision = (int)(--pb);
  pxPrecisionMask = (-1) << pb;
  return FF_Ok;
}

/*
 * pxUnsignedPut1(+Type, +Most, +Least, 0)
 *	integer: Type, Most, Least
 */
FFInteger
pxUnsignedPut1(pt, pm, pl)
     FFInteger pt, pm, pl;
{
  switch (pt) {
  case PX_Integer:
    pxUnsignedArg1 = pm;
    break;
  case PX_SplitVector:
    pxUnsignedArg1 = PX_JoinSplit(pm, pl);
    break;
  default:
    return FF_Fail;
  }
  return FF_Ok;
}

/*
 * pxUnsignedPut2(+Type, +Most, +Least, 0)
 *	integer: Type, Most, Least
 */
FFInteger
pxUnsignedPut2(pt, pm, pl)
     FFInteger pt, pm, pl;
{
  switch (pt) {
  case PX_Integer:
    pxUnsignedArg2 = pm;
    break;
  case PX_SplitVector:
    pxUnsignedArg2 = PX_JoinSplit(pm, pl);
    break;
  default:
    return FF_Fail;
  }
  return FF_Ok;
}

/*
 * pxVectorAlloc(+Type, +Length, 0)
 *	integer: Type, Length
 */
FFInteger
pxVectorAlloc(pt, pl)
     FFInteger pt, pl;
{
  int variable, fixed = 0;
  
  if (pl < 0)
    return FF_Fail;
  switch (pt) {

  case PX_Vector8:
    variable = sizeof(PxVector8);
    break;

  case PX_Vector16:
    variable = sizeof(PxVector16);
    break;
    
  case PX_Vector32:
    variable = sizeof(PxVector32);
    break;

  case PX_Text16:
    variable = sizeof(XChar2b);
    break;

  case PX_ColorVector:
    variable = sizeof(XColor);
    break;

  case PX_ModifierMap:
    variable = sizeof(KeyCode);
    fixed = sizeof(XModifierKeymap);
    break;

  default:
    return FF_Fail;
  }
  
  if (!pl)
    pxMallocBlock = NULL;
  else if ((pxMallocBlock = malloc((unsigned)(variable * pl + fixed))) == NULL)
    return FF_Fail;

  switch (pt) {
  case PX_Vector8:
    pxVector8 = pxMallocBlock;
    break;

  case PX_Vector16:
    pxVector16 = (PxVector16 *)pxMallocBlock;
    break;

  case PX_Vector32:
    pxVector32 = (PxVector32 *)pxMallocBlock;
    break;

  case PX_Text16:
    pxText16 = (XChar2b *)pxMallocBlock;
    break;

  case PX_ColorVector:
    pxColor = (XColor *)pxMallocBlock;
    break;

  case PX_ModifierMap:
    pxVector8 = pxMallocBlock + sizeof(XModifierKeymap);
    ((XModifierKeymap *)pxMallocBlock)->modifiermap = (KeyCode *)pxVector8;
    break;
  }

  pxElements = pl;
  pxType = pt;
  return FF_Ok;
}

/*
 * pxVectorGet (-Element, -Next, 0)
 *	integer: Element, Next
 */
FFInteger
pxVectorGet(pe, pn)
     FFInteger *pe, *pn;
{
  *pe = *pn = PX_Invalid;

  if (!pxElements)
    return FF_Fail;

  switch (pxType) {
  case PX_Vector8:
  case PX_ModifierMap:
    *pe = (int)(*pxVector8++) & 0xFF;
    break;

  case PX_Vector16:
    *pe = (int)(*pxVector16++) & 0xFFFF;
    break;

  case PX_Vector32:
    *pe = *pxVector32++;
    break;

  case PX_FontProperty:
    *pe = pxFontProperty->name;
    pxFontProperty++;
    break;

  default:
    return FF_Fail;
  }

  if (--pxElements == 0) {
    if (pxType == PX_ModifierMap) {
      pxType = PX_Invalid;
      XFreeModifiermap(pxMallocBlock); /* note wierd capitalization */
      pxMallocBlock = NULL;
    } else
      pxVectorXFree();
    *pn = PX_End;
  } else 
    *pn = ((pxType == PX_Vector32) && ((*pxVector32) & pxPrecisionMask)) ?
      PX_SplitVector : PX_Cont;
  return FF_Ok;
}

/*
 * pxVectorPut(+Element, 0)
 *	integer: Element
 */
FFInteger
pxVectorPut(pe)
     FFInteger pe;
{
  switch (pxType) {
  case PX_Vector8:
  case PX_TextItem8:
  case PX_ModifierMap:
    *pxVector8++ = (char)(pe & 0xFF);
    break;

  case PX_Vector16:
    *pxVector16++ = (short)(pe & 0xFFFF);
    break;

  case PX_Vector32:
    *pxVector32++ = pe;
    break;

  case PX_Text16:
  case PX_TextItem16:
    pxText16->byte1 = (unsigned char)(pe & 0xFF00);
    pxText16->byte2 = (unsigned char)(pe & 0xFF);
    pxText16++;
    break;

  case PX_ColorVector:		/* pixel put */
    pxColor->pixel = pe;
    pxColor++;
    break;

  default:
    return FF_Fail;
  }
  return  FF_Ok;
}

/*
 * pxSplitGet(-Most, -Least, -Next, 0)
 *	integer: Most, Least, Next
 */
FFInteger
pxSplitGet(pm, pl, pn)
     FFInteger *pm, *pl, *pn;
{
  *pm = *pl = *pn = PX_Invalid;

  if ((pxElements == 0) || (pxType != PX_Vector32))
    return FF_Fail;

  *pm = PX_MostSplit(*pxVector32);
  *pl = PX_LeastSplit(*pxVector32);
  pxVector32++;
  
  if (--pxElements == 0) {
    pxVectorXFree();
    *pn = PX_End;
  } else
    *pn = ((*pxVector32) & pxPrecisionMask) ? PX_SplitVector : PX_Cont;
  return FF_Ok;
}

/*
 * pxSplitGetOne(-Most, -Least, 0)
 *	integer: Most, Least, 
 */
FFInteger
pxSplitGetOne(pm, pl)
     FFInteger *pm, *pl;
{
  *pm = *pl = PX_Invalid;

  if ((pxElements == 0) || (pxType != PX_Vector32))
    return FF_Fail;

  if ((*pxVector32) & pxPrecisionMask) {
    *pm = PX_MostSplit(*pxVector32);
    *pl = PX_LeastSplit(*pxVector32);
  } else {
    *pm = 0;
    *pl = *pxVector32;
  }
  pxVector32++;
  
  if (--pxElements == 0) 
    pxVectorXFree();
  return FF_Ok;
}

/*
 * pxSplitPut(+Most, +Least, 0)
 *	integer: Most, Least
 */
FFInteger
pxSplitPut(pm, pl)
     FFInteger pm, pl;
{
  if (pxType != PX_Vector32)
    return FF_Fail;

  *pxVector32++ = PX_JoinSplit(pm, pl);
  return FF_Ok;
}

/*
 * pxStringAlloc(+Type, +Elements, +Length, 0)
 *	integer: Type, Elements, Length
 */
FFInteger
pxStringAlloc(pt, pe, pl)
     FFInteger pt, pe, pl;
{
  unsigned ptrs = sizeof(char *) * pe;

  if ((pe < 0) || (pl < 0))
    return FF_Fail;
  if (!pe) {
    pxMallocBlock = NULL;
    pxType = pt;
    return FF_Ok;
  }
  if ((pxMallocBlock = malloc((unsigned)(ptrs + pl + pe))) == NULL)
    return FF_Fail;
  pxStringPointers = (char **)pxMallocBlock;
  pxStringArea = pxMallocBlock + ptrs;
  pxElements = pe;
  pxType = pt;
  return FF_Ok;
}

/*
 * pxStringDealloc
 *
 * other get functions return all data in their arguments
 * but pxStringGet passes a string pointer, but not the actual chars
 * so pxStringGet can't dealloc the storage 
 * so we need this separate predicate
 */
FFInteger
pxStringDealloc()
{
  if (pxMallocBlock != NULL)
    switch (pxType) {
    case PX_VectorFont:
      XFreeFontNames(pxMallocBlock);
      break;
    case PX_VectorPath:
      XFreeFontPath(pxMallocBlock);
      break;
    case PX_VectorExtension:
      XFreeExtensionList(pxMallocBlock);
      break;
    case PX_HostAddress:
      (void)PXVectorFree();
      break;
    default:
      return FF_Fail;
    }
  pxType = PX_Invalid;
  pxMallocBlock = NULL;
  return FF_Ok;
}

/*
 * pxStringGet(-Element, -Next, 0)
 *	integer: Next
 *	string: Element
 */
FFInteger
pxStringGet(pe, pn)
     FFString *pe;
     FFInteger *pn;
{
  *pe = pxNil;
  *pn = PX_Invalid;

  if (!pxElements)
    return FF_Fail;

  switch (pxType) {
  case PX_VectorFont: 
  case PX_VectorPath:
  case PX_VectorExtension:
  case PX_HostAddress:
    break;
  default: return FF_Fail;
  }

  if (pxType == PX_HostAddress) {
    struct hostent *hep;

    if ((hep = gethostbyaddr(pxHostAddress->address,
			     pxHostAddress->length,
			     pxHostAddress->family)) == NULL)
      return PXVectorFree();
    *pe = hep->h_name;
    pxHostAddress++;
  } else 
    *pe = *pxStringPointers++;
  
  if (--pxElements == 0) 
    *pn = PX_End;		/* actual dealloc done in pxStringDealloc */
  else 
    *pn = PX_Cont;
  return FF_Ok;
}

/*
 * pxStringPutOne(+Element)
 * 	string: Element
 */
void
pxStringPutOne(pe)
     FFString pe;
{
  *pxStringPointers++ = pxStringArea;
  (void)strcpy(pxStringArea, pe);
  pxStringArea += (strlen(pe) + 1);
}

/*
 * pxTextAlloc(+Type, +Items, +Characters, 0)
 *	integer: Type, Items, Characters
 */
FFInteger
pxTextAlloc(pt, pi, pchars)
     FFInteger pt, pi, pchars;
{
  unsigned isize, esize;

  if ((pi < 0) || (pchars < 0))
    return FF_Fail;
  switch (pt) {
  case PX_TextItem8:
    isize = sizeof(XTextItem);
    esize = sizeof(PxVector8);
    break;
  case PX_TextItem16:
    isize = sizeof(XTextItem16);
    esize = sizeof(PxVector16);
    break;
  default:
    return FF_Fail;
  }
  if (!pi) {
    pxMallocBlock = NULL;
    pxType = pt;
    return FF_Ok;
  }
  if ((pxMallocBlock = malloc((unsigned)(isize * pi + esize * pchars))) == NULL)
    return FF_Fail;
  if (pt == PX_TextItem8) {
    pxTextItem8 = (XTextItem *)pxMallocBlock;
    pxVector8 = pxMallocBlock + (isize * pi);
  } else {
    pxTextItem16 = (XTextItem16 *)pxMallocBlock;
    pxText16 = (XChar2b *)(pxMallocBlock + (isize * pi));
  }
  pxElements = pi;
  pxType = pt;
  return FF_Ok;
}

/*
 * pxTextPut(+Type, +String, +Length, +Delta, +Font, 0)
 *	integer: Type, Length, Delta, Font
 *	string: String
 */
FFInteger
pxTextPut(pt, ps, pl, pd, pf)
     FFInteger pt, pl, pd, pf;
     FFString ps;
{
  switch (pxType) {
  case PX_TextItem8:
    if (!pl)
      pxTextItem8->chars = NULL;
    else
      pxTextItem8->chars = pxVector8;
    pxTextItem8->nchars = pl;
    pxTextItem8->delta = pd;
    pxTextItem8->font = pf;
    pxTextItem8++;
    break;
  case PX_TextItem16:
    if (!pl)
      pxTextItem16->chars = NULL;
    else
      pxTextItem16->chars = pxText16;
    pxTextItem16->nchars = pl;
    pxTextItem16->delta = pd;
    pxTextItem16->font = pf;
    pxTextItem16++;
    break;
  default:
    return FF_Fail;
  }

  switch (pt) {
  case PX_String:
    if (pl) 
      if (pxType == PX_TextItem8) {
	(void)strncpy(pxVector8, ps, (int)pl);
	pxVector8 += pl;
      } else 
	return FF_Fail;		/* no 16 bit atoms */
    break;
  case PX_TextVector:
    break;
  default: 
    return FF_Fail;
  }
  return FF_Ok;
}

/*
 * pxColorGet(-Pixel, -Red, -Green, -Blue, 0)
 *	integer: Pixel, Red, Green, Blue
 */
FFInteger
pxColorGet(pp, pr, pg, pb)
     FFInteger *pp, *pr, *pg, *pb;
{
  *pp = *pr = *pg = *pb = PX_Invalid;
  
  if (pxType != PX_ColorVector)
    return FF_Fail;
  if (pxColor == NULL)
    return FF_Fail;
  *pp = pxColor->pixel;
  *pr = pxColor->red;
  *pg = pxColor->green;
  *pb = pxColor->blue;
  if (--pxElements == 0) 
    pxVectorXFree();
  else 
    pxColor++;
  return FF_Ok;
}

/*
 * pxColorPut(+Pixel, +Red, +Green, +Blue, +Flags, 0)
 *	integer: Pixel, Red, Green, Blue, Flags
 */
FFInteger
pxColorPut(pp, pr, pg, pb, pf)
     FFInteger pp, pr, pg, pb, pf;
{
  if (pxType != PX_ColorVector)
    return FF_Fail;

  pxColor->pixel = pp;
  pxColor->red = (unsigned short)(pr & 0xFFFF);
  pxColor->green = (unsigned short)(pg & 0xFFFF);
  pxColor->blue = (unsigned short)(pb & 0xFFFF);
  pxColor->flags = (unsigned char)(pf & 0xFF);
  pxColor++;
  return FF_Ok;
}

/*
 * pxListGet(-Element, -Next, 0)
 *	integer: Element, Next
 */
FFInteger
pxListGet(pe, pn)
     FFInteger *pe, *pn;
{
  *pe = *pn = PX_Invalid;

  if ((pxType != PX_List) && (pxType != PX_KeyList))
    return FF_Fail;

  if (pxList == NULL)
    return FF_Fail;
  *pe = (pxType == PX_List) ? pxList->des : pxList->data.key;
  *pn = ((pxList = pxList->next) == NULL) ? PX_End : PX_Cont;
  return FF_Ok;
}

/*
 * eof
 */
