/*
 * Copyright (C) 1989 by The Regents of the University of California.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/*
 * keyboard.c
 *
 * pxDownKeymap
 *
 * pxGetKeyboardAttribute
 * pxSetKeyboardAttribute
 * pxQueryKeyboard
 * pxSetKeyboard
 *
 * pxGetKeyboardMapping
 * pxSetKeyboardMapping
 *
 * pxGetModifierMapping
 * pxSetModifierMapping
 *
 * pxBell
 *
 * pxKeycodeToKeysym
 * pxKeysymToKeycode
 * pxRefreshMapping
 */

#include <stdio.h>
#include <X11/Xlib.h>

#include "dl.h"
#include "conn.h"
#include "ff.h"
#include "attrib.h"

/*
 * pxDownKeymap(+Connection, -Next, 0)
 *	integer: Connection, Next
 */

#define XKEY (pxAttribBuf.xkeymap[0])

FFInteger
pxDownKeymap(pc, pn)
     FFInteger pc, *pn;
{
  PxConnection *pcp;

  PX_ErrorInit("xDownKeymap/2");
  *pn = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  XQueryKeymap(pcp->xDisplay, &XKEY);
  pxMallocBlock = NULL;		/* disables free of pxMallocBlock */
  pxVector8 = &XKEY;
  pxElements = KEYMAPBYTES;
  pxType = PX_Vector8;
  *pn = PX_Cont;
  return FF_Ok;
}

/*
 * pxGetKeyboardAttribute(+Attribute, -Value, 0)
 *	integer: Attribute, Value
 */

#define XKS (pxAttribBuf.xks)

#define PX_GetAttribute(NAME, MEMBER) \
  case NAME: *pv = XKS.MEMBER; \
             break

FFInteger
pxGetKeyboardAttribute(pa, pv)
     FFInteger pa, *pv;
{
  *pv = PX_Invalid;

  switch (pa) {
    PX_GetAttribute(0, key_click_percent);
    PX_GetAttribute(1, bell_percent);
    PX_GetAttribute(2, bell_pitch);
    PX_GetAttribute(3, bell_duration);
				/* may cause problems if > 28 LEDs exist */
    PX_GetAttribute(4, led_mask);
    PX_GetAttribute(7, global_auto_repeat);
    case 8: 
      pxMallocBlock = NULL;		/* disables free of pxMallocBlock */
      pxVector8 = &(XKS.auto_repeats[0]);
      pxElements = KEYMAPBYTES;
      pxType = PX_Vector8;
      *pv = PX_Cont;
      break;
   default: PX_Error(unknown_attribute);
  }
  return FF_Ok;
}

/*
 * pxSetKeyboardAttribute(+Attribute, +Value, 0)
 *	integer: Attribute, Value
 */

#define XKC_Changed pxAttribChanged1
#define XKC_Mask pxAttribMask1
#define XKC (pxAttribBuf.xkc)

#define PX_SetAttribute(A, MEM, BIT) \
  case A: XKC.MEM = pv; \
          XKC_Mask |= BIT; \
          break;

FFInteger
pxSetKeyboardAttribute(pa, pv)
     FFInteger pa, pv;
{
  PX_ErrorInit("xSetKeyboard/3");

  switch (pa) {
    PX_SetAttribute(0, key_click_percent, KBKeyClickPercent);
    PX_SetAttribute(1, bell_percent, KBBellPercent);
    PX_SetAttribute(2, bell_pitch, KBBellPitch);
    PX_SetAttribute(3, bell_duration, KBBellDuration);
    PX_SetAttribute(4, led, KBLed);
    PX_SetAttribute(5, led_mode, KBLedMode);
    PX_SetAttribute(6, key, KBKey);
    PX_SetAttribute(7, auto_repeat_mode, KBAutoRepeatMode);
    default: PX_Error(unknown_attribute);
  }
  XKC_Changed = True;
  return FF_Ok;
}

/*
 * pxQueryKeyboard(+Connection, 0)
 *	integer: Connection
 */
FFInteger
pxQueryKeyboard(pc)
     FFInteger pc;
{
  PxConnection *pcp;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  XGetKeyboardControl(pcp->xDisplay, &XKS);
  return FF_Ok;
}

/*
 * pxSetKeyboard(+Connection, 0)
 *	integer: Connection
 */
FFInteger
pxSetKeyboard(pc)
     FFInteger pc;
{
  PxConnection *pcp;

  /* need to error init here in case pxSetKeyboardAttribute is not called */
  PX_ErrorInit("xSetKeyboard/3");

  if (PXLookupConnection(pc, &pcp)) {
    XKC_Changed = False;
    XKC_Mask = 0;
    return FF_Fail;
  }
  
   if (XKC_Changed) {
     XChangeKeyboardControl(pcp->xDisplay, XKC_Mask, &XKC);
     XKC_Changed = False;
     XKC_Mask = 0;
   }
  return FF_Ok;
}

/*
 * pxGetKeyboardMapping(+Connection, +First, +Count, -Keysyms, -Next, 0)
 *	integer: Connection, First, Count, Keysyms
 */
FFInteger
pxGetKeyboardMapping(pc, pf, pcnt, pk, pn)
     FFInteger pc, pf, pcnt, *pk, *pn;
{
  PxConnection *pcp;

  PX_ErrorInit("xGetKeyboardMapping/4");
  *pk = *pn = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  if (pf < pcp->xDisplay->min_keycode ||
      pf + pcnt - 1 > pcp->xDisplay->max_keycode)
    PX_Error("keycode out of range");
  if (pcnt) {
    pxMallocBlock = (char *)XGetKeyboardMapping(pcp->xDisplay, pf, pcnt, pk);
    if (PX_ServerError(pcp))
      return FF_Fail;
    pxVector32 = (PxVector32 *)pxMallocBlock;
    pxElements = pcnt * (*pk);
    pxType = PX_Vector32;
    *pn = PX_Cont;
  } else {
    *pk = 0;
    *pn = PX_End;
  }
  return FF_Ok;
}

/*
 * pxSetKeyboardMapping(+Connection, +First, +Keysyms, +Count, 0)
 *	integer: Connection, First, Keysyms, Count
 */
FFInteger
pxSetKeyboardMapping(pc, pf, pk, pcnt)
     FFInteger pc, pf, pk, pcnt;
{
  PxConnection *pcp;

  PX_ErrorInit("xSetKeyboardMapping/4");

  if (PXLookupConnection(pc, &pcp))
    return PXVectorFree();

  if (pcnt * pk)
    XChangeKeyboardMapping(pcp->xDisplay, pf, pk, pxMallocBlock, pcnt);
  (void)PXVectorFree();
  return FF_Ok;
}

/*
 * pxGetModifierMapping(+Connection, -Keycodes, 0)
 *	integer: Connection, Keycodes
 */
FFInteger
pxGetModifierMapping(pc, pk)
     FFInteger pc, *pk;
{
  PxConnection *pcp;

  PX_ErrorInit("xGetModifierMapping/2");
  *pk = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  if ((pxMallocBlock = (char *)XGetModifierMapping(pcp->xDisplay)) == NULL) 
    PX_Error(no_memory);

  *pk  = ((XModifierKeymap *)pxMallocBlock)->max_keypermod;
  pxVector8 = (PxVector8 *)(((XModifierKeymap *)pxMallocBlock)->modifiermap);
  pxElements = (*pk) * 8;
  pxType = PX_ModifierMap;
  return FF_Ok;
}

/*
 * pxSetModifierMapping(+Connection, +Keycodes, -Status, 0)
 *	integer: Connection, Keycodes, Status
 */
FFInteger
pxSetModifierMapping(pc, pk, ps)
	FFInteger pc, pk, *ps;
{
  PxConnection *pcp;

  PX_ErrorInit("xSetModifierMapping/3");
  *ps = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return PXVectorFree();

  ((XModifierKeymap *)pxMallocBlock)->max_keypermod = pk;
  if ((*ps = XSetModifierMapping(pcp->xDisplay, pxMallocBlock)) ==
      MappingFailed)
    return PXVectorFree();
  (void)PXVectorFree();
  return FF_Ok;
}

/*
 * pxBell(+Connection, +Percent, 0)
 *	integer: Connection, Percent
 */
FFInteger
pxBell(pc, pp)
     FFInteger pc, pp;
{
  PxConnection *pcp;

  PX_ErrorInit("xBell/2");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  XBell(pcp->xDisplay, pp);
  return FF_Ok;
}

/*
 * pxKeycodeToKeysym(+Connection, +Keycode, +Index, -Keysym, 0)
 *	integer: Connection, Keycode, Index, Keysym
 */
FFInteger
pxKeycodeToKeysym(pc, pkc, pi, pks)
     FFInteger pc, pkc, pi, *pks;
{
  PxConnection *pcp;

  PX_ErrorInit("xKeycodeToKeysym/4");
  *pks = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  *pks = XKeycodeToKeysym(pcp->xDisplay, pkc, pi);
  return FF_Ok;
}

/*
 * pxKeysymToKeycode(+Connection, +Keysym, -Keycode, 0)
 *	integer: Connection, Keysym, Keycode
 */
FFInteger
pxKeysymToKeycode(pc, pks, pkc)
     FFInteger pc, pks, *pkc;
{
  PxConnection *pcp;

  PX_ErrorInit("xKeysymToKeycode/3");
  *pkc = PX_Invalid;

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  *pkc = XKeysymToKeycode(pcp->xDisplay, pks);
  return FF_Ok;
}

#define XEV (pxAttribBuf.ev.xev)
extern PXVectorToEvent();

/*
 * pxRefreshMapping(0)
 */
FFInteger
pxRefreshMapping()
{
  PX_ErrorInit("xRefreshMapping/1");

  (void)PXVectorToEvent();
  if (XEV.type != MappingNotify)
    PX_Error("not a xMappingNotify event");

  XRefreshKeyboardMapping(&XEV);
  return FF_Ok;
}

/*
 * eof
 */
