/*
 * Copyright (C) 1989 by The Regents of the University of California.
 *
 * This software work was developed at UCLA with support in part from
 * DARPA Contract F29601-87-C-0072.
 */

/*
 * conn.c - XWIP connection routines,  a connection is what X calls a "display"
 *
 * PXLookupConnection
 * PXErrorHandler
 *
 * pxOpenConnection
 * pxCloseConnection
 *
 * pxConnections
 * pxGetConnectionAttribute
 *
 * pxScreens
 * pxGetScreenAttribute
 *
 * pxDepths
 * pxGetDepthAttribute
 *
 * pxVisuals
 * pxGetVisualAttribute
 */

#include <stdio.h>
#include <X11/Xlib.h>

#include "dl.h"
#include "conn.h"
#include "ff.h"

extern char *sprintf();		/* commented out of SUN's stdio.h */

/*
 * global DLs for Xlib structures
 *
 * pxVisualList should start at 1, so that CopyFromParent (0) can be used
 * pxScreenList should start from 0, since default_screen in the display 
 * 	structure is an int not a pointer
 * others should start from 0
 */

DL pxImageList = { NULL, sizeof(PxImage), 0},
   pxGCList = { NULL, sizeof(PxGC), 0 },
   pxVisualList = {  NULL, sizeof(PxVisual), 1},
   pxDepthList = { NULL, sizeof(PxDepth), 0},
   pxScreenList = {  NULL, sizeof(PxScreen), 0},
   pxConnectionList = { NULL, sizeof(PxConnection), 0};

/*
 * PXLookupConnection(c, pcpp)
 *
 * looks up connection 
 */
PXLookupConnection(c, pcpp)
     FFInteger c;
     PxConnection **pcpp;
{
  if (DLLookup(pxConnectionList, (int)c, (DLHeader **)pcpp))
    PX_Error(no_such_connection);
  return FF_Ok;
}

/*
 * PXErrorHandler(d, xeep)
 *
 * replaces the default X one
 */

bool pxHandlerInited = false;

PXErrorHandler(d, xeep)
     Display *d;
     XErrorEvent *xeep;
{
#define BUFSIZE 256

  char ibuf[BUFSIZE], obuf[BUFSIZE];

  XGetErrorText(d, xeep->error_code, obuf, BUFSIZE);
  (void)fprintf(stderr, "[ERROR detected by X Server: %s]\n", obuf);
  (void)sprintf(ibuf, "%d", xeep->request_code);
  XGetErrorDatabaseText(d, "XRequest", ibuf, "", obuf, BUFSIZE);
  (void)fprintf(stderr, "[ERROR generated by %s request with serial %d]\n",
	  &(obuf[2]), xeep->serial);
  pxErrorSerial = xeep->serial;
  pxServerError = true;
}

/*
 * PXCloseConnection(pcp)
 *
 * internal close connection function
 * used also to recover from open connection errors
 */

FFInteger
PXCloseConnection(pcp)
  PxConnection *pcp;
{
  PxConnectionScreen *csp;
  PxScreen *psp; PxScreenDepth *sdp;
  PxDepth *pdp; PxDepthVisual *dvp;
  PxVisual *pvp;
  PxGC *pgcp;
  int pc = pcp->des;

				/* remove screens */
  for (csp = (PxConnectionScreen *)pcp->screens.list;
       csp != NULL; csp = csp->next) {

				/* should not fail */
    (void)DLLookup(pxScreenList, csp->global, (DLHeader **)&psp);

				/* remove depths */
    for (sdp = (PxScreenDepth *)psp->depths.list;
	 sdp != NULL; sdp = sdp->next) {

				/* should not fail */
      (void)DLLookup(pxDepthList, sdp->global, (DLHeader **)&pdp);

				/* remove visuals */
      for (dvp = (PxDepthVisual *)pdp->visuals.list;
	   dvp != NULL; dvp = dvp->next) 
	DLRemove(&pxVisualList, dvp->global);
      DLDestroy(&(pdp->visuals));
      DLRemove(&pxDepthList, sdp->global);
    }
    DLDestroy(&(psp->depths));
    DLRemove(&pxScreenList, csp->global);
  }
  DLDestroy(&(pcp->screens));
  DLRemove(&pxConnectionList, pc);

				/* remove created GCs */
  while (DLSearch(pxGCList, pc, (DLHeader **)&pgcp) == DL_Ok)
    DLRemove(&pxGCList, pgcp->des);

  XCloseDisplay(pcp->xDisplay);
  return FF_Ok;
}

/*
 * pxOpenConnection(+Name, -Connection, 0)
 *	string: Name
 *		[]	value of DISPLAY environment variable
 *		string	athena:0.0
 *	integer: Connection
 */
FFInteger
pxOpenConnection(pn, pc)
     FFString pn;
     FFInteger *pc;
{
  Display *xdisp; Screen *xsp; Depth *xdp; Visual *xvp;
  PxConnection *pcp; PxConnectionScreen *csp;
  PxScreen *psp; PxScreenDepth *sdp;
  PxDepth *pdp; PxDepthVisual *dvp;
  PxVisual *pvp; 
  PxGC *pgcp;
  int des, i, j, k, td;
  char *tp;

  if (!pxHandlerInited) {
    XSetErrorHandler(PXErrorHandler);
    pxHandlerInited = true;
  }

  PX_ErrorInit("xOpenConnection/2");
  *pc = PX_Invalid;

  if (!strcmp(pn, pxNil)) {
    if ((xdisp = XOpenDisplay((char *)NULL)) == NULL)
      PX_Error("can't open default connection");
  } else if ((xdisp = XOpenDisplay(pn)) == NULL)
      PX_Error("can't open connection");
      
				/* register connection */
  if (DLAdd(&pxConnectionList, &des, (DLHeader **)&pcp)) {
    XCloseDisplay(xdisp);
    PX_Error(no_memory);
  }

  pcp->xDisplay = xdisp;
  DLCreate(&(pcp->screens), sizeof(PxConnectionScreen), 0);

				/* register screens */
  for (i = 0, xsp = xdisp->screens; i < xdisp->nscreens; i++, xsp++) {
    if (DLAdd(&(pcp->screens), &td, (DLHeader **)&csp)) {
      (void)PXCloseConnection(pcp);
      PX_Error(no_memory);
    }
    if (DLAdd(&pxScreenList, &(csp->global), (DLHeader **)&psp)) {
      DLRemove(&(pcp->screens), td);
      (void)PXCloseConnection(pcp);
      PX_Error(no_memory);
    }
    psp->xScreen = xsp;
    psp->connection = des;
    if (i == xdisp->default_screen) 
      pcp->default_screen = csp->global; /* use global descriptors */

				/* register depths */
    DLCreate(&(psp->depths), sizeof(PxScreenDepth), 0);
    for (j = 0, xdp = xsp->depths; j < xsp->ndepths; j++, xdp++) {
      if (DLAdd(&(psp->depths), &td, (DLHeader **)&sdp)) {
	(void)PXCloseConnection(pcp);
	PX_Error(no_memory);
      }
      if (DLAdd(&pxDepthList, &(sdp->global), (DLHeader **)&pdp)) {
	DLRemove(&(psp->depths), td);
	(void)PXCloseConnection(pcp);
	PX_Error(no_memory);
      }
      pdp->xDepth = xdp;

				/* register visuals */
      DLCreate(&(pdp->visuals), sizeof(PxDepthVisual), 0);
      for (k = 0, xvp = xdp->visuals; k < xdp->nvisuals; k++, xvp++) {
	if (DLAdd(&(pdp->visuals), &td, (DLHeader **)&dvp)) {
	  (void)PXCloseConnection(pcp);
	  PX_Error(no_memory);
	}
	if (DLAdd(&pxVisualList, &(dvp->global), (DLHeader **)&pvp)) {
	  DLRemove(&(pdp->visuals), td);
	  (void)PXCloseConnection(pcp);
	  PX_Error(no_memory);
	}
	pvp->xVisual = xvp;
      }
    }

    if (DLFind(pxVisualList, xsp->root_visual, (DLHeader **)&pvp)) {
      (void)PXCloseConnection(pcp);
      PX_Error("Xlib data inconsistency");
    }
    psp->root_visual = pvp->des;

    if (DLAdd(&pxGCList, &(psp->default_gc), (DLHeader **)&pgcp)) {
      (void)PXCloseConnection(pcp);
      PX_Error(no_memory);    
    }
    pgcp->xGC = xsp->default_gc;
    pgcp->conn = des;
  }

  *pc = des;
  return FF_Ok;
}

/*
 * pxCloseConnection(+Connection, 0)
 *	integer: Connection
 */
FFInteger
pxCloseConnection(pc)
     FFInteger pc;
{
  PxConnection *pcp;

  PX_ErrorInit("xCloseConnection/1");

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  return PXCloseConnection(pcp);
}

/* 
 * pxConnections(-Next)
 *	integer: Next
 */
void
pxConnections(pn)
     FFInteger *pn;
{
  if ((pxList = pxConnectionList.list) == NULL)
    *pn = PX_End;
  else {
    pxType = PX_List;
    *pn = PX_Cont;
  }
}

/* 
 * pxGetConnectionAttribute(+Connection, +Attribute, -String, -Most, -Value, 0)
 *
 *	integer: Connection, Attribute, Most, Value
 *	string: String
 *
 * splits have non-zero Upper
 * integer have zero Upper
 */

#define PX_StringValue(A, V) \
   case A: *ps = (V); \
           break
#define PX_IntegerValue(A, V) \
   case A: *pv = (V); \
           break
#define PX_SplitValue(A, V) \
   case A: if (V & pxPrecisionMask) { \
	     *pm = PX_MostSplit(V); \
             *pv = PX_LeastSplit(V); \
           } else \
             *pv = (V); \
           break

FFInteger
pxGetConnectionAttribute(pc, pa, ps, pm, pv)
     FFInteger pc, pa, *pm, *pv;
     FFString *ps;
{
  PxConnection *pcp;

  PX_ErrorInit("xQueryConnection/2");
  *pm = 0L;			/* default is integer */
  *pv = PX_Invalid;
  *ps = pxNil;		

  if (PXLookupConnection(pc, &pcp))
    return FF_Fail;

  switch (pa) {
    PX_IntegerValue(0, pcp->xDisplay->fd); 
    PX_IntegerValue(1, pcp->xDisplay->proto_major_version); 
    PX_IntegerValue(2, pcp->xDisplay->proto_minor_version); 
    PX_StringValue(3, pcp->xDisplay->vendor); 
    PX_IntegerValue(4, pcp->xDisplay->byte_order); 
    PX_IntegerValue(5, pcp->xDisplay->bitmap_unit);
    PX_IntegerValue(6, pcp->xDisplay->bitmap_pad); 
    PX_IntegerValue(7, pcp->xDisplay->bitmap_bit_order); 
    PX_IntegerValue(8, pcp->xDisplay->release); 
    PX_IntegerValue(9, pcp->xDisplay->qlen); 
    PX_SplitValue(10, pcp->xDisplay->last_request_read);
    PX_SplitValue(11, pcp->xDisplay->request);
    PX_StringValue(12, pcp->xDisplay->display_name); 
    PX_IntegerValue(13, pcp->default_screen); 
    PX_IntegerValue(14, pcp->xDisplay->min_keycode);
    PX_IntegerValue(15, pcp->xDisplay->max_keycode);
    case 16:
      if ((pxList = pcp->screens.list) == NULL) 
	*pv = PX_End;
      else {
	pxType = PX_KeyList;
	*pv = PX_Cont;
      }
      break;
    PX_IntegerValue(17, pcp->xDisplay->motion_buffer);
    default: PX_Error(unknown_attribute);
  }

  return FF_Ok;
}

/* 
 * pxScreens(-Next)
 *	integer: Next
 */
void
pxScreens(pn)
     FFInteger *pn;
{
  if ((pxList = pxScreenList.list) == NULL) 
    *pn = PX_End;
  else {
    pxType = PX_List;
    *pn = PX_Cont;
  }
}

/* 
 * pxGetScreenAttribute(+Screen, +Attribute, -Value, 0)
 *	integer: Screen, Attribute, Value
 */
FFInteger
pxGetScreenAttribute(psc, pa, pv)
     FFInteger psc, pa, *pv;
{
  PxScreen *pscp;

  PX_ErrorInit("xQueryScreen/2");
  *pv = PX_Invalid;

  if (DLLookup(pxScreenList, (int)psc, (DLHeader **)&pscp))
    PX_Error(no_such_screen);

  switch (pa) {
    PX_IntegerValue(0, pscp->connection);
    PX_IntegerValue(1, pscp->xScreen->root);
    PX_IntegerValue(2, pscp->xScreen->width);
    PX_IntegerValue(3, pscp->xScreen->height);
    PX_IntegerValue(4, pscp->xScreen->mwidth);
    PX_IntegerValue(5, pscp->xScreen->mheight);
    PX_IntegerValue(6, pscp->xScreen->root_depth);
    PX_IntegerValue(7, pscp->root_visual);
    PX_IntegerValue(8, pscp->default_gc);
    PX_IntegerValue(9, pscp->xScreen->cmap);
    PX_IntegerValue(10, pscp->xScreen->white_pixel);
    PX_IntegerValue(11, pscp->xScreen->black_pixel);
    PX_IntegerValue(12, pscp->xScreen->max_maps);
    PX_IntegerValue(13, pscp->xScreen->min_maps);
    PX_IntegerValue(14, pscp->xScreen->backing_store);
    PX_IntegerValue(15, pscp->xScreen->save_unders);
    PX_IntegerValue(16, pscp->xScreen->root_input_mask);
    case 17:
      if ((pxList = pscp->depths.list) == NULL)
        *pv = PX_End;
      else {
        pxType = PX_KeyList;
        *pv = PX_Cont;
      }
      break;
    default: PX_Error(unknown_attribute);
  }
  return FF_Ok;
}

/*
 * pxDepths(-Next)
 *	integer: Next
 */
void
pxDepths(pn)
     FFInteger *pn;
{
  if ((pxList = pxDepthList.list) == NULL)
    *pn = PX_End;
  else {
    pxType = PX_List;
    *pn = PX_Cont;
  }
}

/*
 * pxGetDepthAttribute(+Depth, +Attribute, -Value, 0)
 *	integer: Depth, Attribute, Value
 */
FFInteger
pxGetDepthAttribute(pd, pa, pv)
     FFInteger pd, pa, *pv;
{
  PxDepth *pdp;

  PX_ErrorInit("xQueryDepth/2");
  *pv = PX_Invalid;

  if (DLLookup(pxDepthList, (int)pd, (DLHeader **)&pdp))
    PX_Error(no_such_depth);

  switch (pa) {
    PX_IntegerValue(0, pdp->xDepth->depth);
    case 1:
      if ((pxList = pdp->visuals.list) == NULL)
        *pv = PX_End;
      else {
        pxType = PX_KeyList;
        *pv = PX_Cont;
      }
      break; 
    default: PX_Error(unknown_attribute);
  }
  return FF_Ok;
}

/* 
 * pxVisuals(-Next)
 *	integer: Next
 */
void
pxVisuals(pn)
     FFInteger *pn;
{
  if ((pxList = pxVisualList.list) == NULL)
    *pn = PX_End;
  else {
    pxType = PX_List;
    *pn = PX_Cont;
  }
}

/* 
 * pxGetVisualAttribute(+Visual, +Attribute, -Value, 0)
 *	integer: Visual, Attribute, Value
 */
FFInteger
pxGetVisualAttribute(pvis, pa, pv)
     FFInteger pvis, pa, *pv;
{
  PxVisual *pvp;

  PX_ErrorInit("xQueryVisual/2");
  *pv = PX_Invalid;

  if (DLLookup(pxVisualList, (int)pvis, (DLHeader **)&pvp))
    PX_Error(no_such_visual);

  switch (pa) {
    PX_IntegerValue(0, pvp->xVisual->class);
    PX_IntegerValue(1, pvp->xVisual->red_mask);
    PX_IntegerValue(2, pvp->xVisual->green_mask);
    PX_IntegerValue(3, pvp->xVisual->blue_mask);
    PX_IntegerValue(4, pvp->xVisual->bits_per_rgb);
    PX_IntegerValue(5, pvp->xVisual->map_entries);
    default: PX_Error(unknown_attribute);
  }
  return FF_Ok;
}

/*
 * eof
 */
