/* Copyright (C) 1994 Mauro Gaspari - Dipartimento di Matematica,
   University on Bologna, Italy. */

% Sicstus library supporting an indexing technique for Data Driven Rules

% Syntax of rules is 
% LHS then RHS.

% Rules are translated into:
% p(N):- LHS ! RHS.
% the compiler generate a Prolog file and then the Prolog
% compiler is called.

% working memory declaration.
% wm n1/a1, ..., nk/ak.

% Abstract goals are used to build the initial bitmaps
% abs_goal(Where,Functor,Arity,Tag).

% This is a sicstus library

:- module(fw_rules, [
           run/0,
	   run/1,
	   fw_init/0,
	   fw_compile/1,
	   fw_consult/1,
	   fw_compile/2,
	   fw_consult/2,
	   fw_assert/1,
	   fw_asserta/1,
	   fw_assertz/1,
	   fw_retract/1,
	   set_wm/1,
	   upd/1,
	   del/1,
	   fw_map/2
       ]).

:- use_module(library(lists),[append/3,reverse/2,member/2]).


% foreign functions declaration 

%foreign_file('fw_rules.o', [c_set_wm,c_set_oc,c_set_rg,c_upd_wm,c_del_wm,c_run]).
% this is asserted at compile time since the object file is custumized and 
% depende from the particular application
foreign(c_del_wm,del(+integer)).
foreign(c_upd_wm,upd(+integer)).
foreign(c_run,c_run).
foreign(c_init,c_init).
foreign(c_set_wm, c_set_wm(+integer)).
foreign(c_set_oc, c_set_oc(+integer,+integer)).
foreign(c_set_rg, c_set_rg(+integer,+integer)).

:- dynamic abs_map/3, wm_decl/0, wm_items/1, init_wm/2, init_rule/2, foreign_file/2, fw_module/1, dir/1.

:- op(1001, xfx, then).
:- op(1001, fy, wm).
:- op(900,  fy, not).

fw_consult(File):-
	format(user_error,"consulting rules ~p~n",[File]),
	prolog_flag(typein_module,M),
	compile_to_prolog(File,PFile,M),
	format(user_error,"rules compiled in ~p~n",[PFile]),
	format(user_error,"custumizing runtime ... ~n",[]),
	compile_c_file(PFile,M,normal),
        consult(PFile).

fw_consult(File,Flag):-
	format(user_error,"consulting rules ~p with ~p~n",[File,Flag]),
	prolog_flag(typein_module,M),
	compile_to_prolog(File,PFile,M),
	format(user_error,"rules compiled in ~p~n",[PFile]),
	format(user_error,"custumizing runtime ... ~n",[]),
	compile_c_file(PFile,M,Flag),
        consult(PFile).

fw_compile(File):-
	format(user_error,"compiling rules ~p~n",[File]),
	prolog_flag(typein_module,M),
	compile_to_prolog(File,PFile,M),
	format(user_error,"rules compiled in ~p~n",[PFile]),
	format(user_error,"custumizing runtime ... ~n",[]),
	compile_c_file(PFile,M,normal),
        compile(PFile),
	clean_up.

fw_compile(File,Flag):-
	format(user_error,"compiling rules ~p with ~p~n",[File,Flag]),
	prolog_flag(typein_module,M),
	compile_to_prolog(File,PFile,M),
	format(user_error,"rules compiled in ~p~n",[PFile]),
	format(user_error,"custumizing runtime ... ~n",[]),
	compile_c_file(PFile,M,Flag),
        compile(PFile),
	clean_up.


compile_to_prolog(File,PFile,M):-
	init_clean,
	rules_dir(Rdir),
	assert(user:library_directory(Rdir)),
	absolute_file_name(File,AbsFile),
	read_rules_from_file(AbsFile),
	generate_name(AbsFile,PFile,'.pl'),
	build_prolog_file(PFile,M).

% the .o file is generated dynamically in the current directory 
% and loaded issuing load_foreign_file

compile_c_file(File,M,Flag):-
	(fw_module(FM,_),!,CM=FM;
	 CM=M),
	(Flag == normal, DFLAG = '',!;
	 Flag == time, DFLAG = ' -DTIME',!;
	 Flag == debug, DFLAG = ' -DDEBUG',!;
	 DFLAG = ''),
	add_quote(CM,QCM),
	need_name(File,Name),
	atomconcat([Name,'.o'],NewName),
	(foreign_file(NewName,_),!;
	 assert(foreign_file(NewName,[c_set_wm,c_set_oc,c_set_rg,c_upd_wm,c_del_wm,c_run,c_init])),
	 sp_dir(SPDir),
	 rules_dir(RDir),
	 rule_count(RULES),
	 wm_items(N),
	 SIZE is N // 32 + 1,
	 atomconcat(['gcc -O -c -o ',NewName,
			 ' -DWMSIZE=',SIZE,
			 ' -DMODULE=',QCM,
			 ' -DNRULES=',RULES,
			 DFLAG,
			 ' -I',
			 SPDir,'/Runtime ',
			 RDir,'/fw_rules.c -n'],Command),
	 format(user_error,"~p~n",[Command]),
	 unix(system(Command)),
	 load_foreign_files([NewName],[])).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% this read rules from files and assert them building
% the mapping to create bitmaps
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

read_rules_from_file(File):- 
	prolog_flag(syntax_errors,Oldflag,fail),
	seeing(Old),
	see(File),
	  repeat,
	  read(T),
	  elaborate(T),
	  T == end_of_file,
	!,
	seen,
	see(Old),
	prolog_flag(syntax_errors,_,Oldflag).

elaborate(end_of_file).
elaborate((wm P)):-
	\+ wm_decl, !,
	assert(fw((wm P))),
	get_wm(P,0,N),
	assert(wm_items(N)),
	assert(wm_decl), !.

elaborate((wm _)):-
	wm_decl, !,
	format(user_error,"WARNING: Only one working memory is supported ~n",[]).

elaborate((_P then _Q)):-
	assert(fw((_P then _Q))),
	wm_decl,
	!.

elaborate((_P then _Q)):-
	\+ wm_decl, !,
	format(user_error,"ERROR: wm not declared ~n",[]),
	abort.

elaborate(Fact):-
	functor(Fact,Name,Arity),
	\+ built_in(Name,Arity),
	abs_map(Name,Arity,_N),!,
	assert(fw(Fact)).
elaborate((:- module(M))):-!,
	assert(fw_module(M,[])).
elaborate((:- module(M,Exp))):-!,
	assert(fw_module(M,Exp)).	
elaborate((:- Directive)):-!,
	assert(dir(Directive)).
elaborate(Code):-
	assert(code(Code)).

get_wm((Name/Arity,Q),N,K):- !,
	assert(abs_map(Name,Arity,N)),
	assert(wm_restore(Name)),
	N1 is N + 1,
	get_wm(Q,N1,K).
get_wm(Name/Arity,N,N):-
	assert(abs_map(Name,Arity,N)),
	assert(wm_restore(Name)).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Prolog code generation.
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

build_prolog_file(PFile,M):-
	open(PFile,write,Stream),
	process(Stream,M),
	close(Stream).

process(Stream,M):-
	create_module(Stream,M),
	create_directives(Stream),
	create_dynamic(Stream),
	create_wm(Stream),
	create_rules(Stream),
	write_prolog_code(Stream),
	initialize_bitmaps(Stream).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Dynamic declaration


create_module(Stream,M):-
	(fw_module(FM,Exp),!,
	format(Stream,":- module(~q,~q).~n",[FM,Exp]);
	format(Stream,":- module(~q).~n",[M])).

create_directives(Stream):-
	dir(D),
	format(Stream,":- ~q.~n",[D]),
	fail.
create_directives(_Stream).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%  Dynamic declaration

create_dynamic(Stream):-
	fw((wm P)),
	format(Stream,":- dynamic ~p.~n",[P]).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% wm compilation

create_wm(Stream):-
	fw(Fact),
	functor(Fact,Name,Arity),
	\+ built_in(Name,Arity),
	abs_map(Name,Arity,N),
	update_init_wm(N),
	format(Stream,"~q.~n",[Fact]),
	fail.
create_wm(_).

built_in(then,2).
built_in(wm,1).
built_in(not,1).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% rules compilation

create_rules(Stream):-
	assert(rule_count(0)),
	fw((P then Q)),
	retract(rule_count(N)),
	compile_rule(Stream,P,Q,N),
	N1 is N + 1,
	assert(rule_count(N1)),
	fail.
create_rules(_).

compile_rule(Stream,P,Q,N):-
	format(Stream,"p(~d):-",[N]),
	compile_lhs(P,Stream,N,RefL),
	format(Stream,"!",[]),
	compile_rhs(Q,Stream,RefL),
	format(Stream,".~n",[]).

compile_lhs(LHS,Stream,N,RefL):-
	compile_lhs1(LHS,Stream,N,[],RefL).
compile_lhs1((P,Q),Stream,N,RefL,NRefL):-!,
	compile_condition(P,Stream,N,X),
	(X == none,!,compile_lhs1(Q,Stream,N,RefL,NRefL);
	 compile_lhs1(Q,Stream,N,[X|RefL],NRefL)).
compile_lhs1(P,Stream,N,RefL,NRefL):-
	compile_condition(P,Stream,N,X),
	(X == none,!,NRefL = RefL;
	 NRefL = [X|RefL]).

compile_condition(not(P),Stream,_,none):- !,
	format(Stream,"\+ ~q,",[P]).  
        % indexing is not performed on negative conditions in this version
compile_condition({P,VarP},Stream,N,ref(VarP,M)):- !,
	(get_mapping(P,_,M),!,
	format(Stream,"clause(~q,true,~p),",[P,VarP]);
	format(user_error,"ERROR: ~p is not a wm element ~n",[P])
        ),
	update_init_rule(N,M).
compile_condition(P,Stream,N,none):-
	get_mapping(P,Arity,M),!,
	(Arity == 0, !;
	% propositional conditions are not inserted into the prolog code
	% because they have been already tested in bitwise operations
	 format(Stream,"~q,",[P])
        ),
	update_init_rule(N,M).
compile_condition(P,Stream,_,none):-   % this is a Prolog call 
	format(Stream,"~q,",[P]).

compile_rhs(RHS,Stream,RefL):-
	compile_rhs1(RHS,Stream,[],UPD,RefL),
	compile_wm_updates(Stream,UPD).
compile_rhs1((P,Q),Stream,UPD,LUPD,RefL):-!,
	compile_action(P,Stream,X,RefL),
	inv_op(X,Y),
	(X == none,!,compile_rhs1(Q,Stream,UPD,LUPD,RefL);
	 member(Y,UPD),!,delete_one(UPD,Y,NUPD),
	 compile_rhs1(Q,Stream,NUPD,LUPD,RefL);
	 compile_rhs1(Q,Stream,[X|UPD],LUPD,RefL)).
compile_rhs1(P,Stream,UPD,NUPD,RefL):-!,
	compile_action(P,Stream,X,RefL),
	inv_op(X,Y),
	(X == none,!,NUPD = UPD;
	 member(Y,UPD),!,delete_one(UPD,Y,NUPD);
	 NUPD = [X|UPD]).

compile_action(assert(P),Stream,upd(N),_RefL):-!,
	functor(P,Name,Arity),
	(abs_map(Name,Arity,N),!,
	format(Stream,",assert(~q)",[P]);
	format(user_error,"ERROR: ~p is not a wm element ~n",[P])).
compile_action(asserta(P),Stream,upd(N),_RefL):-!,
	functor(P,Name,Arity),
	(abs_map(Name,Arity,N),!,
	format(Stream,",asserta(~q)",[P]);
	format(user_error,"ERROR: ~p is not a wm element ~n",[P])).
compile_action(assertz(P),Stream,upd(N),_RefL):-!,
	functor(P,Name,Arity),
	(abs_map(Name,Arity,N),!,
	format(Stream,",assertz(~q)",[P]);
	format(user_error,"ERROR: ~p is not a wm element ~n",[P])).
compile_action(retract(P),Stream,del(N),_RefL):-!,
	functor(P,Name,Arity),
	(abs_map(Name,Arity,N),!,
	format(Stream,",retract(~q)",[P,N]);
	format(user_error,"ERROR: ~p is not a wm element ~n",[P])).
compile_action(erase(R),Stream,del(N),RefL):-
	member_samevar(ref(R,N),RefL),!,
	format(Stream,",erase(~q)",[R]).
compile_action(P,Stream,none,_RefL):-!,
	format(Stream,",~q",[P]).

compile_wm_updates(_Stream,[]).
compile_wm_updates(Stream,[H|T]):-
	format(Stream,",~p",[H]),
	compile_wm_updates(Stream,T).
	
inv_op(del(X),upd(X)):-!.
inv_op(upd(X),del(X)):-!.
inv_op(none,_).

member_samevar(ref(V,N),[ref(VH,N)|_]):-V == VH,!.
member_samevar(El,[_|T]):-
	member_samevar(El,T).

delete_one([], _, []).
delete_one([Element|Tail], Element, Tail) :- !.
delete_one([Head|Tail], Element, [Head|Rest]) :-
	delete_one(Tail, Element, Rest).


%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Prolog compilation

write_prolog_code(Stream):-
	code(Code),
	format(Stream,"~q.~n",[Code]),
	fail.
write_prolog_code(_Stream).
	

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% utilities

update_init_wm(N):-
	\+ init_wm(N,_),!,
	assert(init_wm(N,1)).
update_init_wm(N):-
	retract(init_wm(N,K)),!,
	K1 is K + 1,
	assert(init_wm(N,K1)).

update_init_rule(N,M):-
	\+ init_rule(N,M),!,
	assert(init_rule(N,M)).
update_init_rule(_N,_M).

get_mapping(P,Arity,M):-
	functor(P,Name,Arity),
	abs_map(Name,Arity,M).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% initialize bitmaps
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

initialize_bitmaps(Stream):-
	format(Stream,"fw_rules:init_system:-!",[]),
	build_initial_wm(Stream),
	build_rule_bitmaps(Stream),
	format(Stream,".~n",[]).

build_initial_wm(Stream):-
	init_wm(M,N),
	format(Stream,",fw_rules:c_set_wm(~d)",[M]),
	format(Stream,",fw_rules:c_set_oc(~d,~d)",[M,N]),
	fail.
build_initial_wm(_).

build_rule_bitmaps(Stream):-
	init_rule(N,M),
	format(Stream,",fw_rules:c_set_rg(~d,~d)",[N,M]),
	fail.
build_rule_bitmaps(_Stream).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Clean predicates
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

init_clean:-         % issued before the compiler
	retractall(fw(_)),
	retractall(code(_)),
	retractall(rule_count(_)),
	retractall(abs_map(_,_,_)),
	retractall(wm_decl),
	retractall(init_rule(_,_)),
	retractall(init_wm(_,_)),
	retractall(wm_items(_)),
	retractall(fw_module(_,_)),
	retractall(delt(_)),
	abolish(p),
	abolish(init_system),
	clean_wm.

clean_wm:-
	retract(wm_restore(Name)),
	prolog_flag(typein_module,M),
	abolish(M:Name),
	fail.
clean_wm.

clean_up:-           % issued after the compiler
	retractall(code(_)),
	retractall(delt(_)),
	retractall(fw(_)),
	retractall(init_rule(_,_)),
	retractall(init_wm(_,_)),
	retractall(wm_items(_)),
	retractall(rule_count(_)),
	retractall(wm_decl).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% built-in predicates
% fw_assert and fw_retract
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fw_assert(P):-
	get_mapping(P,_,N),
	assert(P),
	upd(N),!.
fw_assert(P):-
	format(user_error,"WARNING: cannot perform wm upd on ~p",[P]),
	assert(P).

fw_asserta(P):-
	get_mapping(P,_,N),
	asserta(P),
	upd(N),!.
fw_asserta(P):-
	format(user_error,"WARNING: cannot perform wm upd on ~p",[P]),
	asserta(P).

fw_assertz(P):-
	get_mapping(P,_,N),
	assertz(P),
	upd(N),!.
fw_assertz(P):-
	format(user_error,"WARNING: cannot perform wm upd on ~p",[P]),
	assertz(P).

fw_retract(P):-
	get_mapping(P,_,N),
	retract(P),
	del(N).
fw_retract(P):-
	format(user_error,"WARNING: cannot perform wm del on ~p",[P]),
	retract(P).

run:-
    prolog_flag(unknown,Old,fail),
    c_init,
    init_system,
    c_run,
    prolog_flag(unknown,_,Old).

run(Term):-
    prolog_flag(unknown,Old,fail),
    c_init,
    init_system,
    c_run_check(Term),
    prolog_flag(unknown,_,Old).

fw_init:-
	clean_wm.

fw_map(P,N):-
	get_mapping(P,_,N).

% this is used to call the condition from Prolog

callfw(Term):-
    prolog_flag(typein_module,M),
    call(M:Term).

%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% predicates to manage unix pathnames and shell commands
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

generate_name(File,PFile,PostFix):-
	path_structure(File,Dir,Name,_),
	atomconcat([Dir,Name,PostFix],PFile).

path_structure(Path,Dir,Name,Prefix):-
        name(Path,PathName),
        reverse(PathName,PathRevName),
        get_structure(PathRevName,DirRevName,NameRevName,PrefixRevName),
        reverse(DirRevName,DirName),
	reverse(NameRevName,NameName),
        reverse(PrefixRevName,PrefixName),
	name(Name,NameName),
	name(Prefix,PrefixName),
        name(Dir,DirName).

% find_dir/4
% find / in a char list

get_structure(Rev,DirRev,NameRev,PrefixRev):-
	find_prefix_now(Rev,RestRev,PrefixRevA,Flag),
	(Flag == no,
	 NameRev = PrefixRevA, 
	 PrefixRev = [],
	 DirRev = RestRev, !;
	 PrefixRev = PrefixRevA,
	 RestRev = [_|RestRevc],
	 find_name_now(RestRevc,DirRev,NameRev)).

find_prefix_now([47|T],[47|T],[],no):- !.  % acc is nil
find_prefix_now([46|T],[46|T],[],yes):- !.
find_prefix_now([H|T],Tnew,[H|Tprefix],Flag):- 
	find_prefix_now(T,Tnew,Tprefix,Flag).

find_name_now([47|T],[47|T],[]):- !.
find_name_now([H|T],Tnew,[H|Tname]):- 
	find_name_now(T,Tnew,Tname).

% This is the BIN Prolog compatible atomconcat.

atomconcat([],''):- !.
atomconcat([H|T],A):-
	atomconcat(T,A1),
	name(H,H_List),
	name(A1,A1_List),
	append(H_List,A1_List,A_List),
	name(A,A_List).

need_name(PFile,Name):-
	path_structure(PFile,_,Name,_).

add_quote(Atom,QAtom):-
	name(Atom,AName),
	append(AName,[34,39],T),
	name(QAtom,[39,34|T]).

