program ProTest;

{$R+}

{ See description at end. }

const  maxfilesize = 50000;  { Number of characters in file }
       mapsize     = 190;    { Number of entries in map }
       namesize    = 32;     { Length of variable name or functor }
       return      = ^M;
       newline     = ^J;
       tab         = ^I;
       eofmark     = ^Z;
       blank       = ' ';
       quote       = '''';

type   cardinal = word;   { Nonnegative integer }
       symbol   = string[namesize];

type   mapentry = record
                    name: symbol;          { The atom or variable   }
                    freq,                  { Number of occurrences  }
                    line: cardinal         { Line where first found }
                  end;

type   map      = record
                    size: cardinal;        { How many items in map  }
                    item: array[1..mapsize] of mapentry
                  end;


{ Globals }

var       inf,                       { Input file  }
          outf:          text;       { Output file }
          infname,                   { File name }
          outfname:      string[64]; { File name }

          wholefile:     array[1..maxfilesize] of char;
          position:      cardinal;   { Position in wholefile }
          actualsize:    cardinal;   { Size of wholefile }

          vmap,                      { Variable map }
          fmap:          map;        { Functor/atom map }
          ante,                      { Antepenultimate character }
          prev,                      { Previous character }
          this:          char;       { Current character }

          currentname:   symbol;     { Symbol being processed }
          currentline:   cardinal;   { Line counter }


{ UTILITY PROCEDURES }

procedure Header(var f:text);
begin
 writeln(f,'ProTest -- the PROlog TESTer     Copyright 1988 Michael A. Covington');
 writeln(f,'Version of 06/10/88            Advanced Computational Methods Center');
 writeln(f,'                             University of Georgia, Athens, GA 30602');
 writeln(f);
end;

procedure Incr(var i:cardinal);
begin
  i := i+1
end;

procedure Warn(line:cardinal; msg:string);
begin
  writeln(outf,'Line ',line,': ',msg)
end;

function StringOf(x:cardinal):string;
var
  s: string[64];
begin
  str(x,s);
  StringOf := s
end;


{ FILE INPUT PROCEDURES }

procedure LoadFile;
  { Copies entire file into array "wholefile". }
begin
  actualsize := 0;
  while (actualsize < maxfilesize) and not eof(inf) do
    begin
      Incr(actualsize);
      read(inf,wholefile[actualsize])
    end;
  if not eof(inf) then
    begin
      warn(0,'FILE TOO BIG; PARTIAL FILE READ. Max is '
                            + StringOf(maxfilesize) + ' bytes')
    end;
  position := 0
end;

procedure ReadChar;
  { Advances one character in file, updates ante, prev, this }
  { Does not do file i/o; uses wholefile instead. }
begin
  if this = newline then Incr(currentline);
  ante := prev;
  prev := this;
  { read(inf,this) }
  Incr(position);
  this := wholefile[position]
end;

function EndOfFile: boolean;
begin
  EndOfFile := (position = actualsize)
end;


{ MAP-CONSTRUCTING PROCEDURES }

procedure InitMap(var m:map);
begin
  m.size := 0
end;

procedure AddToMap(var m:map; n:symbol);
var
  i: cardinal;
  found: boolean;
begin
  { If n is already in m, increment its frequency, else add it. }
  found := false;
  i := 0;
  while (i < m.size) and (i < mapsize) and not found do
    begin
      i:=i+1;
      found := (m.item[i].name = n)
    end;
  if found then
    Incr(m.item[i].freq)
  else
    { Add new item to map }
    begin
      Incr(m.size);
      { Note that m.size keeps being incremented...
        it can go much higher than mapsize. }
      if m.size = mapsize then
        Warn(currentline,'REACHED MEMORY LIMIT. Some material will be ignored');
      if m.size <= mapsize then
        begin
          m.item[m.size].name := n;
          m.item[m.size].freq := 1;
          m.item[m.size].line := currentline
        end
    end
end;

procedure DisplayMap(m:map);
var
  i: cardinal;
begin
  writeln(outf,'Atom or':namesize,
               blank:3,
               'Occur-':7,
               blank:3,
               'First occurs');
  writeln(outf,'functor':namesize,
               blank:3,
               'rences':7,
               blank:3,
               'on line');
  writeln(outf);
  for i:=1 to m.size do
   begin
    write(outf,m.item[i].name:namesize,
               blank:1,
               m.item[i].freq:7);
    if m.item[i].freq = 1 then
      write(outf,'*')
    else
      write(outf,' ');
    writeln(outf,blank:4,
                 m.item[i].line)
  end
end;

procedure CheckMap(m:map; msg:string);
  { Checks map for singletons, writes messages }
var
  i: cardinal;
begin
  if m.size > mapsize then
    m.size := mapsize;
  for i:=1 to m.size do
    if m.item[i].freq = 1 then
      Warn(m.item[i].line,
           m.item[i].name + ' -- ' + msg)
end;

procedure SortMap(var m:map);
  { Puts entries in order by name }
var
  i,pos: cardinal;
  value: mapentry;
begin
  { Normalize m.size, which may be too high }
  if m.size > mapsize then
    m.size := mapsize;
  { Perform an insertion sort }
  for i:=2 to m.size do
    begin
      value := m.item[i];
      pos := i;
      while (pos > 1) and (m.item[pos-1].name > value.name) do
        begin
          m.item[pos] := m.item[pos-1];
          pos := pos-1
        end;
      m.item[pos] := value
    end
end;



{ LOW-LEVEL SYNTAX ANALYSIS PROCEDURES }

function LowerCase(c:char): boolean;
begin
  lowercase := (c >= 'a') and (c <= 'z')
end;

function UpperCase(c:char): boolean; { includes '_' }
begin
  uppercase := (c = '_') or
               ((c >= 'A') and (c <= 'Z'))
end;

function Alphanumeric(c:char): boolean;
begin
  alphanumeric := lowercase(c) or
                  uppercase(c) or
                  ((c >= '0') and (c <= '9'))
end;


procedure CompleteName;
  { Called when this is the first character of
    a variable name or atom. Creates currentname
    and exits with this = the character following
    the variable name or atom. }
begin
  currentname := '';
  while Alphanumeric(this) and not EndOfFile do
    begin
      currentname := currentname + this;
      ReadChar
    end
end;

function EndOfComment: boolean;
begin
  EndOfComment := (prev = '*') and (this = '/')
end;

function Arity: cardinal;
  { Assuming that the current character is the first }
  { character following a functor, this function     }
  { determines its arity.                            }
var
  argcount,psave,csave: cardinal;
begin
  argcount := 0;
  { Save current position }
  psave := position;
  csave := currentline;
  { Gobble any blanks between functor and arglist }
  while (this in [blank,tab,return,newline]) and not EndOfFile do
    ReadChar;
  { If we have found an argument list, work through it }
  if this = '(' then
    begin
      Incr(argcount);
      while (this <> ')') and not EndOfFile do
        begin
          ReadChar;
          if this = ',' then
            Incr(argcount)
          else if this = '(' then    { Skip parenthesized expression }
            begin
              while (this <> ')') and not EndOfFile do ReadChar;
              if this=')' then this := blank;
                { Trick to prevent mistake because the higher
                  level loop is also looking for ')' and we
                  shouldn't use the same ')' to satisfy both }
            end
          else if this = '[' then    { Skip bracketed expression }
            begin
              while (this <> ']') and not EndOfFile do ReadChar
            end
          else if this = quote then  { Skip quoted string }
            begin
              this := blank;
              while (this <> quote) and not EndOfFile do ReadChar;
              this := blank
            end
          else if this = '"' then    { Skip double quoted string }
            begin
              this := blank;
              while (this <> '"') and not EndOfFile do ReadChar;
              this := blank
            end
          else if (prev = '/') and (this = '*') then   { Skip comment }
            begin
              while not (EndOfComment or EndOfFile) do ReadChar
            end
          else if (this = '%') then { Skip comment to end of line }
            begin
              while not ((this = newline) or EndOfFile) do ReadChar
            end
          { Other characters call for no action }
        end;
        { Restore position }
        position := psave;
        currentline := csave;
        { If we ran off EOF, say so }
        if EndOfFile then
          warn(currentline,'Could not find end of argument list');
      end;
  Arity := argcount
end;

{ HIGH-LEVEL SYNTAX ANALYSIS PROCEDURES }
{ Each of these calls ReadChar at least once. }

procedure Comment;
  { Called when prev = '/' and this = '*' }
begin
  while not (EndOfFile or EndOfComment) do
    begin
      ReadChar;
      if (prev = '/') and (this = '*') then
        Warn(currentline,'Comment within comment');
      if (prev = ':') and (this = '-') then
        Warn(currentline,'Comment contains ":-"')
    end;
  if EndOfFile and not EndOfComment then
    Warn(currentline,'Comment extends past end of file')
end;

procedure PercentComment;
  { Called when this = '%', comment to end of line }
begin
  while not (EndOfFile or (this = newline)) do
    begin
      ReadChar;
      if (prev = '/') and (this = '*') then
        Warn(currentline,'Comment within comment');
      if (prev = ':') and (this = '-') then
        Warn(currentline,'Comment contains ":-"')
    end
end;

procedure QuotedString;
  { Called when this = quote }
var
  warned: boolean;
begin
  warned := false;
  this := blank; { Trick to prevent self-triggering }
  while (this <> quote) and not EndOfFile do
    begin
      ReadChar;
      if (this = newline) and not warned then
        begin
          Warn(currentline,'Quoted string extends past end of line');
          warned:=true
        end
    end;
  this := blank; { Trick to prevent self-triggering }
  if EndOfFile and not (this = quote) then
    Warn(currentline,'Quoted string extends past end of file')
end;

procedure DoubleQuotedString;
  { Called when this = double quote }
var
  warned: boolean;
begin
  warned := false;
  this := blank; { Trick to prevent self-triggering }
  while (this <> '"') and not EndOfFile do
    begin
      ReadChar;
      if (this = newline) and not warned then
        begin
          Warn(currentline,'Quoted string extends past end of line');
          warned:=true
        end
    end;
  this := blank; { Trick to prevent self-triggering }
  if EndOfFile and not (this = '"') then
    Warn(currentline,'Quoted string extends past end of file')
end;

procedure Variable;
  { Called when this = first char of variable name }
begin
  CompleteName;
  if currentname <> '_' then
     AddToMap(vmap,currentname)
end;

procedure Atom;
  { Called when this = first char of atom or functor }
begin
  CompleteName;
  AddToMap(fmap,currentname+'/'+StringOf(Arity))
end;

procedure NewClause;
  { Called at end of clause }
begin
  CheckMap(vmap,'Variable occurs only once in clause');
  InitMap(vmap);
  ReadChar
end;


{ TOP-LEVEL CONTROL PROCEDURES }

procedure InitGlobals;
begin
     this := blank;
     prev := blank;
     currentline := 1;
     InitMap(vmap);
     InitMap(fmap)
end;

procedure ScanWholeFile;
begin
  { First time through, prev and this are both blanks }
  while not EndOfFile do
    begin
      if (prev = '/') and (this = '*') then
         Comment
      else if this = '%' then
         PercentComment
      else if this = quote then
         QuotedString
      else if this = '"' then
         DoubleQuotedString
      else if LowerCase(this) then
         Atom
      else if UpperCase(this) then
         Variable
      else if (ante <> '.') and  { To avoid treating =.. as end of clause }
              (prev = '.') and
              (this in [blank,return,newline,tab,eofmark]) then
         NewClause
      else
         ReadChar
     end;
   CheckMap(fmap,'Occurs only once in whole file')
end;


procedure Spiel;
begin
 Header(output);
 writeln('Anyone may distribute this program free of charge.');
 writeln;
 writeln('ProTest finds some of the most common typing errors in Prolog');
 writeln('programs by pointing out the following conditions:');
 writeln(' * Variable used only once in a clause (check spelling or use ''_'' )');
 writeln(' * Atom or functor used only once in a program');
 writeln(' * Quoted string extending past end of line');
 writeln(' * Comment or quoted string extending past end of file');
 writeln(' * Comment containing ''/*'' or '':-'' (previous comment not closed?)');
 writeln('Not everything that ProTest objects to is actually an error.');
 writeln;
 writeln('Command examples:');
 writeln('   protest xxx.pro               -- Test xxx.pro, output on screen');
 writeln('   protest xxx.pro prn           -- Same, output on printer');
 writeln('   protest xxx.pro yyy.zzz       -- Same, output on file yyy.zzz');
 writeln('If output is not sent to the screen, a map of functors is generated.');
 writeln;
 writeln('Hint for computational linguists: ProTest can be used with GULP.');
 writeln('Put natural-language words in quotes (''donkey'', not donkey) so that');
 writeln('ProTest will not keep saying ''Atom or functor occurs only once''.');
 halt
end;

{ MAIN PROGRAM }

var
  i: cardinal;

begin
  if ParamCount = 0 then
    Spiel;

  infname := ParamStr(1);
  for i:=1 to length(infname) do infname[i]:=upcase(infname[i]);

  if ParamCount = 2 then
    outfname := ParamStr(2)
  else
    outfname := 'CON';
  for i:=1 to length(outfname) do outfname[i]:=upcase(outfname[i]);

  assign(inf,infname);   reset(inf);
  assign(outf,outfname); rewrite(outf);

  Header(outf);
  writeln(outf,'Processing file ',infname);
  writeln(outf);

  LoadFile;
  InitGlobals;
  ScanWholeFile;

  if outfname <> 'CON' then
    begin
      writeln(outf);
      writeln(outf,'Map of functors and atoms');
      SortMap(fmap);
      writeln(outf);
      DisplayMap(fmap)
    end;

  if outfname = 'PRN' then
    write(outf,^L)
end.