(* Counts characters, lines and words in one or several files. *)

let chars = ref 0
and words = ref 0
and lines = ref 0
;;

type state = Inside_word | Outside_word;;

let count_channel in_channel =
  let rec count status =
    let c = input_char in_channel in
    incr chars;
    match c with
      `\n` ->
        incr lines; count Outside_word
    | ` ` | `\t` ->
        count Outside_word
    | _ ->
        if status = Outside_word then begin incr words; () end;
        count Inside_word
  in
    try
      count Outside_word
    with End_of_file ->
      ()
;;

try
  if vect_length sys__command_line <= 1 then
    count_channel std_in                (* No command-line arguments *)
  else
    for i = 1 to vect_length sys__command_line - 1 do
      let ic = open_in sys__command_line.(i) in
        count_channel ic;
        close_in ic
    done;
  print_int !chars; print_string " characters, ";
  print_int !words; print_string " words, ";
  print_int !lines; print_string " lines\n";
  exit 0
with sys__Sys_error s ->
  print_string "I/O error: "; print_string s;
  print_newline();
  exit 1
;;
