/* unix.c : appels systeme Unix */

#include <fcntl.h>
#include "../Include/z2k2.h"
#include "fail.h"

obj_t unix_mode__fprdonly=MLINT(O_RDONLY), /* ML */
      unix_mode__fpwronly=MLINT(O_WRONLY),
      unix_mode__fprdwr=MLINT(O_RDWR),
      unix_mode__fpcreat=MLINT(O_CREAT),
      unix_mode__fptrunc=MLINT(O_TRUNC),
      unix_mode__fpexcl=MLINT(O_EXCL),
      unix_rights__fpread=MLINT(4),
      unix_rights__fpwrite=MLINT(2),
      unix_rights__fpexec=MLINT(1),
      unix_rights__fpowner=MLINT(01),
      unix_rights__fpgroup=MLINT(010),
      unix_rights__fpothers=MLINT(0100),
      unix_rights__fpeverything=MLINT(7),
      unix_rights__fpeverybody=MLINT(0111);

char * 
#ifdef __STDC__
error_message(int err)
#else
error_message(err)
     int err;
#endif
{ extern int sys_nerr;
  extern char* sys_errlist[];

  if (err < 0 || err >= sys_nerr)
    return "unknown error";
  else
    return sys_errlist[err];
}

obj_t     /* ML */
#ifdef __STDC__
os_error_message(obj_t errcode)
#else
os_error_message(errcode)
     obj_t errcode;
#endif
{ char *mes = error_message(CINT(errcode));

  return alloc_string(strlen(mes),mes);
}

extern int errno;

obj_t                /* ML */
#ifdef __STDC__
os_errno(void) 
#else
os_errno() 
#endif
{
  return MLINT(errno);
}

obj_t          /* ML */
#ifdef __STDC__
unix_exit(obj_t status)
#else
unix_exit(status)
     obj_t status;
#endif
{
#ifdef MONITOR_GC
  _report_execution_ends();
#endif    
  exit(CINT(status));
}

obj_t  /* ML */
#ifdef __STDC__
unix_open(obj_t path, obj_t flags, obj_t mode)
#else
unix_open(path, flags, mode)
     obj_t path, flags, mode;
#endif
{ char buf[1024];
  ulint len = CINT(STRING_LENGTH(path));

  if (len>=1024)
    failwith("unix_open: path too long");
  bcopy(STRING(path), buf, len);
  buf[len] = '\0';
  return MLINT(open(buf, CINT(flags), CINT(mode)));
}

obj_t           /* ML */
#ifdef __STDC__
unix_close(obj_t des) 
#else
unix_close(des) 
     obj_t des;
#endif
{
  return MLINT(close (CINT(des)));
}

obj_t  /* ML */
#ifdef __STDC__
unix_read(obj_t d, obj_t buf, obj_t offset, obj_t nbytes)
#else
unix_read(d, buf, offset, nbytes)
     obj_t nbytes, offset, buf, d;
#endif
{
  return MLINT(read (CINT(d), &BYTE(buf, CINT(offset)), CINT(nbytes)));
}

obj_t  /* ML */
#ifdef __STDC__
unix_write(obj_t d, obj_t buf, obj_t offset, obj_t nbytes)
#else
unix_write(d, buf, offset, nbytes)
     obj_t nbytes, offset, buf, d;
#endif
{
  return MLINT(write (CINT(d), &BYTE(buf, CINT(offset)), CINT(nbytes)));
}

obj_t /* ML */
#ifdef __STDC__
unix_lseek(obj_t des, obj_t offset, obj_t whence)
#else
unix_lseek(des, offset, whence)
     obj_t whence, offset, des;
#endif
{
  return MLINT(lseek (CINT(des), CINT(offset), CINT(whence)));
}

obj_t          /* ML */
#ifdef __STDC__
unix_unlink(obj_t name)
#else
unix_unlink(name)
     obj_t name;
#endif
{ char buf[1024];
  ulint len = CINT(STRING_LENGTH(name));

  if (len>=1024)
    failwith("unix_unlink: name too long");
  bcopy(STRING(name),buf,len);
  buf[len] = '\0';
  return MLINT(unlink(buf));
}

/* Ce qui suit n'est en general defini que sur les vraies Unix boxes. */

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/times.h>
#include <sys/param.h>

obj_t  /* ML */
#ifdef __STDC__
unix_truncate(obj_t path, obj_t length)
#else
unix_truncate(path, length)
     obj_t path, length;
#endif
{ char buf[1024];
  ulint len = CINT(STRING_LENGTH(path));

  if (len>=1024)
    failwith("unix_truncate: path too long");
  bcopy(STRING(path),buf,len);
  buf[len] = '\0';
  return MLINT(truncate(buf, CINT(length)));
}

static obj_t 
#ifdef __STDC__
stat_aux(struct stat *buf)
#else
stat_aux(buf)
     struct stat * buf;
#endif
{
  obj_t v=tuple_alloc(11);
  FIELD (v, 0) = MLINT (buf->st_dev);
  FIELD (v, 1) = MLINT (buf->st_ino);
  FIELD (v, 2) = MLINT (buf->st_mode);
  FIELD (v, 3) = MLINT (buf->st_nlink);
  FIELD (v, 4) = MLINT (buf->st_uid);
  FIELD (v, 5) = MLINT (buf->st_gid);
  FIELD (v, 6) = MLINT (buf->st_rdev);
  FIELD (v, 7) = MLINT (buf->st_size);
  FIELD (v, 8) = MLINT (buf->st_atime);
  FIELD (v, 9) = MLINT (buf->st_mtime);
  FIELD (v, 10) = MLINT (buf->st_ctime);
  return v;
}

obj_t            /* ML */
#ifdef __STDC__
unix_stat(obj_t path)
#else
unix_stat(path)
     obj_t path;
#endif
{ struct stat stat_buf;
  char buf[1024];
  ulint len = CINT(STRING_LENGTH(path));

  if (len>=1024)
    failwith("unix_stat: path too long");
  bcopy(STRING(path),buf,len);
  buf[len] = '\0';
  if (stat(buf, &stat_buf) == -1)
    os_error();
  return stat_aux(&stat_buf);
}

obj_t            /* ML */
#ifdef __STDC__
unix_fstat(obj_t fd)
#else
unix_fstat(fd)
     obj_t fd;
#endif
{ struct stat stat_buf;

  if (fstat(CINT(fd), &stat_buf) == -1)
    os_error();
  return stat_aux(&stat_buf);
}

obj_t    /* ML */
#ifdef __STDC__
unix_link(obj_t name1, obj_t name2)
#else
unix_link(name1, name2)
     obj_t name1, name2;
#endif
{ char buf1[1024], buf2[1024];
  ulint len1 = CINT(STRING_LENGTH(name1));
  ulint len2 = CINT(STRING_LENGTH(name2));

  if (len1>=1024 || len2>=1024)
    failwith("unix_link: name too long");
  bcopy(STRING(name1),buf1,len1);
  buf1[len1] = '\0';
  bcopy(STRING(name2),buf2,len2);
  buf2[len2] = '\0';
  return MLINT(link(buf1, buf2));
}

obj_t                /* ML */
#ifdef __STDC__
unix_pipe(void)
#else
unix_pipe()
#endif
{ int fd[2];
  obj_t res;

  if (pipe(fd) == -1)
    os_error();
  res = tuple_alloc(2);
  FIELD(res, 0) = MLINT(fd[0]);
  FIELD(res, 1) = MLINT(fd[1]);
  return res;
}

obj_t               /* ML */
#ifdef __STDC__
unix_fork(void) 
#else
unix_fork() 
#endif
{
  return MLINT(fork());
}

obj_t     /* ML */
#ifdef __STDC__
unix_execv(obj_t name, obj_t args)
#else
unix_execv(name, args)
     obj_t name, args;
#endif
{ char ** argv;
  int argc, i;
  ulint len = CINT(STRING_LENGTH(name))+1;
  char *p;

  argc = CINT(VECSIZE(args));
  for (i=0; i<argc; i++)
    len += CINT(STRING_LENGTH(FIELD(args,i+1)))+1;
  argv = (char**)STRING(make_string(len+(argc+1)*sizeof(obj_t)));
  p = (char*)&argv[argc+1];
  for (i = 0; i < argc; i++)
  { argv[i] = p;
    len = CINT(STRING_LENGTH(FIELD(args,i+1)));
    bcopy(STRING(FIELD(args, i+1)),p,len);
    p += len;
    *p++ = '\0';
  }
  argv[argc] = NULL;
  len = CINT(STRING_LENGTH(name));
  bcopy(STRING(name),p,len);
  p[len] = '\0';
  execv(p, argv);
/*  free(argv) */
/*  return MLVOID; */
}

obj_t               /* ML */
#ifdef __STDC__
unix_wait(void)
#else
unix_wait()
#endif 
{ int status, pid;
  obj_t res;

  status = 0;
  pid = wait(&status);
  res = tuple_alloc(2);
  FIELD(res, 0) = MLINT(pid);
  FIELD(res, 1) = MLINT(status);
  return res;
}

obj_t               /* ML */
#ifdef __STDC__
unix_times(void)
#else
unix_times()
#endif
{ obj_t res,t[5];
  struct tms buffer;
  long real_time;
  int i;
  
#ifndef HZ
#define HZ 60
#endif

  real_time = times(&buffer);
  t[0] = alloc_float((double) real_time / HZ);
  t[1] = alloc_float((double) buffer.tms_utime / HZ);
  t[2] = alloc_float((double) buffer.tms_stime / HZ);
  t[3] = alloc_float((double) buffer.tms_cutime / HZ);
  t[4] = alloc_float((double) buffer.tms_cstime / HZ);
  res = tuple_alloc(5);
  for (i = 0; i < 5; i++)
    FIELD(res, i) = t[i];
  return res;
}
