/***************************************************************************
  Copyright (C) Nitsan Seniak 1989, 1990

  This file is part of the K2 compiler.
  Permission to copy this software, in whole or in part, to use this
  software for any lawful noncommercial purpose, and to redistribute
  this software is granted subject to the restriction that all copies
  made of this software must include this copyright notice in full.
  The author(s) makes no warranties or representations of any kind, either
  express or implied, including but not limited to implied warranties
  of merchantability or fitness for any particular purpose.
  All materials developed as a consequence of the use of this
  software shall duly acknowledge such use, in accordance with the usual
  standards of acknowledging credit in research.
 ***************************************************************************/ 

/**********************************************************************
 *                                                                    *
 *			   UTILITAIRES                                *
 *                                                                    *
 **********************************************************************
 *
 * Ce module contient les fonctions d'allocation temporaire, de gestion
 * des listes, et de gestion de la table des symboles, ainsi que
 * quelques petites fonctions utilitaires.
 *
 **********************************************************************/


#include <stdlib.h>
#include <stdio.h>
#include <setjmp.h>
#include <string.h>
#include <malloc.h>
#include "util.h"
#include "objects.h"
#include "yystype.h"
#include "k2.h"

static void alloc_live_page(void);
static unsigned hash(char *,  struct hash_table *);
static int khash (char *);
static void init_allocl(void);
static void free_all(void);


/**********************************************************************
 *
 *                     Allocation temporaire
 *
 **********************************************************************
 *
 * Types :
 *
 *	page_desc	descripteur de page.
 *
 * Fonctions :
 *
 * 	init_allocl	initialise le systeme d'initialisation.
 * 	allocl		comme malloc, mais alloue temporairement.
 * 	free_all	libere la memoire allouee par allocl.
 * 	allocp		comme malloc, alloue de facon permanente.
 * 
 **********************************************************************/


#define PAGESIZE		4096

struct page_desc {
     void *addr;		/* L'adresse de la zone memoire. */
     unsigned left;		/* La place qui reste. */
     struct page_desc *next;	/* Page suivante de la liste. */
};

static struct page_desc *free_page = NULL; /* Liste des pages libres */
static struct page_desc *live_page = NULL; /* Liste des page occupees */

typedef long int align_type;

void *allocl(unsigned s)
{
     unsigned need = ((s-1)/sizeof(align_type)+1)*sizeof(align_type);

     if (need > PAGESIZE)
       return allocp(s);
     if (live_page->left < need) alloc_live_page();

     live_page->left -= need;
     return (void *)(((char *)(live_page->addr))+live_page->left);
}

static void free_all(void)
{
     while (live_page != NULL)
       {
	    struct page_desc *d;

	    d = live_page->next;
	    live_page->next = free_page;
	    free_page = live_page;
	    live_page = d;
       }
}

static void alloc_live_page(void)
{

     if (free_page != NULL)
       {
	    struct page_desc *new;
	    
	    new = free_page;
	    free_page = free_page->next;
	    new->next = live_page;
	    live_page = new;
       }
     else
       {
	    struct page_desc *new = allocp(sizeof(struct page_desc));
	    void *addr = allocp(PAGESIZE);
	    
	    new->addr = addr;
	    new->next = live_page;
	    live_page = new;
       }

     live_page->left = PAGESIZE /*-sizeof(align_type)*/;
}

void *allocp(unsigned s)
{
     void *addr = malloc(s);

     if (addr == NULL)
       {
	    fprintf(Err, "%s panic: not enough room for malloc\n", Me);
	    exit(3);
       }

     return addr;
}

static void init_allocl(void)
{
     alloc_live_page();
}



/**********************************************************************
 *
 *                     Traitement des listes
 *
 **********************************************************************
 *
 * Fonctions :
 *
 *	cons		sans commentaires.
 *
 *	emptyq		construit une queue vide.
 *	addq		ajoute un element en fin de queue.
 *	qlist		rend la liste des elements d'une queue.
 *
 *	addenv		ajoute une associatoin a un environnement.
 *	readenv		rend la valeur, ou NULL.
 *
 * Les environnements sont representes par les p-listes et non des
 * a-listes, pour des raisons de typage (le CDR d'une liste est de
 * type list et pas de type void *).
 *
 * A utiliser donc avec precautions en raison des cas limite.
 *
 *	memberp		rend un booleen
 *
 **********************************************************************/

list cons(void *car, list cdr)
{
     list l = NEW(struct cons_cell);
     
     CAR(l) = car;
     CDR(l) =  cdr;
     return l;
}

list emptyq(void)
{
     list l = cons(NIL, NIL);
     
     return cons(l, l);
}

list addq(void *elt, list q)
{
     list l = cons(elt, NIL);
     
     CDR(CAR(q)) = l;
     CAR(q) = l;

     return q;
}

list qlist(list q)
{
     return CDR(CDR(q));
}

list addenv(void *key, void *val, list env)
{
     return cons(key, cons(val, env));
}

void *readenv(void *key, list env)
{
     list l = env;
     
     while (l != NIL)
       {
	    if (CAR(l) == key) return CAR(CDR(l));
	    l = CDR(CDR(l));
       }

     return NULL;
}

boolean memberp(void *key, list l)
{
     list m = l;
     
     while (m != NIL)
       {
	    if (CAR(m) == key) return TRUE;
	    m = CDR(m);
       }
     
     return FALSE;
}



/**********************************************************************
 *
 * Copie d'une chaine de caracteres dans le tas temporaire
 *
 **********************************************************************
 *
 * Fonctions :
 *
 *	sncopy		prend la longueur en parametre (sans compter le
 *			'\0',  contrairement a strncpy).
 *	scopy		ne prend pas la longueur en parametre.
 *
 **********************************************************************/

char *sncopy(char *s, int l)
{
     return strncpy(allocl(l+1), s, l+1);
}

char *scopy(char *s)
{
     return sncopy(s, strlen(s));
}


/**********************************************************************
 *
 * Copie d'une chaine de caracteres dans le tas permanent
 *
 **********************************************************************/

extern char *pscopy(char *s)
{
     int l = strlen(s);
     
     return strncpy(allocp(l+1), s, l+1);
}


/***********************************************************************
 *
 *                  Gestion des tables de hashage
 *
 **********************************************************************
 *
 * Fonctions :
 *
 * hash			Calcul de la valeur de hashage
 * create_hash_table	Creation de la table de hash
 * find			Recherche sans creation
 * find_create		Recherche avec creation
 * 
 **********************************************************************/


static unsigned hash(char *str, struct hash_table *ht)
{
     unsigned hsize = ht->size;
     char *s;
     unsigned val;
     
     val = 0;
     for (s = str; *s != '\0'; s++)
       {
	    val = ((val << 3) + (*s & 07)) % hsize;
       }
     return val;
}

struct hash_table *create_hash_table(int size, int class)
{
     struct hash_table *ht;
     struct hash_cell *(*cells)[];
     int i;
     
     if (class == H_LOCAL)
       {
	    ht = (struct hash_table *)allocl(sizeof(struct hash_table));
	    cells = (struct hash_cell *(*)[])
		 allocl(size*sizeof(struct hash_cell *));
       }
     else
       {
	    ht = (struct hash_table *)allocp(sizeof(struct hash_table));
	    cells = (struct hash_cell *(*)[])
		 allocp(size*sizeof(struct hash_cell *));
       }
     
     for (i = 0; i<size; i++) (*cells)[i] = NULL;
     
     ht->size = size;
     ht->class = class;
     ht->cells = cells;

     return ht;
}

extern struct hash_cell *find(struct hash_table *ht, char *str)
{
     unsigned h = hash(str, ht);
     struct hash_cell *s, *p;

     p = NULL;
     s = (*ht->cells)[h];

     while ((s != NULL) && (strcmp(s->str, str) != 0))
	  {
	       p = s;
	       s = s->next;
	  };

     return s;
}

struct hash_cell *find_create(struct hash_table *ht, char *str)
{
     unsigned h = hash(str, ht);
     struct hash_cell *s, *p;

     p = NULL;
     s = (*ht->cells)[h];

     while ((s != NULL) && (strcmp(s->str, str) != 0))
	  {
	       p = s;
	       s = s->next;
	  };

     if (s == NULL)
       {
	    if (ht->class == H_LOCAL)
	      {
		   s = NEW(struct hash_cell);
		   s->str = scopy(str);
	      }
	    else
	      {
		   s = PNEW(struct hash_cell);
		   s->str = pscopy(str);
	      }
	    
	    s->contents = NULL;
	    s->next = NULL;
	    
	    if (p == NULL) (*ht->cells)[h] = s;
	    else p->next = s;
       }
     
     return s;
}



/**********************************************************************
 *
 *             Initialisation pour les besoins du module
 *
 **********************************************************************
 *
 * Fonctions :
 *
 * 	InitUtil	a appeler avant toute compilation.
 * 	InitPre		a appeler avant la compilation de chaque forme
 * 			de toplevel.
 * 	InitPost	a appeler apres la compilation de chaque forme
 * 			de toplevel.
 *
 **********************************************************************/
     
void InitUtil(void)
{
     init_allocl();
}

void UtilPre(void)
{
}

void UtilPost(void)
{
     free_all();
     init_allocl();
}
