/***************************************************************************
  Copyright (C) Nitsan Seniak 1989, 1990

  This file is part of the K2 compiler.
  Permission to copy this software, in whole or in part, to use this
  software for any lawful noncommercial purpose, and to redistribute
  this software is granted subject to the restriction that all copies
  made of this software must include this copyright notice in full.
  The author(s) makes no warranties or representations of any kind, either
  express or implied, including but not limited to implied warranties
  of merchantability or fitness for any particular purpose.
  All materials developed as a consequence of the use of this
  software shall duly acknowledge such use, in accordance with the usual
  standards of acknowledging credit in research.
 ***************************************************************************/ 

/**********************************************************************
 *                                                                    *
 *		PROPAGATION DES PROPRIETES COLLECTEES                 *
 *                                                                    *
 **********************************************************************
 *
 * Ce module contient le coeur du compilateur. On y determine
 * comment doit etre compilee chaque fonction (comme une fonction
 * globale C, comme un label), et chaque variable (doit-elle etre
 * referencee par une indirection ?).
 *
 **********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <setjmp.h>
#include "util.h"
#include "objects.h"
#include "yystype.h"
#include "collect.h"
#include "k2.h"
#include "tokens.h"
#include "print.h"
#include "propagate.h"


static void ComputeBrainDamagedCalled(function *);
static void ComputeStupidCalled(function *);
static void ComputeCalled(function *);
static void GiveContinuation(function *, scont);
static void CollectContinuations(function *);
static void CalledContinuations(function *);
static void Continuations(function *);
static void CollectContinuations(function *);
static void SetFunctionsStatus(void);
static void CollectGlobalFunctions(void);
static void PrepareFunctions(void);
static void VisitCalled(function *);
static void Visit(function *);
/*static void CalledFreeVariables(function *);*/
static void FreeVariables(function *);
static void CollectFreeVariables(function *);
static void CollectCalledLabels(function *);
static void CollectLabels(function *);


/**********************************************************************
 *
 *                   Fonction de propagation
 *
 **********************************************************************
 *
 * On calcule d'abord le statut des fonctions, ie : si elles sont
 * locales, globales, expansees en ligne... puis on calcule les variables
 * libres des fonctions globales, transitivement. Ensuite, on marque
 * les variables "reference", ie qui doivent etre accedees par une
 * indirection a l'interieur des fonctions dans lesquelles elles sont
 * libres. Finalement, on attribue a chaque fonction globale ses labels
 * et a chaque label sa fonction globale.
 *
 * Fonctions :
 *
 *	PropagateInformations	La fonction principale.
 *
 * Variables :
 *
 * 	CurrentFunction		La fonction globale en cours de traitement.
 * 	GlobalFunctions		La liste des fonctions globales.
 * 	Iteratep		Un flag servant a terminer l'iteration.
 *
 ***********************************************************************/

static function *CurrentFunction;
list GlobalFunctions;
static boolean Iteratep;


void PropagateInformations(void)
{

     list l;
     

     DEBUG
       {
	    fprintf(Out, "Propagating informations\n");
       }
     END

				/* Calcul de called_t et called_n */
				/* ------------------------------ */

     if (OptimLevel <= 1)
	  for (l = AllFunctions; l != NIL; l = CDR(l))
	       ComputeBrainDamagedCalled(CAR(l));
     else if (OptimLevel == 2)
	  for (l = AllFunctions; l != NIL; l = CDR(l))
	       ComputeStupidCalled(CAR(l));
     else
       {
	    for (l = AllFunctions; l != NIL; l = CDR(l))
	      {
		   function *fun = CAR(l);
		   list m;
	    
		   if (fun->status == ROOT)
			GiveContinuation(fun, UNKSCONT);

		   for (m = fun->called; m != NIL; m = CDR(m))
		     {
			  struct edge *e = CAR(m);
			  function *cfun = e->fun;
			  scont c = e->scont;
		   
			  if (!(TERMSCONTP(c) || DUMSCONTP(c)))
			       GiveContinuation(cfun, c);
		     }
	      }

	    PrepareFunctions();
     
	    for (l = AllFunctions; l != NIL; l = CDR(l))
	      {
		   function *fun = CAR(l);
	    
		   CurrentFunction = fun;
		   CalledContinuations(fun);
		   fun->visited = TRUE;
	      }

	    for (l = AllFunctions; l != NIL; l = CDR(l))
		 ComputeCalled(CAR(l));
       }



				/* Calcul des statuts des fonctions */
				/* -------------------------------- */

     SetFunctionsStatus();

     do
       {
	    CollectGlobalFunctions();
	    PrepareFunctions();

	    Iteratep = FALSE;
	    for (l = GlobalFunctions; l != NIL; l = CDR(l))
		 {
		      function *fun = CAR(l);
		      CurrentFunction = fun;
		      VisitCalled(fun);
		 }
       }
     while (Iteratep);



				/* Calcul des variables libres */
				/* --------------------------- */

     PrepareFunctions();

     for (l = GlobalFunctions; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);

	    CurrentFunction = fun;
	    FreeVariables(fun);
       }



				/* Calcul des variables "reference" */
				/* -------------------------------- */

     for (l = GlobalFunctions; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);
	    list m;
	    
	    for (m = fun->freevars; m != NIL; m = CDR(m))
	      {
		   variable *var = CAR(m);
		   if (var->mutated) var->reference = TRUE;
	      }
       }


				/* Attributions fonction <-> label */
				/* ------------------------------- */

     for (l = GlobalFunctions; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);
	    
	    CurrentFunction = fun;
	    CollectCalledLabels(fun);
       }


				/* Calcul des informations de compilation */
				/* -------------------------------------- */
     
     
     for (l = AllFunctions; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);
	    
	    if ((fun->continued) || (fun->status == ROOT)) Infos.roots++;
	    else if (fun->contvar == NULL) /* Pas associee a un block */
	      {
		   if (fun->status == GLOBAL) Infos.globals++;
		   else if (fun->status == LABEL) Infos.labels++;
	      }
	    else
	      {
		   if (fun->status == GLOBAL) Infos.bglobals++;
		   else if (fun->status == LABEL) Infos.blabels++;
	      }
       }


				/* Petit hack pour les blocs */
				/* ------------------------- */

     for (l = AllBlocks; l != NIL; l = CDR(l))
       {
	    block *blk = CAR(l);
	    function *fun = blk->fun;

	    if (fun->status != GLOBAL) fun->contvar = NULL;
       }

     CollectGlobalFunctions();

     DEBUG
       {
	    PrintFunctions(AllFunctions);
	    PrintVariables(AllVariables);
       }
     END
}


/**********************************************************************
 *
 * Calcul de called_t et called_n
 *
 **********************************************************************
 *
 * Il y a trois facons de calculer called_t et called_n :
 *
 *	- La brain-damagee
 * 	- La stupide
 * 	- La maline
 *
 * Fonctions :
 *
 * 	ComputeBrainDamagedCalled
 * 			Calcule called_t et called_n pour une fonction
 * 			donnee un considerant que TOUT appel est non
 * 			terminal.
 * 	ComputeStupidCalled
 * 			Calcule called_t et called_n pour une fonction
 * 			donnee sans tenir compte du cablage de
 * 			continuations.
 *	GiveContinuation
 * 			Ajoute une continuation possible a une
 * 			fonction, en la propageant le long des
 * 			arcs d'appels terminaux.
 *	CalledContinuations
 *			Comptabilise les continuations des fonctions
 *			des appelees de maniere terminale par une
 * 			fonction donnee.
 *	Continuations	Comptabilise toutes continuations
 * 			(transitivement) d'une fonction donnee.
 *	CollectContinuations
 *			Comptabilise les continuations pour
 * 			une fonction appelee donnee.
 * 	
 **********************************************************************/


/*
 * Calcul stupide de called_n et called_t
 */

static void ComputeBrainDamagedCalled(function *fun)
{
     list l;
     
     for (l = fun->called; l != NIL; l = CDR(l))
       {
	    struct edge *e = CAR(l);
	    function *cfun = e->fun;
     
	    if (!memberp(cfun, fun->called_n))
		 fun->called_n = cons(cfun, fun->called_n);
       }
}

static void ComputeStupidCalled(function *fun)
{
     list l;
     
     for (l = fun->called; l != NIL; l = CDR(l))
       {
	    struct edge *e = CAR(l);
	    scont c = e->scont;
	    function *cfun = e->fun;
	    
	    if (TERMSCONTP(c) || DUMSCONTP(c))
	      {
		   if (!memberp(cfun, fun->called_t))
			fun->called_t = cons(cfun, fun->called_t);
	      }
	    else if (!memberp(cfun, fun->called_n))
		 fun->called_n = cons(cfun, fun->called_n);
       }
}


/*
 * Propagation d'une continuation le long des arcs d'appels terminaux.
 */

static void GiveContinuation(function *fun, scont c)
{
     if (!memberp(c, fun->sconts))
       {
	    list l;
	    
	    fun->sconts = cons(c, fun->sconts);
	    for (l = fun->called; l != NIL; l = CDR(l))
	      {
		   struct edge *e = CAR(l);
		   function *cfun = e->fun;
		   scont cc = e->scont;
		   
		   if (TERMSCONTP(cc)) GiveContinuation(cfun, c);
	      }
       }
}

/*
 * Collecte des continuations d'une fonction et des fonctions appelees
 * par elle de facon terminale.
 */

static void CalledContinuations(function *fun)
{
     list l;

     fun->visitor = CurrentFunction;
     
/*     for (l = fun->called_t; l != NIL; l = CDR(l))
	  Continuations(CAR(l)); */
     
     for (l = fun->called; l != NIL; l = CDR(l))
       {
	   struct edge *e = CAR(l);
	   function *cfun = e->fun;
	   scont cc = e->scont;
		   
	   if (TERMSCONTP(cc)) Continuations(cfun);
	}
}

static void Continuations(function *fun)
{
     if (fun->visitor == CurrentFunction) ; /* On boucle. */
     else if (fun->visited) CollectContinuations(fun); /* Deja visitee */
				/* lors du calcul des continuations */
				/* d'une autre fonction, donc */
				/* on connait deja le resultat. */
     else
       {			/* Jamais visitee. */
	    CollectContinuations(fun);
	    CalledContinuations(fun);
       }
}

static void CollectContinuations(function *fun)
{
     list l;
     list new_c = CurrentFunction->sconts;
     
     for (l = fun->sconts; l != NIL; l = CDR(l))
       {
	    scont c = CAR(l);
	    
	    if (!memberp(c, new_c))
		 new_c = cons(c, new_c);
       }
     
     CurrentFunction->sconts = new_c;
}


/*
 * Calcul de called_n et called_t qui tient compte de la possibilite
 * de cabler des continuations.
 */

static void ComputeCalled(function *fun)
{
     list l;

     for (l = fun->called; l != NIL; l = CDR(l))
       {
	    struct edge *e = CAR(l);
	    scont c = e->scont;
	    function *cfun = e->fun;
	    
	    if ((TERMSCONTP(c) || DUMSCONTP(c))
		|| (CDR(cfun->sconts) == NULL))
	      {
		   if (!memberp(cfun, fun->called_t))
			fun->called_t = cons(cfun, fun->called_t);
	      }
	    else if (!memberp(cfun, fun->called_n))
		 fun->called_n = cons(cfun, fun->called_n);
       }
}


/***********************************************************************
 *
 * Recherche du statut des fonctions
 *
 **********************************************************************
 *
 * On cherche les fonctions atteignables par deux chemins d'appels
 * terminaux,  partant de deux fonctions globales et sans intersection
 * (cf. rapport).
 *
 * Fonctions :
 *
 * 	SetFunctionsStatus	Calcule le statut initial des fonctions,
 * 				a partir des donnees collectees par
 * 				le parcours du code.
 * 	CollectGlobalFunctions	Parcourt les fonctions en les initialisant
 * 				pour la recherche, et collecte les fonctions
 * 				globales.
 *	PrepareFunctions	Pour initialiser les champs visited et
 * 				visitor, utilises pour les parcours
 * 				de graphe.
 *	VisitCalled		Visite les fonctions appelees de maniere
 * 				terminale par une fonction donnee.
 * 	Visit			Visite une fonctions donnee : verifie si cette
 * 				fonction a deja ete visitee, si elle doit
 * 				changer de statut,  etc...
 *
 ***********************************************************************/


static void SetFunctionsStatus(void)
{
     list l;
     
     for (l = AllFunctions; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);
	    list m;

	    for (m = fun->called_n; m != NIL; m = CDR(m))
	      {
		   function *cfun = CAR(m);
		   if (cfun->status == UNKNOWN)
			cfun->status = GLOBAL;
	      }
       }

     for (l = AllFunctions; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);
	    if (fun->status == UNKNOWN)
		 if (fun->callsno == 0) fun->status = DUMMY;
		 else fun->status = LABEL;
       }
}

static void CollectGlobalFunctions(void)
{
     list l;
     
     GlobalFunctions = NIL;
     
     for (l = AllFunctions; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);

	    if ((fun->status == GLOBAL) || (fun->status == ROOT))
		 GlobalFunctions = cons(fun, GlobalFunctions);
       }
}

static void PrepareFunctions(void)
{
     list l;
     
     for (l = AllFunctions; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);

	    fun->visitor = NULL;
	    fun->visited = FALSE;
       }
}

static void VisitCalled(function *fun)
{
     list l;

     fun->visitor = CurrentFunction;
     
     for (l = fun->called_t; l != NIL; l = CDR(l))
	  Visit(CAR(l));
}

static void Visit(function *fun)
{
     if ((fun->status == GLOBAL) || (fun->status == ROOT)) ; /* Arret. */
     else if (fun->visitor == CurrentFunction) ; /* On boucle. */
     else if (fun->visitor == NULL) VisitCalled(fun); /* Jamais visitee. */
     else			/* Deja visitee par une AUTRE fonction */
       {			/* globale. */
	    fun->status = GLOBAL;
	    Iteratep = TRUE;
       }
}



/***********************************************************************
 * 
 * Calcul des variables libres des fonctions globales
 *
 **********************************************************************
 *
 * Fonctions :
 *
 *	CalledFreeVariables	Comptabilise les variables libres
 *				des fonctions appelees par une fonction
 * 				donnee.
 *	FreeVariables		Comptabilise toutes les variables libres
 * 				(transitivement) d'une fonction donnee.
 *	CollectFreeVariables	Comptabilise les variables libres pour
 * 				une fonction appelee donnee.
 *
 ***********************************************************************/


static void FreeVariables(function *fun)
{
     if (fun->visited) ; /* Deja visitee */
     else
       {			/* Jamais visitee. */
	    fun->visited = TRUE;
	    CollectFreeVariables(fun);
       }
}

static void CollectFreeVariables(function *fun)
{
     list l;
     int level = fun->level;
     list new_fv = fun->freevars;     


     for (l = fun->called_t; l != NIL; l = CDR(l))
       {   list v;
	   function *cfun = CAR(l);
	    
 	   FreeVariables(cfun);
	   for (v = cfun->freevars; v != NIL; v = CDR(v))
	     {
	       variable *var = CAR(v);
	    
	       if ((var->level < level) && !memberp(var, new_fv))
		 new_fv = cons(var, new_fv);
	     }
       }

     for (l = fun->called_n; l != NIL; l = CDR(l))
       {   list v;
	   function *cfun = CAR(l);
	    
 	   FreeVariables(cfun);
	   for (v = cfun->freevars; v != NIL; v = CDR(v))
	     {
	       variable *var = CAR(v);
	    
	       if ((var->level < level) && !memberp(var, new_fv))
		 new_fv = cons(var, new_fv);
	     }
       }
     
     fun->freevars = new_fv;
}



/***********************************************************************
 *
 * Calcul des labels des fonctions globales
 *
 **********************************************************************
 *
 * Fonctions :
 *
 * 	CollectLabels		Comptabilise les labels d'une fonction.
 *	CollectCalledLabels	Comptabilise les labels des appelees
 * 				d'une fonction donnee.
 *
 * Rappel : une fonction dont on a decide qu'elle serait compilee
 * comme un label est forcement appelee par une seule fonction
 * globale : la fonction dans laquelle elle est un label.
 *
 ***********************************************************************/

static void CollectCalledLabels(function *fun)
{
     list l;
     
     for (l = fun->called_t; l != NIL; l = CDR(l))
	  CollectLabels(CAR(l));
}

static void CollectLabels(function *fun)
{
     if ((fun->status == GLOBAL) || (fun->status == ROOT)) ;
     else if (fun->labelof != NULL) ; /* On est deja passe. */
     else
       {
	    CurrentFunction->labels = cons(fun, CurrentFunction->labels);
	    fun->labelof = CurrentFunction;
	    CollectCalledLabels(fun);
       }
}
