/***************************************************************************
  Copyright (C) Nitsan Seniak 1989, 1990

  This file is part of the K2 compiler.
  Permission to copy this software, in whole or in part, to use this
  software for any lawful noncommercial purpose, and to redistribute
  this software is granted subject to the restriction that all copies
  made of this software must include this copyright notice in full.
  The author(s) makes no warranties or representations of any kind, either
  express or implied, including but not limited to implied warranties
  of merchantability or fitness for any particular purpose.
  All materials developed as a consequence of the use of this
  software shall duly acknowledge such use, in accordance with the usual
  standards of acknowledging credit in research.
 ***************************************************************************/ 

/**********************************************************************
 *                                                                    *
 *                       IMPRESSION POUR LE DEBUG		      *
 *                                                                    *
 **********************************************************************/

/*
 * $Log:	print.c,v $
 * Revision 1.3  89/10/30  15:37:56  seniak
 * *** empty log message ***
 * 
 * Revision 1.2  89/09/05  17:41:09  seniak
 * *** empty log message ***
 * 
 * Revision 1.1  89/08/29  15:51:46  seniak
 * Initial revision
 * 
 */

#include <stdlib.h>
#include <stdio.h>
#include <setjmp.h>
#include "util.h"
#include "objects.h"
#include "tokens.h"
#include "yystype.h"
#include "atoms.h"
#include "k2.h"
#include "print.h"


static void PrintArgs(list);
static void PrintDefsList(list);
void PrintBlockRef(struct block_ref *);
static void PrintBindingsList(list);
static void PrintBinding(struct binding *);
static void PrintClauses(list);
static void PrintClause(struct clause *);
static void PrintTextsList(list);
static void PrintFunctionsNames(list);
static void PrintFunctionName(function *);
static void PrintVariableName(variable *);
static void PrintVariablesNames(list);
static void PrintEdges(list);
char *ScontString(scont);
static void PrintSconts(list);
static char *BooleanString(boolean);
static char *StatusString(enum fun_status);


void PrintLex(int token, void *lval)
{
     fprintf(Out, "%d\t", token);
     switch(token)
       {
       case TEXT:
	    fprintf(Out, "TEXT = %s", lval);
	    break;
       case VAR:
	 {
	      struct lex_variable *lexvar = lval;
	      fprintf(Out, "VAR = %s(%s)", lexvar->name,
		      lexvar->cname, lexvar->name);
	      break;
	 }
       case WRONG:
	    fprintf(Out, "WRONG");
	    break;
       default:
	    fprintf(Out, "(KEYWORD)");
	    
       }

     fprintf(Out, "\n");
}

void PrintForm(form *f)
{
     switch (f->tag)
       {
       case BLOCK:
	    fprintf(Out, "(BLOCK ");
	    PrintBlockRef(f->node.block_i.bref);
	    fprintf(Out, " ");
	    PrintForm(f->node.block_i.body);
	    fprintf(Out, ")");
	    break;
	    
       case CASE:
	    fprintf(Out, "(CASE ");
	    PrintForm(f->node.case_i.val);
	    fprintf(Out, " ");
	    PrintClauses(f->node.case_i.clauses);
	    fprintf(Out, " DEFAULT ");
	    PrintForm(f->node.case_i.def);
	    fprintf(Out, ")");
	    break;

       case CONTINUE:
	    fprintf(Out, "(CONTINUE ");
	    PrintForm(f->node.continue_i.cont);
	    fprintf(Out, " ");
	    PrintForm(f->node.continue_i.val);
	    fprintf(Out, ")");
	    break;

       case EXPRESSION:
	    fprintf(Out, "(EXPRESSION %s", f->node.expression_i.text);
	    PrintArgs(f->node.expression_i.args);
	    break;

       case FLET:
	    fprintf(Out, "(FLET ");
	    PrintDefsList(f->node.flet_i.defs);
	    fprintf(Out, " ");
	    PrintForm(f->node.flet_i.body);
	    fprintf(Out, ")");
	    break;

       case FUNCALL:
	    fprintf(Out, "(FUNCALL ");
	    PrintForm(f->node.funcall_i.fun);
	    PrintArgs(f->node.funcall_i.args);
	    break;

       case FUNCTION:
	    fprintf(Out, "(FUNCTION ");
	    PrintFunctionRef(f->node.function_i);
	    fprintf(Out, ")");
	    break;

       case IF:
	    fprintf(Out, "(IF ");
	    PrintForm(f->node.if_i.test);
	    fprintf(Out, " ");
	    PrintForm(f->node.if_i.iftrue);
	    fprintf(Out, " ");
	    PrintForm(f->node.if_i.iffalse);
	    fprintf(Out, ")");
	    break;

       case LABELS:
	    fprintf(Out, "(LABELS ");
	    PrintDefsList(f->node.flet_i.defs);
	    fprintf(Out, " ");
	    PrintForm(f->node.flet_i.body);
	    fprintf(Out, ")");
	    break;

       case LET:
	    fprintf(Out, "(LET ");
	    PrintBindingsList(f->node.let_i.binds);
	    fprintf(Out, " ");
	    PrintForm(f->node.let_i.body);
	    break;

       case PROGN:
	    fprintf(Out, "(PROGN");
	    PrintArgs(f->node.progn_i.body);
	    break;

       case RETURN_FROM:
	    fprintf(Out, "(RETURN-FROM ");
	    PrintBlockRef(f->node.return_from_i.bref);
	    fprintf(Out, " ");
	    PrintForm(f->node.return_from_i.val);
	    fprintf(Out, ")");
	    break;
	    
       case SETQ:
	    fprintf(Out, "(SETQ ");
	    PrintVariableRef(f->node.setq_i.vref);
	    fprintf(Out, " ");
	    PrintForm(f->node.setq_i.val);
	    fprintf(Out, ")");
	    break;
	    
       case STATEMENT:
	    fprintf(Out, "(STATEMENT %s", f->node.statement_i.text);
	    PrintArgs(f->node.statement_i.args);
	    break;

       case TEST:
	    fprintf(Out, "(TEST %s", f->node.test_i.text);
	    PrintArgs(f->node.test_i.args);
	    break;

       case THE_CONTINUATION:
	    fprintf(Out, "(THE-CONTINUATION)");
	    break;

       case VARIABLE:
	    PrintVariableRef(f->node.variable_i);
	    break;

       case APPLICATION:
	    fprintf(Out, "(");
	    PrintFunctionRef(f->node.application_i.fref);
	    PrintArgs(f->node.application_i.args);
	    break;
	    
       default:
	    fprintf(Err, "%s panic: unrecognized tag in PrintForm: %d", 
		    Me, f->tag);
	    exit(3);
       }
}

void PrintVariableRef(struct variable_ref *var)
{
     fprintf(Out, "%s", var->lexvar->name);
}

void PrintBlockRef(struct block_ref *bref)
{
     fprintf(Out, "%s", bref->lexvar->name);
}

void PrintFunctionRef(struct function_ref *fun)
{
     fprintf(Out, "%s", fun->lexvar->name);
}

static void PrintArgs(list args)
{
     list l = args;
     
     while (l != NIL)
       {
	    fprintf(Out, " ");
	    PrintForm(CAR(l));
	    l = CDR(l);
       }
     fprintf(Out, ")");
}

static void PrintDefsList(list defs)
{
     list l = defs;
     boolean sp = FALSE;

     fprintf(Out, "(");
     
     while (l != NIL)
       {
	    if (sp) fprintf(Out, " "); else sp = TRUE;
	    PrintDefinition(CAR(l));
	    l = CDR(l);
       }

     fprintf(Out, ")");
}

void PrintVariableRefsList(list vrefs)
{
     list l = vrefs;
     boolean sp = FALSE;
     
     fprintf(Out, "(");

     while (l != NIL)
       {
	    if (sp) fprintf(Out, " "); else sp = TRUE;
	    PrintVariableRef(CAR(l));
	    l = CDR(l);
       }
     
     fprintf(Out, ")");
}

void PrintDefinition(struct definition *d)
{
     fprintf(Out, "(");
     PrintFunctionRef(d->fref);
     fprintf(Out, " ");
     PrintVariableRefsList(d->vrefs);
     fprintf(Out, " ");
     PrintForm(d->body);
     fprintf(Out, ")");
}

static void PrintBindingsList(list binds)
{
     list l = binds;
     boolean sp = FALSE;
     
     fprintf(Out, "(");
     
     while (l != NIL)
       {
	    if (sp) fprintf(Out, " "); else sp = TRUE;
	    PrintBinding(CAR(l));
	    l = CDR(l);
       }
     
     fprintf(Out, ")");
}

static void PrintBinding(struct binding *b)
{
     fprintf(Out, "(");
     PrintVariableRef(b->vref);
     fprintf(Out, " ");
     PrintForm(b->val);
     fprintf(Out, ")");
}

static void PrintClauses(list clauses)
{
     list l = clauses;
     
     while (l != NIL)
       {
	    PrintClause(CAR(l));
	    l = CDR(l);
       }
}

static void PrintClause(struct clause *c)
{
     fprintf(Out, "(");
     PrintTextsList(c->texts);
     fprintf(Out, " ");
     PrintForm(c->body);
     fprintf(Out, ")");
}

static void PrintTextsList(list texts)
{
     list l = texts;
     boolean sp = FALSE;
     
     fprintf(Out, "(");
     
     while (l != NIL)
       {
	    if (sp) fprintf(Out, " "); else sp = TRUE;
	    printf("%s ", CAR(l));
	    l = CDR(l);
       }
     
}

void PrintFunction(function *fun)
{
     fprintf(Out, "Function %s_%d, level %d\n",
	     fun->lexvar->name, fun->suffix, fun->level);
     fprintf(Out, "\tDefinition : %s_%d\n", 
	     fun->def->fref->fun->lexvar->name, fun->def->fref->fun->level);

     fprintf(Out, "\tFree vars : ");
     PrintVariablesNames(fun->freevars);

     fprintf(Out, "\tcalled : ");
     PrintEdges(fun->called);

     fprintf(Out, "\tsconts : ");
     PrintSconts(fun->sconts);

     fprintf(Out, "\tcalled_t : ");
     PrintFunctionsNames(fun->called_t);

     fprintf(Out, "\tcalled_n : ");
     PrintFunctionsNames(fun->called_n);

     fprintf(Out, "\tstatus : %s\n", StatusString(fun->status));

     fprintf(Out, "\tcontinued : %s\n", BooleanString(fun->continued));

     fprintf(Out, "\tcallsno : %d\n", fun->callsno);

     fprintf(Out, "\tlabels : ");
     PrintFunctionsNames(fun->labels);

     fprintf(Out, "\tlabelof : ");
     PrintFunctionName(fun->labelof);

     fprintf(Out, "\tcontvar : ");
     PrintVariableName(fun->contvar);
}

void PrintFunctions(list funs)
{
     list l;
     for (l = funs; l != NIL; l = CDR(l)) PrintFunction(CAR(l));
}

static void PrintFunctionsNames(list funs)
{
     list l;
     for (l = funs; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);
	    fprintf(Out, "%s_%d ", fun->lexvar->name, fun->suffix);
       }
     fprintf(Out, "\n");
}

static void PrintFunctionName(function *fun)
{
     if (fun == NULL)
	  fprintf(Out, "none\n");
     else fprintf(Out, "%s_%d\n", fun->lexvar->name, fun->suffix);
}

static void PrintVariablesNames(list vars)
{
     list l;
     for (l = vars; l != NIL; l = CDR(l))
       {
	    variable *var = CAR(l);
	    fprintf(Out, "%s_%d ", var->lexvar->name, var->suffix);
       }
     
     fprintf(Out, "\n");
}

static void PrintEdges(list edges)
{
     list l;
     for (l = edges; l != NIL; l = CDR(l))
       {
	    struct edge *e = CAR(l);
	    fprintf(Out, "<%s_%d,%s> ", e->fun->lexvar->name, e->fun->suffix,
		    ScontString(e->scont));
       }
     
     fprintf(Out, "\n");
}

static void PrintSconts(list sconts)
{
     list l;
     for (l = sconts; l != NIL; l = CDR(l))
       {
	    scont c = CAR(l);
	    fprintf(Out, "%s ", ScontString(c));
       }
     fprintf(Out, "\n");
}

char *ScontString(scont c)
{
     if (UNKSCONTP(c)) return "unknown";
     else if (TERMSCONTP(c)) return "term";
     else if (DUMSCONTP(c)) return "dummy";
     else 
       {
	    static char r[20];
	    sprintf(r, "cont%d", (int)c);
	    return r;
       }
}

static void PrintVariableName(variable *var)
{
     if (var == NULL)
	  fprintf(Out, "none\n");
     else fprintf(Out, "%s_%d\n", var->lexvar->name, var->suffix);
}
     
void PrintVariable(variable *var)
{
     fprintf(Out, "Variable %s_%d, level %d =\n",
	     var->lexvar->name, var->suffix, var->level);
     fprintf(Out, "\tMutated : %s\n", BooleanString(var->mutated));
     fprintf(Out, "\tReference : %s\n", BooleanString(var->reference));
}

void PrintVariables(list vars)
{
     list l;
     for (l = vars; l != NIL; l = CDR(l))
	  PrintVariable(CAR(l));
}

static char *BooleanString(boolean b)
{
     if (b) return "TRUE";
     else return "FALSE";
}

static char *StatusString(enum fun_status s)
{
     switch (s)
       {
       case ROOT:
	    return "ROOT";
       case UNKNOWN:
	    return "UNKNOWN";
       case GLOBAL:
	    return "GLOBAL";
       case LABEL:
	    return "LABEL";
       case DUMMY:
	    return "DUMMY";
       case BLOCK:
	    return "BLOCK";
       default:
	    return "Unknown status";
       }
}
