/***************************************************************************
  Copyright (C) Nitsan Seniak 1989, 1990

  This file is part of the K2 compiler.
  Permission to copy this software, in whole or in part, to use this
  software for any lawful noncommercial purpose, and to redistribute
  this software is granted subject to the restriction that all copies
  made of this software must include this copyright notice in full.
  The author(s) makes no warranties or representations of any kind, either
  express or implied, including but not limited to implied warranties
  of merchantability or fitness for any particular purpose.
  All materials developed as a consequence of the use of this
  software shall duly acknowledge such use, in accordance with the usual
  standards of acknowledging credit in research.
 ***************************************************************************/ 

/**********************************************************************
 *                                                                    *
 * 			     LE COMPILATEUR                           *
 *                                                                    *
 **********************************************************************
 *
 * Ce module contient le "main" du compilateur ainsi que les fonctions
 * de compilation appelees par l'analyseur syntaxique.
 *
 * Les arguments et options de la commande de compilation (nommee
 * typiquement "k2") sont analyses et les valeurs de variables globales
 * exportees sont affectees.
 *
 * La syntaxe de la commande "k2" est la suivante :
 *
 *	     k2 [ input ] [ -o output ] [ -s ] [ -d ] [ -r ] [ -l ]
 * 		[-c nnn ] [ -O nnn ] [ -v ] [ -t ] [ -k str ]*
 *
 * Options :
 *
 *   input est le fichier contenant le source K2 ; l'entree stan-
 *   dard est prise par defaut.
 *
 *   -o output specifie un fichier recevant le code C  produit  ;
 *   la sortie standard est prise par defaut.
 *
 *   -s indique que le code produit doit contenir des  directives
 *   "#line".
 *
 *   -d specifie le mode "debug". En utilisation normale,  cette option
 *   est desactivee.
 *
 *   -r indique que l'assembleur doit pratiquer une  reprise  sur
 *   erreur  ;  par defaut, il s'arrete a la premiere erreur ren-
 *   contree.
 *
 *   -l indique que les messages d'erreur  doivent  contenir  une
 *   localisation  plus precise de l'erreur dans le source, quand
 *   c'est possible.
 *
 *   -c nnn specifie la valeur initiale du compteur utilise pour suffixer
 *   les noms des fonctions locales compilees comme des fonctions globales C
 *   (10000 par defaut).
 *
 *   -O nnn specifie le degre d'optimisation dans la compilation des fonctions
 *   locales et des blocs lexicaux (3 par defaut) :
 *
 *    		0	Toutes les fonctions sont compilees comme des
 * 			fonctions C, et tous les appels sont compiles
 * 			comme des appels a des fonctions C.
 *
 *		1	Idem que 0, mais les appels auto-recursifs sont
 * 			compiles comme des sauts.
 *
 *		2	Certaines fonctions locales sont compilees comme
 * 			des etiquettes, mais une fonction appelee de facon
 * 			non terminale est systematiquement compilee
 * 			comme une fonction C.
 *
 * 		3	Niveau maximal d'optimisation.
 *
 *    A noter que l'option -O influe egalement sur la compilation des
 *    echappements lexicaux.
 *
 *    -v indique le mode bavard ; le compilateur affiche des informations
 *    et des statistiques sur la compilation des fonctions et des blocs.
 *    En utilisation normale,  cette option est desactivee.
 *
 *    -t indique que le compilateur doit produire du C traditionnel, plutot
 *    que du C purement ANSI.
 *
 *    -k str indique que str est un mot cle, et qu'aucun identificateur
 *    genere par K2 ne doit lui etre egal.
 *
 *
 * La compilation se fait selon le schema suivant :
 *
 *
 * - Analyse syntaxique et internenement des lexemes et des formes
 *   syntaxiques (atoms.l, forms.y, yystype.h).
 *
 * - Parcours du code interne pour collecter des informations
 *   d'ordre syntaxiques (collect.c, objects.h).
 *
 * - Propagation de ces informations par une iteration de point fixe
 *   pour determiner des proprietes sur les fonctions et les
 *   variables (propagate.c).
 *
 * - Production de code C (generation.c).
 *
 *
 * La compilation se fait fonction par fonction, sans aucune information
 * sur l'environnement global de compilation.
 *
 ***********************************************************************/


#include <stdlib.h>
#include <stdio.h>
#include <setjmp.h>
#include <string.h>
#include "util.h"
#include "objects.h"
#include "tokens.h"
#include "yystype.h"
#include "atoms.h"
#include "forms.h"
#include "k2.h"
#include "print.h"
#include "ident.h"
#include "collect.h"
#include "propagate.h"
#include "generation.h"


/**********************************************************************
 *
 *            Les correspondant aux options de compilation
 *
 **********************************************************************
 *
 * Variables :
 *
 *	In		canal d'entree du source.
 *	Out		canal de sortie du genere.
 *	InName		nom du fichier d'entree (si In != stdin).
 *	OutName		nom du fichier de sortie (si Out != stdout).
 *
 *	Me		nom du compilateur lui-meme (ie,  argv[0]).
 *
 *	Debug		flag affecte par l'option -d.
 *	Recover		flag affecte par l'option -r.
 *	Localize	flag affecte par l'option -l.
 *	SharpLine	flag affecte par l'option -s.
 *	OptimLevel 	variable affectee par l'option -O.
 *	Verbose		flag affecte par l'option -v.
 *	Traditional	flag affecte par l'option -t.
 *
 **********************************************************************/

FILE *In = { stdin };
FILE *Out = { stdout };
FILE *Err = { stderr };

char *InName = "<stdin>";
char *OutName = "<stdout>";
char *Me;

boolean Debug = FALSE;
boolean Recover = FALSE;
boolean Localize =  FALSE;
boolean SharpLine =  FALSE;
int OptimLevel = 3;
boolean Verbose = FALSE;
boolean Traditional = FALSE;


/**********************************************************************
 *
 *           Autres variables exprortees par le compilateur
 *
 **********************************************************************
 *
 *
 *	Error		flag pour savoir si le code genere est valide ou non.
 *	ObjType		le nom du type des objets K2.
 *	DefunExit	pour les erreurs de compilation du defun.
 *	GlobalFunctionsCount
 *			Une compteur pour renommer les fonctions
 * 			globalisees.
 * 	Infos		variable collectant des infos sur la compilation.
 *	SymbolsTable	Table de hash des symboles
 *	DeclareTable	Table des fonctions et variables declarees par :declare
 * 
 **********************************************************************/

boolean Error = FALSE;
char *ObjType = "obj_t ";
jmp_buf DefunExit;
int GlobalFunctionsCount = 10000;
struct info_struct Infos = { 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0 };
struct hash_table *SymbolsTable;
struct hash_table *DeclareTable;


/**********************************************************************
 *
 *                  Point d'entree du compilateur
 *
 **********************************************************************
 *
 * Variables :
 *
 *	CompileExit	pour quitter le compilateur.
 *
 * Fonctions :
 *
 *	ParseOptions	analyse des options de compilation.
 *	Optionp		indique si une chaine de caracteres
 * 			peut etre une option ou pas.
 *
 **********************************************************************/
 
static void ParseOptions(int, char *[]);
static boolean Optionp(char *);

static jmp_buf CompileExit;

/*
 * Point d'entree.
 */

void main(int argc, char *argv[])
{

     InitUtil();
     InitLex();
     InitIdent();

     Me = argv[0];

     ParseOptions(argc, argv);
     
     yyin = In;

     DeclareTable = create_hash_table(H_GLOBAL, 307);

     if (setjmp(CompileExit) == FALSE)
       {
	    do yyparse();
	    while (!Eof && Recover);
       }

     if (In != stdin) fclose(In);
     if (Out != stdout) fclose(Out);

     if (Error)
       {
	    if (Recover)
		 fprintf(Err, "%s: compilation aborted\n", Me);
/*	    if (Out != stdout)
		 remove(OutName);
*/				/* C'est de l'ANSI mais pas du Sun... */
	    exit(4);
       }

     if (Verbose)
       {
	    fprintf(Err, "(");
	    fprintf(Err, "(roots %d)", Infos.roots);
	    fprintf(Err, "(globals %d)", Infos.globals);
	    fprintf(Err, "(labels %d)", Infos.labels);
	    fprintf(Err, "(bglobals %d)", Infos.bglobals);
	    fprintf(Err, "(blabels %d)", Infos.blabels);
	    fprintf(Err, "(calls %d)", Infos.calls);
	    fprintf(Err, "(lcalls %d)", Infos.lcalls);
	    fprintf(Err, "(gosubs %d)", Infos.gosubs);
	    fprintf(Err, "(args %d)", Infos.args);
	    fprintf(Err, "(exits %d)", Infos.exits);
	    fprintf(Err, "(longjmps %d)", Infos.longjmps);
	    fprintf(Err, ")\n");
       }
     
     exit(0);
}


/*
 * Analyse des options de compilation.
 */

static void ParseOptions(int argc, char *argv[])
{
     boolean file_opt = FALSE;
     boolean o_opt = FALSE;

     int i;

     for (i = 1; i < argc; i++)
       if (Optionp(argv[i]))
	{	
	   switch (argv[i][1])
	     {
	     case 'o':
		  if (o_opt)
		    {
			 fprintf(Err, "%s: more than one output file\n",Me);
			 exit(1);
		    }
		  
		  if ((i == argc-1) || (Optionp(argv[i+1])))
		    {
			 fprintf(Err, "%s: missing -o argument\n", Me);
			 exit(1);
		    }
		  
		  o_opt = TRUE;
		  OutName = argv[++i];
		  
		  if ((Out = fopen(OutName, "w")) == NULL)
		    {
			 fprintf(Err, "%s: cannot open %s\n", Me, OutName);
			 exit(1);
		    }
		  
		  break;


	     case 'd':
		  Debug = TRUE;
		  break;

	     case 'r':
		  Recover = TRUE;
		  break;
		  
	     case 'l':
		  Localize = TRUE;
		  break;

	     case 's':
		  SharpLine = TRUE;
		  break;

	     case 'c':
		  if ((i == argc-1) || (Optionp(argv[i+1])))
		    {
			 fprintf(Err, "%s: missing -c argument\n", Me);
			 exit(1);
		    }
		  
		  GlobalFunctionsCount = atoi(argv[++i]);
		  break;

	     case 'O':
		  if ((i == argc-1) || (Optionp(argv[i+1])))
		    {
			 fprintf(Err, "%s: missing -O argument\n", Me);
			 exit(1);
		    }
	       {
		    char *opt = argv[++i];

		    if ((strlen(opt) != 1) || (opt[0] < '0') || (opt[0] > '3'))
		      {
			   fprintf(Err, "%s: bad argument to -O\n", Me);
			   exit(1);
		      }

		    OptimLevel = opt[0]-'0';
		    break;
	       }


	     case 'v':
		  Verbose = TRUE;
		  break;


	     case 't':
		  Traditional = TRUE;
		  break;

	     case 'k':
		  if ((i == argc-1) || (Optionp(argv[i+1])))
		    {
			 fprintf(Err, "%s: missing -k argument\n", Me);
			 exit(1);
		    }

		  find_create(CKeywordsTable, argv[++i]);
		  break;
		  
	     default:
		  fprintf(Err, "%s: unknown option : %s\n", Me, argv[i]);
		  exit(1);
	     }
      }
       else	
	 {
	     if (file_opt)
	       {
		    fprintf(Err, "%s: more than one input file\n", Me);
		    exit(2);
	       }
	     
	     file_opt = TRUE;
	     InName = argv[i];
	     
	     if ((In = fopen(InName, "r")) ==  NULL)
	       {
		    fprintf(Err, "%s: cannot open %s\n", Me, InName);
		    exit(2);
	       }
	}
}

static boolean Optionp(char *s)
{
     if (s[0] == '-')
	  if (strlen(s) != 2)
	    {
		 fprintf(Err, "%s: bad option : %s\n", Me, s);
		 exit(1);
	    }
	  else return TRUE;
     return FALSE;
}

     

/**********************************************************************
 *
 *                     Fonctions de compilation
 *
 **********************************************************************
 *
 * Fonctions :
 *
 * 	CompileDefun   invoquee par yacc lors de la reduction d'un defun.
 *	CompileDefvar  invorquee par yacc lors de la reduction d'un defvar.
 *	CompileDeclaration
 *		       invoquee par yacc lors de la reduction d'un declaration.
 *
 **********************************************************************/

void CompileDefun(struct definition *def)
{
     DEBUG
       {
	    printf("Compiling DEFUN :\n");
	    PrintDefinition(def);
       }
     END
     
     if (setjmp(DefunExit) == FALSE)
       {
	    CollectInformations(def);
	    PropagateInformations();
	    Generate();
       }
     else 
       {
	    fprintf(Err, "%s: compilation of function %s aborted\n", Me,
		    def->fref->lexvar->name);
	    Error = TRUE;
	    if (!Recover) longjmp(CompileExit, TRUE);
       }
}

void
 CompileDefvar(struct variable_ref *var, int lineno, int vlineno, char *vname)
{
     struct hash_cell *c;
     
     DEBUG
       {
	    printf("Compiling DEFVAR :\n");
	    PrintVariableRef(var);
       }
     END

     if (SharpLine) PrintSharpLine(lineno, vlineno, vname);
     
     c = find(DeclareTable, var->lexvar->name);
     
     if (c != NULL)
       {
	    int class = (int)c->contents;
	    
	    switch (class)
	      {
	      case STATIC:
		   break;
		   
	      case EXTERN:
		   GENS(ObjType);
		   PrintVariableRefIdent(var);
		   GENS(";\n\n");
		   break;
	      }
       }

}

void CompileDeclaration(char *text, list vars,
			int lineno, int vlineno, char *vname)
{
     list l;
     char *c;
     
     DEBUG
       {
	    printf("Compiling DECLARATION :\n");
	    printf("\"%s\" ", text);
	    PrintVariableRefsList(vars);
       }
     END
     
     if (SharpLine) PrintSharpLine(lineno, vlineno, vname);

     l = vars;
     for (c = text; *c != '\0'; c++)
       {
	    if (*c != '\01') GENC(*c);
	    else if (l != NIL)
	      {
		   PrintVariableRefIdent(CAR(l));
		   l = CDR(l);
	      }
	    else
	      {
		   PrintError("inlined arguments exhausted", 
			      lineno, vlineno, vname);
		   Error = TRUE;
		   goto end;
	      }
       }

     if (l != NIL)
       {
	    PrintError("too many inlined arguments", 
		       lineno, vlineno, vname);
	    Error = TRUE;
       }

 end:
     GENS("\n\n");

     if (Error && !Recover) longjmp(CompileExit, TRUE);
}

/**********************************************************************
 *
 * Prototypes
 *
 **********************************************************************/

void FunPrototype(struct lex_variable *lexvar, int nargs, int class,
		  int lineno, int vlineno, char *vname)
{
     char c = '(';
     int i;
     struct hash_cell *h;
     
     if (SharpLine) PrintSharpLine(lineno, vlineno, vname);

     switch (class)
       {
       case STATIC:
	    GENS("static ");
	    break;
	    
       case EXTERN:
	    GENS("extern ");
	    break;
	    
       case INLINE:
	    GENS("inline ");
	    break;
       }
     
     GENS(ObjType);
     PrintLexVariable(lexvar);

     if (Traditional)
       {
	    GENS("();\n");
       }
     else
       {
	    for (i = 0; i<nargs; i++)
	      {
		   GENC(c);
		   c = ',';
		   GENS(ObjType);
	      }
     
	    if (c == '(') GENS("(void");
	    GENS(");\n");
       }
     
     h = find_create(DeclareTable, lexvar->name);
     h->contents = (void *)class;
}

void VarPrototype(struct lex_variable *lexvar, int class, 
		  int lineno, int vlineno, char *vname)
{
     struct hash_cell *c;
     
     if (SharpLine) PrintSharpLine(lineno, vlineno, vname);

     switch (class)
       {
       case STATIC:
	    GENS("static ");
	    break;
	    
       case EXTERN:
	    GENS("extern ");
	    break;
       }

     GENS(ObjType);
     PrintLexVariable(lexvar);
     GENS(";\n");

     c = find_create(DeclareTable, lexvar->name);
     c->contents = (void *)class;
}


/**********************************************************************
 *
 * Initialisation
 *
 **********************************************************************
 *
 * Fonctions :
 *
 * InitPre		invoquee par yacc reconnaissant le debut
 *			d'une forme de toplevel.
 * InitPost		invoquee par yacc apres reduction d'une
 *			forme de toplevel.
 *
 **********************************************************************/

void InitPre(void)
{
     UtilPre();
     SymbolsTable = create_hash_table(H_LOCAL, 101);
				/* Rajouter ici d'eventuels */
				/* appels */
}

void InitPost(void)
{
     UtilPost();
     SymbolsTable = NULL;
}
