/***************************************************************************
  Copyright (C) Nitsan Seniak 1989, 1990

  This file is part of the K2 compiler.
  Permission to copy this software, in whole or in part, to use this
  software for any lawful noncommercial purpose, and to redistribute
  this software is granted subject to the restriction that all copies
  made of this software must include this copyright notice in full.
  The author(s) makes no warranties or representations of any kind, either
  express or implied, including but not limited to implied warranties
  of merchantability or fitness for any particular purpose.
  All materials developed as a consequence of the use of this
  software shall duly acknowledge such use, in accordance with the usual
  standards of acknowledging credit in research.
 ***************************************************************************/ 

/***********************************************************************
 *                                                                     *
 * 			      GENERATION DE CODE C		       *
 *                                                                     *
 ***********************************************************************
 *
 * Ce module contient les fonctions parcourant les formes lues et
 * "affiche" le code C genere. C'est du code laborieux et pas tres joli.
 *
 ***********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <setjmp.h>
#include <string.h>
#include "util.h"
#include "objects.h"
#include "yystype.h"
#include "forms.h"
#include "propagate.h"
#include "k2.h"
#include "ident.h"
#include "atoms.h"
#include "tokens.h"
#include "generation.h"


/**********************************************************************
 *
 * Variables :
 *
 * 	Temp		un compteur pour la fabrication de variables
 * 			temporaires.
 *	CurrentFunction	la fonction globale en cours de generation.
 * 
 **********************************************************************/

static int Temp;
static function *CurrentFunction;



/**********************************************************************
 * 
 *                          Continuations
 *
 **********************************************************************
 *
 * Une continuation peut designer quatre actions possibles :
 *
 * - Ignorer le resultat. Les macros associees sont :
 * 
 * 	IGNCONT		la continuation "ignorer".
 * 	IGNCONTP(c)	teste si c est la continuation "ignorer".
 *
 * - Rendre le resultat par "return".
 *
 * 	RETCONT		la continuation "return".
 * 	RETCONTP(c)	teste si c est la continuation "reurn".
 *
 * - Placer le resultat dans un certain temporaire.
 *
 * 	TMPCONT(t)	la continuation "placer le resultat dans
 * 			le temporaire numero t".
 * 	TMPCONTP(c)	teste si c est une continuation "placer dans
 * 			un temporaire".
 *	CONTTMP(c)	rend le numero du temporaire dans lequel
 * 			doit etre place le resultat.
 *
 * - Se brancher a deux certains labels selon la valeur logique
 *   du resultat.
 *
 * 	LABCONT(l)	la continuation "si le resultat est #t, alors
 * 			sauter au label numero l, sinon sauter au label
 * 			numero l+1".
 * 	LABCONTP(c)	teste si c est une continuation "sauter a deux
 * 			labels".
 * 	CONTLAB(c)	rend le numero du label vers lequel on doit
 * 			sauter si le resultat est #t.
 * 
 **********************************************************************/

/*
 * Ignorer le resultat.
 */

#define IGNCONT		((cont)-1)
#define IGNCONTP(c)	((c) == IGNCONT)


/*
 * Rendre le resultat par "return".
 */

#define RETCONT		((cont)-2)
#define RETCONTP(c)	((c) == RETCONT)


/*
 * Placer le resultat dans un certain temporaire.
 */

#define TMPCONT(t)	((cont)(t))
#define TMPCONTP(c)	((int)(c) >= 0)
#define CONTTMP(c)	((int)(c))


/*
 * Se brancher a deux labels selon la valeur logique du resultat.
 */

#define LABCONT(l)	((cont)(-(l)-3))
#define LABCONTP(c)	((int)(c) < -2)
#define CONTLAB(c)	(-((int)(c))-3)



/**********************************************************************
 *
 *                    Macros de production
 *
 **********************************************************************
 * 
 *	OPEN, CLOSE, POPEN, PCLOSE
 * 			pour la generation de caracteres souvent utilises.
 *	NEWTMP		produit un nouveau numero de variable temporaire.
 * 	GENTMP		genere le nom de la variable temporaire dont
 * 			on specifie le numero.
 * 	DECTMP		genere la declaration d'un temporaire.
 * 	DECVAR		genere une declaration de variable.
 * 	DECREF		variation de DECVAR.
 * 	
 **********************************************************************/

/*
 * Generation de caracteres usuels.
 */

#define OPEN		GENC('{')
#define CLOSE		GENC('}')
#define POPEN		GENC('(')
#define PCLOSE		GENC(')')


/*
 * Variables temporaires.
 */

#define NEWTMP		Temp++
#define DECTMP(n)	{ GENS(ObjType); GENTMP(n); GENC(';'); }


/*
 * Declaration de variables.
 */

#define DECVAR(v)	{ GENS(ObjType); PrintVariableIdent(v); GENC(';'); }
#define DECREF(r)	{ GENS(ObjType); PrintVariableRefIdent(r); GENC(';'); }



static void GenerateGlobalFunction(function *);
static void GeneratePrototype(function *);
static void GenerateForm(form *, cont);
static void GenerateBlock(form *, cont);
static void GenerateCase(form *, cont);
static void GenerateContinue(form *, cont);
static void GenerateExpression(form *, cont);
static void GenerateFlet(form *, cont);
static void GenerateFuncall(form *, cont);
static void GenerateFunction(form *, cont);
static void GenerateIf(form *, cont);
static void GenerateLabels(form *, cont);
static void GenerateLet(form *, cont);
static void GenerateStack_Allocate(form *, cont);
static void GenerateProgn(form *, cont);
static void GenerateReturn_From(form *, cont);
static void GenerateSetq(form *, cont);
static void GenerateStatement(form *, cont);
static void GenerateTest(form *, cont);
static void GenerateThe_Continuation(form *, cont);
static void GenerateVariable(form *, cont);
static void GenerateApplication(form *, cont);
static void Substitute(form *, char *, int, list);
static void GenerateCallList(int, int);
static void GenerateCallListWithFreevars(function *, int, int);
static void DeclareVariablesRefs(list);
static void GenerateLocalFunctions(list, form *, cont);
static void GenerateContBefore(cont);
static void GenerateContAfter(cont);
static int PrepareArguments(list, int *);
static int PrepareCArguments(list);
static void GenerateArgsBinding(form *, list, int, int);



/**********************************************************************
 *
 *                     La fonction principale
 *
 **********************************************************************
 *
 * Parcourt la liste des fonctions globales et produit une definition C
 * pour chacune d'elles.
 *
 **********************************************************************/

void Generate(void)
{
     list l;
     
     for (l = GlobalFunctions; l != NIL; l = CDR(l))
       {			/* Derniere mise au point pour */
				/* les suffixes de fonctions globales. */
	    function *fun = CAR(l);
	         
	    if (fun->status != ROOT) fun->suffix = (GlobalFunctionsCount++);
	    else fun->suffix = 0;
       }

     for (l = GlobalFunctions; l != NIL; l = CDR(l))
       {
	    function *fun = CAR(l);
	    
	    if (fun->status != ROOT)
		 GeneratePrototype(fun);
       }

     for (l = GlobalFunctions; l != NIL; l = CDR(l))
	  GenerateGlobalFunction(CAR(l));
}

static void GeneratePrototype(function *fun)
{
     struct definition *def = fun->def;
     list l;
     char c = '(';
     
     GENS("static ");
     GENS(ObjType);
     PrintFunctionIdent(fun);

     if (Traditional)
       {
	    GENS("()");
       }
     else
       {	/* Parametres explicites. */
	    for (l = def->vrefs; l != NIL; l = CDR(l))
	      {
		   GENC(c);
		   c = ',';
		   GENS(ObjType);
	      }

	        /* Parametres implicites. */
	    for (l = fun->freevars; l != NIL; l = CDR(l))
	      {
		   variable *var = CAR(l);
	    
		   GENC(c);
		   c = ',';
		   GENS(ObjType);
		   if (var->reference) GENC('*');
	      }

	    if (c == '(') GENS("(void");
	    PCLOSE;
       }

     GENC(';');
     GENC('\n');
}

static void GenerateGlobalFunction(function *fun)
{
     struct definition *def = fun->def;
     list l;
     char c = '(';

     Temp = 0;
     CurrentFunction = fun;

     if (SharpLine)
	  PrintSharpLine(def->origin.lineno, def->origin.vlineno,
			 def->origin.vname);
     
     
     if (fun->status != ROOT) GENS("static ");
     else
       {
	    struct hash_cell *c;
	    
	    c = find(DeclareTable, fun->lexvar->name);
	    if (c == NULL)
		 GENS("extern ");
	    else
	      {
		   int class = (int)c->contents;
		   
		   switch (class)
		     {
		     case STATIC:
			  GENS("static ");
			  break;
	    
		     case EXTERN:
			  GENS("extern ");
			  break;
	    
		     case INLINE:
			  GENS("inline ");
			  break;
		     }
	      }
       };
	    
     GENS(ObjType);
     PrintFunctionIdent(fun);

				/* Les parametres de la fonction. */

     for (l = def->vrefs; l != NIL; l = CDR(l))	/* Parametres explicites. */
       {
	    struct variable_ref *vref = CAR(l);
	    
	    GENC(c);
	    c = ',';
	    if (!Traditional)
	      {
		   GENS(ObjType);
	      }
	    PrintVariableRefIdent(vref);
       }
     
     for (l = fun->freevars; l != NIL; l = CDR(l)) /* Parametres implicites. */
       {
	    variable *var = CAR(l);
	    
	    GENC(c);
	    c = ',';
	    if (!Traditional)
	      {
		   GENS(ObjType);
		   if (var->reference) GENC('*');
	      }
	    PrintVariableIdent(var);
       }
     
     if (!Traditional)
       {
	    if (c == '(') GENS("(void");
	    GENC(')');
       }
     else
       {
	    if (c == '(') GENS("()\n");
	    else
	      {
		   PCLOSE;
		   GENC('\n');
		   GENS(ObjType);
		   c = ' ';

		   /* Parametres explicites. */
		   for (l = def->vrefs; l != NIL; l = CDR(l))
		     {
			  struct variable_ref *vref = CAR(l);
	    
			  GENC(c);
			  c = ',';
			  PrintVariableRefIdent(vref);
		     }
     
		   /* Parametres implicites. */	
		   for (l = fun->freevars; l != NIL; l = CDR(l))
		     {
			  variable *var = CAR(l);
	    
			  GENC(c);
			  c = ',';
			  if (var->reference) GENC('*');
			  PrintVariableIdent(var);
		     }
		   GENC(';');
		   GENC('\n');
	      }
       }

     
     OPEN;

     if (fun->contvar != NULL)	/* Memorisation de la continuation d'appel */
       {			/* pour une fonction-bloc ou une fonction */
				/* a continuation. */
	    int buf_tmp = NEWTMP;

	    GENS("jmp_buf ");
	    GENTMP(buf_tmp);
	    GENC(';');
	    GENS(ObjType);
	    PrintVariableIdent(fun->contvar);
	    GENC(';');
	    GENS("if (setjmp(");
	    GENTMP(buf_tmp);
	    GENS(")!=0)return __ContinueValue;else ");
	    PrintVariableIdent(fun->contvar);
	    GENC('=');
	    GENS("KCONT(");
	    GENTMP(buf_tmp);
	    PCLOSE;
	    GENC(';');
	    OPEN;
       }	    
	    
     for (l = fun->labels; l != NIL; l = CDR(l)) /* Les parametres des */
       {			/* fonctions locales compilees en labels. */
	    function *labfun = CAR(l);
	    if (labfun->level <= fun->level)
		 DeclareVariablesRefs(labfun->def->vrefs);
       }

     PrintFunctionLabel(fun);	/* Pour la self-recursion. */
     GENC(':');
     GenerateForm(def->body, RETCONT);

     if ((fun->continued) || (fun->contvar != NULL)) CLOSE;
     CLOSE;
     GENC('\n');
}     



/**********************************************************************
 *
 *                   Generation de code pour une forme
 *
 **********************************************************************
 *
 * Fonctions :
 *
 * 	GenerateForm	genere du code pour une forme quelconque.
 * 	GenerateXXX	genere du code pour une forme XXX.
 *
 * Les fonctions de generation se passent deux paremetres :
 *
 * - La forme a generer.
 * - La continuation.
 *
 * La fonction GenerateTest effectue un traitement particulier de la
 * continuation "sauter au label...".
 *
 **********************************************************************/


static void GenerateForm(form *f, cont c)
{
     if (SharpLine)
	  PrintSharpLine(f->origin.lineno, f->origin.vlineno, f->origin.vname);
     else GENC('\n');

     switch (f->tag)
       {
       case BLOCK:
	    GenerateBlock(f, c);
	    break;
	    
       case CASE:
	    GenerateCase(f, c);
	    break;
	    
       case CONTINUE:
	    GenerateContinue(f, c);
	    break;
	    
       case EXPRESSION:
	    GenerateExpression(f, c);
	    break;
	    
       case FLET:
	    GenerateFlet(f, c);
	    break;
	    
       case FUNCALL:
	    GenerateFuncall(f, c);
	    break;
	    
       case FUNCTION:
	    GenerateFunction(f, c);
	    break;
	    
       case IF:
	    GenerateIf(f, c);
	    break;
	    
       case LABELS:
	    GenerateLabels(f, c);
	    break;
	    
       case LET:
	    GenerateLet(f, c);
	    break;
	    
       case STACK_ALLOCATE:
	    GenerateStack_Allocate(f, c);
	    break;
	    
       case PROGN:
	    GenerateProgn(f, c);
	    break;

       case RETURN_FROM:
	    GenerateReturn_From(f, c);
	    break;
	    
       case SETQ:
	    GenerateSetq(f, c);
	    break;
	    
       case STATEMENT:
	    GenerateStatement(f, c);
	    break;
	    
       case TEST:
	    GenerateTest(f, c);
	    break;
	    
       case THE_CONTINUATION:
	    GenerateThe_Continuation(f, c);
	    break;
	    
       case VARIABLE:
	    GenerateVariable(f, c);
	    break;
	    
       case APPLICATION:
	    GenerateApplication(f, c);
	    break;
	    
       default:
	    fprintf(Err, "%s panic: unknown tag in GenerateForm: %d\n",
		    Me, f->tag);
	    exit(3);
       }
}

static void GenerateBlock(form *f, cont c)
{
     struct block_ref *bref = f->node.block_i.bref;
     form *body = f->node.block_i.body;
     block *blk = bref->block;
     function *fun = blk->fun;

     if (fun->status == DUMMY) GenerateForm(body, c);
     else if (fun->status == GLOBAL)
       {
	    GenerateContBefore(c);
	    PrintFunctionIdent(fun);
	    GenerateCallListWithFreevars(fun, 0, 0);
	    GenerateContAfter(c);
       }
     else
       {
	    int end_label = NEWTMP;
	    
	    blk->cont = c;
	    blk->label = end_label;
	    
	    GenerateForm(body, c);
	    GENTMP(end_label);
	    GENC(':');
	    GENC(';');
       }
}

static void GenerateCase(form *f, cont c)
{
     form *val = f->node.case_i.val;
     list clauses = f->node.case_i.clauses;
     form *def = f->node.case_i.def;
     int val_tmp = NEWTMP;

     list l;
     
     OPEN;

     DECTMP(val_tmp);
     GenerateForm(val, TMPCONT(val_tmp));
     GENS("switch((int)");

     GENTMP(val_tmp);
     PCLOSE;

     OPEN;
     for (l = clauses; l != NIL; l = CDR(l))
       {
	    struct clause *clause = CAR(l);
	    list texts = clause->texts;
	    form *body = clause->body;

	    list m;

	    for (m = texts; m != NIL; m = CDR(m))
	      {
		   GENS("case ");
		   GENS((char *)CAR(m));
		   GENC(':');
	      }

	    GenerateForm(body, c);
	    GENS("break;");
       }
     
     GENS("default:");
     GenerateForm(def, c);
     CLOSE;
     CLOSE;
}

static void GenerateContinue(form *f, cont c)
{
     form *cnt = f->node.continue_i.cont;
     form *val = f->node.continue_i.val;

     int cont_tmp = NEWTMP;
     int val_tmp = NEWTMP;
     
     OPEN;
     DECTMP(cont_tmp);
     DECTMP(val_tmp);
     
     GenerateForm(cnt, TMPCONT(cont_tmp));
     GenerateForm(val, TMPCONT(val_tmp));

     GENS("__ContinueValue=");
     GENTMP(val_tmp);
     GENC(';');
     GENS("longjmp(CCONT(");
     GENTMP(cont_tmp);
     GENS("),1);");

     CLOSE;
}

static void GenerateExpression(form *f, cont c)
{
     char *text = f->node.expression_i.text;
     list args = f->node.expression_i.args;

     int first_tmp;

     OPEN;
     first_tmp = PrepareCArguments(args);
     GenerateContBefore(c);

     Substitute(f, text, first_tmp, args);

     GenerateContAfter(c);
     CLOSE;
}

static void GenerateFlet(form *f, cont c)
{
     GenerateLocalFunctions(f->node.flet_i.defs, f->node.flet_i.body, c);
}

static void GenerateFuncall(form *f, cont c)
{
     form *fun = f->node.funcall_i.fun;
     list args = f->node.funcall_i.args;
     int fun_tmp = NEWTMP;
     int first_tmp;
     int count;
     
     OPEN;
     DECTMP(fun_tmp);
     first_tmp = PrepareArguments(args, &count);
     GenerateForm(fun, TMPCONT(fun_tmp));

     GenerateContBefore(c);
     
     GENS("CFUN(");
     GENTMP(fun_tmp);
     PCLOSE;
     GenerateCallList(first_tmp, count);

     GenerateContAfter(c);
     CLOSE;
}

static void GenerateFunction(form *f, cont c)
{
     GenerateContBefore(c);

     GENS("KFUN(&");
     PrintFunctionRefIdent(f->node.function_i);
     PCLOSE;

     GenerateContAfter(c);
}


static void GenerateIf(form *f, cont c)
{
     form *test = f->node.if_i.test;
     form *iftrue = f->node.if_i.iftrue;
     form *iffalse = f->node.if_i.iffalse;
     int labt = NEWTMP;
     int labf = NEWTMP;
     int labend = NEWTMP;
     
     GenerateForm(test, LABCONT(labt));

     GENTMP(labt);
     GENC(':');
     GenerateForm(iftrue, c);
     GENS("goto ");
     GENTMP(labend);
     GENC(';');
     
     GENTMP(labf);
     GENC(':');
     GenerateForm(iffalse, c);

     GENTMP(labend);
     GENC(':');
     GENC(';');
}

static void GenerateLabels(form *f, cont c)
{
     GenerateLocalFunctions(f->node.flet_i.defs, f->node.flet_i.body, c);
}

static void GenerateLet(form *f, cont c)
{
     list binds = f->node.let_i.binds;
     form *body = f->node.let_i.body;
     list l;
     
     OPEN;

     for (l = binds; l != NIL; l = CDR(l))
       {
	    struct binding *bind = CAR(l);
	    struct variable_ref *vref= bind->vref;
	    DECREF(vref);
       }
     
     for (l = binds; l != NIL; l = CDR(l))
       {
	    struct binding *bind = CAR(l);
	    struct variable_ref *vref= bind->vref;
	    form *val = bind->val;
	    int val_tmp = NEWTMP;
	    
	    OPEN;
	    DECTMP(val_tmp);
	    GenerateForm(val, TMPCONT(val_tmp));
	    PrintVariableRefIdent(vref);
	    GENC('=');
	    GENTMP(val_tmp);
	    GENC(';');
	    CLOSE;
       }
     
     GenerateForm(body, c);
     
     CLOSE;
}

static void GenerateStack_Allocate(form *f, cont c)
{
     list sizes = f->node.stack_allocate_i.sizes;
     form *body = f->node.stack_allocate_i.body;
     list l;
     
     OPEN;

     for (l = sizes; l != NIL; l = CDR(l))
       {
	    struct sizing *size = CAR(l);
	    struct variable_ref *vref= size->vref;
	    int val_tmp = NEWTMP;
	    int i = size->size;

	    GENS(ObjType); 
	    GENTMP(val_tmp);
	    GENC('[');
	    fprintf(Out,"%d",i);
	    GENC(']');
	    GENC(',');
	    PrintVariableRefIdent(vref);
	    GENC('=');
	    GENS("KPTR");
	    GENC('(');
	    GENTMP(val_tmp);
	    GENC(')');
	    GENC(';');
       }
     
     GenerateForm(body, c);
     
     CLOSE;
}

static void GenerateProgn(form *f, cont c)
{
     list body = f->node.progn_i.body;
     list l;
     
     for (l = body; l != NIL; l = CDR(l))
       {
	    if (CDR(l) != NIL)
		 GenerateForm(CAR(l), IGNCONT);
	    else GenerateForm(CAR(l), c);
       }
}

static void GenerateReturn_From(form *f, cont c)
{
     struct block_ref *bref = f->node.return_from_i.bref;
     form *val = f->node.return_from_i.val;
     block *blk = bref->block;
     function *fun = blk->fun;
     
     Infos.exits++;

     if (fun->status != GLOBAL)
       {
	    GenerateForm(val, blk->cont);
	    GENS("goto ");
	    GENTMP(blk->label);
	    GENC(';');
       }
     else if ((fun == CurrentFunction) && (OptimLevel > 1))
	  GenerateForm(val, RETCONT);
     else
       {
	    variable *var = fun->contvar;
	    int val_tmp = NEWTMP;

	    Infos.longjmps++;
	    
	    OPEN;
	    DECTMP(val_tmp);
	    
	    GenerateForm(val, TMPCONT(val_tmp));
	    GENS("__ContinueValue=");
	    GENTMP(val_tmp);
	    GENC(';');
	    GENS("longjmp(CCONT(");
	    PrintVariableIdent(var);
	    GENS("),1);");
	    
	    CLOSE;
       }
}

static void GenerateSetq(form *f, cont c)
{
     struct variable_ref *vref = f->node.setq_i.vref;
     form *val = f->node.setq_i.val;
     variable *var = vref->var;
     int val_tmp = NEWTMP;

     OPEN;
     DECTMP(val_tmp);
     GenerateForm(val, TMPCONT(val_tmp));

     if ((var != NULL) && (var->reference)
	 && (var->level <= CurrentFunction->level))
	  GENC('*');

     PrintVariableRefIdent(vref);
     GENC('=');
     GENTMP(val_tmp);
     GENC(';');
     GenerateContBefore(c);
     GENTMP(val_tmp);
     GenerateContAfter(c);
     CLOSE;
}

static void GenerateStatement(form *f, cont c)
{
     char *text = f->node.statement_i.text;
     list args = f->node.statement_i.args;

     int first_tmp;

     OPEN;
     first_tmp = PrepareCArguments(args);

     Substitute(f, text, first_tmp, args);
     GENC(';');

     GenerateContBefore(c);
     GENS(False_lit);
     GenerateContAfter(c);
     CLOSE;
}

static void GenerateTest(form *f, cont c)
{
     char *text = f->node.test_i.text;
     list args = f->node.test_i.args;

     int first_tmp;

     OPEN;
     first_tmp = PrepareCArguments(args);

     if (!LABCONTP(c))
       {
	    GenerateContBefore(c);
	    POPEN;
	    Substitute(f, text, first_tmp, args);
	    GENC('?');
	    GENS(True_lit);
	    GENC(':');
	    GENS(False_lit);
	    PCLOSE;
	    GenerateContAfter(c);
       }
     else
       {
	    int labt = CONTLAB(c);
	    int labf = labt+1;
	    
	    GENS("if (");
	    Substitute(f, text, first_tmp, args);
	    GENS(")goto ");
	    GENTMP(labt);
	    GENS(";else goto ");
	    GENTMP(labf);
	    GENC(';');
       }
     
     CLOSE;
}

static void GenerateThe_Continuation(form *f, cont c)
{
     GenerateContBefore(c);
     PrintVariableIdent(f->node.the_continuation_i.contvar);
     GenerateContAfter(c);
}

static void GenerateVariable(form *f, cont c)
{	
     struct variable_ref *vref = f->node.variable_i;
     variable *var = vref->var;
     
     GenerateContBefore(c);

     if ((var != NULL) && (var->reference)
	 && (var->level <= CurrentFunction->level))
	  GENC('*');

     PrintVariableRefIdent(vref);

     GenerateContAfter(c);
}

static void GenerateApplication(form *f, cont c)
{
     struct function_ref *fref = f->node.application_i.fref;
     list args = f->node.application_i.args;
     function *fun = fref->fun;
     int count;
     int first_tmp;
     
     Infos.calls++;

     OPEN;
     first_tmp = PrepareArguments(args, &count);
     
     if (fun == NULL)		/* Fonction globale. */
       {
	    GenerateContBefore(c);
	    PrintFunctionRefIdent(fref);
	    GenerateCallList(first_tmp, count);
	    GenerateContAfter(c);
       }
     else
       {
	    Infos.lcalls++;

	    if (((fun->status == LABEL) && (fun->generated == TRUE))
		|| ((fun == CurrentFunction) && RETCONTP(c)
		    && (OptimLevel > 0)))
	      {			/* Appel a label deja genere, ou */
				/* self-appel a une fonction globale. */
		   struct definition *def = fun->def;
		   struct function_ref *fref = def->fref;
		   list vrefs = def->vrefs;
	    
		   OPEN;
		   GenerateArgsBinding(f, vrefs, first_tmp, count);
	    
		   GENS("goto ");
		   PrintFunctionRefLabel(fref);
		   GENC(';');
		   CLOSE;
	      }
	    else if ((fun->status == LABEL) && (fun->generated == FALSE))
	      {			/* Premier appel a fonction locale. */
		   struct definition *def = fun->def;
		   list vrefs = def->vrefs;
	    
		   OPEN;
		   GenerateArgsBinding(f, vrefs, first_tmp, count);
	    
		   PrintFunctionRefLabel(def->fref);
		   GENC(':');
		   fun->generated = TRUE; /* Pour ne pas generer deux fois. */

		   GenerateForm(def->body, c);
		   CLOSE;
	      }
	    else
	      {			/* Appel de fonction globale. */
		   Infos.gosubs++;

		   GenerateContBefore(c);
		   PrintFunctionIdent(fun);
		   GenerateCallListWithFreevars(fun, first_tmp, count);
		   GenerateContAfter(c);
	      }
       }
     
     CLOSE;
}



/**********************************************************************
 *
 *                 Auxiliaires de generation
 *
 **********************************************************************
 *
 * Fonctions :
 *
 *  PrepareArguments	prend une liste d'arguments, calcule leurs
 * 			valeurs, les place dans des temporaires de numeros
 * 			consecutifs, et rend deux resultats : le numero
 * 			du premier temporaire et le nombre d'arguments.
 *  PrepareCArguments	idem que PrepareArguments, mais ne genere pas de
 *			variable temporaire pour les arguments qui sont
 *			des variables ou des litteraux. Utilise pour
 *			preparer les arguments des formes EXPRESSION,
 *			STATEMENT et TEST.
 *  GenerateArgsBinding	exploite le travail de PrepareArguments : affecte
 * 			les valeurs des arguments aux parametres d'une
 * 			fonction.
 *  Substitute		exploite le travail de PrepareCArguments : genere
 * 			le code C specifie par une chaine de controle du
 * 			type de celle donnee en argument aux formes
 * 			EXPRESSION, STATEMENT et TEST.
 *  GenerateCallList	exploite le travail de PrepareArguments : genere
 * 			une liste d'arguments pour un appel de fonction C.
 *  GenerateCallListWithFreevars
 *			idem, mais on est dans le cas d'un appel a une
 * 			fonction locale et on ajoute ses variables libres
 * 			a ses arguments.
 *
 *  GenerateContBefore	produit le code C preparant l'"application
 * 			d'une continuation" sur un resultat.
 *  GenerateContAfter	produit le code C cloturant l'"application
 * 			d'une continuation" sur un resultat.
 *
 *  GenerateLocalFunctions
 *			Auxiliaire appelee par GenerateFlet et GenerateLabels.
 *
 *  PrintSharpLine	affiche une directive #line.
 *
 **********************************************************************/


/*
 * Le calcul d'une liste d'arguments et son utilisation.
 */

static int PrepareArguments(list args, int *count_res)
{
     int first_tmp = Temp;
     int count = 0;
     int i;
     list l;
     
     for (l = args; l != NIL; l = CDR(l))
       {
	    int tmp = NEWTMP;
	    DECTMP(tmp);
	    count++;
       }
     
     l = args;
     for (i = 0; i<count; i++, l = CDR(l))
	  GenerateForm(CAR(l), TMPCONT(first_tmp+i));

     *count_res = count;
     return first_tmp;
}

static int PrepareCArguments(list args)
{
     int first_tmp = Temp;
     int i;
     list l;
     
     for (l = args; l != NIL; l = CDR(l))
       {
	    form *f = CAR(l);
	    if (f->tag != VARIABLE)
	    {
		   int tmp = NEWTMP;
	    	   DECTMP(tmp);
	    }
       }
     
     for (l = args, i = 0; l != NIL; l = CDR(l))
       {
	    form *f = CAR(l);
	    if (f->tag != VARIABLE)
	    	   GenerateForm(CAR(l), TMPCONT(first_tmp+(i++)));
       }

     return first_tmp;
}

static void GenerateArgsBinding(form *f, list vrefs, int first_tmp, int count)
{
     list l;
     int i = 0;

     for (l = vrefs; l != NIL; l = CDR(l), i++)
       {
	    struct variable_ref *vref = CAR(l);
	    
	    if (i == count)
	      {
		   PrintError("too few arguments to (known) function", 
			      f->origin.lineno, f->origin.vlineno, 
			      f->origin.vname);
		   longjmp(DefunExit, TRUE);
	      }
	    
	    PrintVariableRefIdent(vref);
	    GENC('=');
	    GENTMP(first_tmp+i);
	    GENC(';');
       }
     
     if (i != count)
       {
	    PrintError("too many arguments to (known) function", 
		       f->origin.lineno, f->origin.vlineno, 
		       f->origin.vname);
	    longjmp(DefunExit, TRUE);
       }
}

static void Substitute(form *f, char *text, int first_tmp, list args)
{
     char *c;
     int i = first_tmp;
     list l = args;

     for (c = text; *c != '\0'; c++)
       {
	    if (*c != '\01') GENC(*c);
	    else if (l != NIL)
	      {
	      	   form *f = CAR(l);
	      	   
	      	   switch (f->tag)
	      	   {
	      	   	case VARIABLE:
	      	         {
	      	            struct variable_ref *vref = f->node.variable_i;
			    variable *var = vref->var;

			    if ((var != NULL) && (var->reference)
				&& (var->level <= CurrentFunction->level))
				 GENC('*');
  
			    PrintVariableRefIdent(vref);
			    break;
		         }
		     	default:
	      	     	   GENTMP(i);
	      	     	   i++;
	      	     	   break;
	      	   }
		     
		   l = CDR(l);
	      }
	    else
	      {
		   PrintError("inlined arguments exhausted", 
			      f->origin.lineno, f->origin.vlineno, 
			      f->origin.vname);
		   longjmp(DefunExit, TRUE);
	      }
       }

     if (l != NIL)
       {
	    PrintError("too many inlined arguments", 
		       f->origin.lineno, f->origin.vlineno, 
		       f->origin.vname);
	    longjmp(DefunExit, TRUE);
       }
}

static void GenerateCallList(int first_tmp, int count)
{
     int i = 0;
     char c = '(';

     for (i = 0; i != count; i++)
	{
	     GENC(c);
	     c = ',';
	     GENTMP(first_tmp+i);
	}
	     
     if (c == '(') POPEN;
     
     PCLOSE;

     Infos.args += count;
}

static void 
     GenerateCallListWithFreevars(function *fun, int first_tmp, int count)
{
     list l;
     int i = 0;
     char c = '(';

     for (i = 0; i != count; i++)
	{
	     GENC(c);
	     c = ',';
	     GENTMP(first_tmp+i);
	}

     Infos.args += count;
     
     for (l = fun->freevars; l != NIL; l = CDR(l))
       {
	    variable *var = CAR(l);
	    
	    GENC(c);
	    c = ',';
	    if ((var->reference) && !(var->level <= CurrentFunction->level))
		 GENC('&');
	    PrintVariableIdent(var);

	    Infos.args++;
       }
	     
     if (c == '(') POPEN;
     
     PCLOSE;
}

static void GenerateContBefore(cont c)
{
     if (IGNCONTP(c))
       {
	    if (!Traditional) GENS("(void)");
       }
     else if (RETCONTP(c))
       {
	    GENS("return ");
       }
     else if (TMPCONTP(c))
       {
	  GENTMP(CONTTMP(c));
	  GENC('=');
       }
     else if (LABCONTP(c))
       {
	    GENS("if (");
       }
     else
       {
 	    fprintf(Err, "%s panic: unknown continuation: %d\n",
		    Me, c);
	    exit(3);
       }
}

static void GenerateContAfter(cont c)
{
     if (!LABCONTP(c)) GENC(';');
     else
       {
	    int labt = CONTLAB(c);
	    int labf = labt+1;
	    
	    GENS("!=");
	    GENS(False_lit);
            GENS(")goto ");
	    GENTMP(labt);
	    GENS(";else goto ");
	    GENTMP(labf);
	    GENC(';');
       }
}

static void GenerateLocalFunctions(list defs, form *body, cont c)
{
     list l;
     
     OPEN;
     
     for (l = defs; l != NIL; l = CDR(l))
       {
	    struct definition *def = CAR(l);
	    function *fun = def->fref->fun;
	    if ((fun->status == LABEL) && (fun->labelof == CurrentFunction))
		 DeclareVariablesRefs(def->vrefs);
       }

     GenerateForm(body, c);
     
     CLOSE;
}

static void DeclareVariablesRefs(list vrefs)
{
     list l;
     
     for (l = vrefs; l != NIL; l = CDR(l))
       {
	    struct variable_ref *vref = CAR(l);
	    DECREF(vref);
       }
}

void PrintSharpLine(int lineno, int vlineno, char *vname)
{
     GENC('\n');
     GENS("#line ");
     
     fprintf(Out, "%d ", (vlineno >= 0 ? vlineno : lineno));

     if (vname != NULL) 
       {
	    GENC('"');
            GENS(vname);
	    GENC('"');
       }
     else if (In != stdin) 
       {
	    GENC('"');
            GENS(InName);
	    GENC('"');
       }     

     GENC('\n');
}

