/***************************************************************************
  Copyright (C) Nitsan Seniak 1989, 1990

  This file is part of the K2 compiler.
  Permission to copy this software, in whole or in part, to use this
  software for any lawful noncommercial purpose, and to redistribute
  this software is granted subject to the restriction that all copies
  made of this software must include this copyright notice in full.
  The author(s) makes no warranties or representations of any kind, either
  express or implied, including but not limited to implied warranties
  of merchantability or fitness for any particular purpose.
  All materials developed as a consequence of the use of this
  software shall duly acknowledge such use, in accordance with the usual
  standards of acknowledging credit in research.
 ***************************************************************************/ 

/*****************************************************************************
 *                                                                           *
 *                               SYNTAXE DE K2                               *
 *                                                                           *
 *****************************************************************************
 *
 * Ce module contient des directives YACC et des actions d'internement des
 * formes syntaxiques de K2 ; en outre, les reductions de "(:defun ...)"
 * et de "(:defvar ...)" entrainent l'appel des fonctions de compilation
 * et de generation de code.
 *
 * A chaque forme internee sont attachees les coordonnees dans le source
 * du debut de la forme, telles qu'elles sont transmises par l'analyseur
 * lexical dans des variables globales.
 *
 * Il ne se passe rien de bien palpitant dans ce module ; voir plutot
 * dans yystype.h une description des types utilises pour representer
 * les formes syntaxiques.
 *
 * Important : tous les objets crees sont alloues dans le tas temporaire,
 * sauf exception.
 * 
 *****************************************************************************/



/*****************************************************************************
 *
 *                              Declarations
 *
 *****************************************************************************
 *
 * Ici sont declares les tokens, auxquels Yacc attribue des numeros distincts.
 *
 * Les tokens VAR, NUM et TEXT correspondent aux variables, entiers et au 
 * texte C ; les autres correspondent aux differentes formes syntaxiques, 
 * a l'exception de WRONG qui est utilise seulement pour rattraper les erreurs
 * d'analyse lexicale.
 *
 ****************************************************************************/
%{

#include <stdlib.h>
#include <stdio.h>
#include <setjmp.h>
#include "util.h"
#include "objects.h"
#include "yystype.h"
#include "atoms.h"
#include "forms.h"
#include "k2.h"

#define YYMAXDEPTH 3000		/* Moins ridicule que le 150 par defaut. */


static void DeclareExpression(char *, char *);
static void DeclareTest(char *, char *);
static void DeclareStatement(char *, char *);
static void DeclareKeyword(int, char *, char *);

static struct variable_ref *Variable_ref(struct lex_variable *);
static struct function_ref *Function_ref(struct lex_variable *);
static struct block_ref *Block_ref(struct lex_variable *);
static int *Int_ref(struct lex_variable *);

static struct definition *Definition(void);
static struct definition *SetDefinition(struct definition *, 
					struct function_ref *, list, form *);
static struct binding *Binding(struct variable_ref *, form *);
static struct sizing *Sizing(struct variable_ref *, int *);
static struct clause *Clause(list, form *);

static form *Form(void);
static form *BlockForm(form *, struct block_ref *, form *);
static form *CaseForm(form *, form *, list, form *);
static form *ContinueForm(form *, form *, form *);
static form *ExpressionForm(form *, char *, list);
static form *FletForm(form *, list, form *);
static form *FuncallForm(form *, form *, list);
static form *FunctionForm(form *, struct function_ref *);
static form *IfForm(form *, form *, form *, form *);
static form *LabelsForm(form *, list, form *);
static form *LetForm(form *, list, form *);
static form *Stack_AllocateForm(form *, list, form *);
static form *PrognForm(form *, list);
static form *Return_FromForm(form *, struct block_ref *, form *);
static form *SetqForm(form *, struct variable_ref *, form *);
static form *StatementForm(form *, char *, list);
static form *TestForm(form *, char *, list);
static form *The_ContinuationForm(form *);
static form *KeywordForm(form *, struct keyword *, list);
static form *ApplicationForm(form *, struct function_ref *, list);

static form *VariableForm(struct variable_ref *);

static int vcount;		/* Pour compter les arguments d'un */
				/* prototype... */
%}


%start file

%token WRONG

%token DEFUN DEFVAR DECLARATION DECLARE STATIC EXTERN INLINE
%token TEXT VAR UNIN_KEYWORD KEYWORD NUM

%token BLOCK CASE CONTINUE EXPRESSION FLET FUNCALL FUNCTION
%token IF LABELS LET OTHERWISE PROGN RETURN_FROM
%token SETQ STATEMENT TEST THE_CONTINUATION STACK_ALLOCATE

%token VARIABLE APPLICATION



%%

/****************************************************************************
 *
 *                                 Regles
 *
 ****************************************************************************
 *
 * Pour permettre l'annotation correcte des formes syntaxiques avec leurs
 * coordonnees dans le texte source, celles-ci sont construites en deux
 * etapes : premierement, des l'entree dans la regle la structure
 * adequate est allouee et ses coordonnees sont initialisees ; puis, a
 * la sortie de la regle, ses champs sont correctement initialises.
 *
 ****************************************************************************/


/*
 * Regles produisant des lexemes.
 */

var
	: VAR				{ $$ = Variable_ref($1); }
	;

fun
	: VAR				{ $$ = Function_ref($1); }
	;

blk
        : VAR				{ $$ = Block_ref($1); }
        ;

integer
        : NUM                           { $$ = Int_ref($1); }
        ;


/*
 * Formes de toplevel, provoquant la compilation.
 */

file
	: /* empty */
	| file { InitPre(); } mform { InitPost(); }
	;

mform
	: '(' mform_body
	;

mform_body
	: DEFUN def_body ')'		{ CompileDefun($2); }
	| DEFVAR var ')'		{ CompileDefvar($2, Lineno,
							VirtualLineno,
							VirtualInName); }
	| DECLARATION TEXT vars ')'	{ CompileDeclaration($2, $3, Lineno, 
							     VirtualLineno,
							     VirtualInName); }
	| DECLARE declaration ')'
	;

/*
 * Sous-formes de toplevel.
 */

declaration
	: EXPRESSION UNIN_KEYWORD TEXT	{ DeclareExpression($2, $3); }
	| TEST UNIN_KEYWORD TEXT	{ DeclareTest($2, $3); }
	| STATEMENT UNIN_KEYWORD TEXT	{ DeclareStatement($2, $3); }
	| STATIC DEFUN VAR '(' VARs ')' { FunPrototype($3, vcount, STATIC,
						       Lineno, VirtualLineno,
						       VirtualInName); }
	| EXTERN DEFUN VAR '(' VARs ')' { FunPrototype($3, vcount, EXTERN,
						       Lineno, VirtualLineno,
						       VirtualInName); }
	| INLINE DEFUN VAR '(' VARs ')' { FunPrototype($3, vcount, INLINE,
						       Lineno, VirtualLineno,
						       VirtualInName); }
	| STATIC DEFVAR VAR		{ VarPrototype($3, STATIC,
						       Lineno, VirtualLineno,
						       VirtualInName); }
	| EXTERN DEFVAR VAR		{ VarPrototype($3, EXTERN,
						       Lineno, VirtualLineno,
						       VirtualInName); }
	;

VARs	: { vcount = 0; } VARs_list
	;

VARs_list
	:
	| VARs_list { vcount++; } VAR 
	;       

/*
 * Sous-formes.
 */

def
	: '(' def_body ')'		{ $$ = $2; }
	;

def_body
	: { $$ = Definition(); } fun '(' vars ')' body
                                        { $$ = SetDefinition($1, $2, $4, $6); }
	;

body
	: { $$ = Form(); } form forms
					{ $$ = PrognForm($1, cons($2, $3)); }
	;

args
	: forms				{ $$ = $1; }
	;

bind
	: '(' var form ')'		{ $$ = Binding($2, $3); }
	;

size
	: '(' var integer ')'		{ $$ = Sizing($2, $3); }
	;

clause
	: '(' texts body ')'	{ $$ = Clause($2, $3); }
	;

default_clause
	: OTHERWISE body	{ $$ = $2; }
	;


/*
 * Les regles suivantes sont obtenues par expansion par cpp du fichier
 * forms.y.include.
 */

defs : defs_list { $$ = qlist($1); } ; defs_list :  { $$ = emptyq(); } | defs_list def { $$ = addq($2, $1); } ;
vars : vars_list { $$ = qlist($1); } ; vars_list :  { $$ = emptyq(); } | vars_list var { $$ = addq($2, $1); } ;
forms : forms_list { $$ = qlist($1); } ; forms_list :  { $$ = emptyq(); } | forms_list form { $$ = addq($2, $1); } ;
binds : binds_list { $$ = qlist($1); } ; binds_list :  { $$ = emptyq(); } | binds_list bind { $$ = addq($2, $1); } ;
sizes : sizes_list { $$ = qlist($1); } ; sizes_list :  { $$ = emptyq(); } | sizes_list size { $$ = addq($2, $1); } ;
clauses : clauses_list { $$ = qlist($1); } ; clauses_list :  { $$ = emptyq(); } | clauses_list clause { $$ = addq($2, $1); } ;
texts : texts_list { $$ = qlist($1); } ; texts_list :  { $$ = emptyq(); } | texts_list TEXT { $$ = addq($2, $1); } ;


/*
 * Les formes.
 */

form
	: var				{ $$ = VariableForm($1); }
	| '(' form_switch ')'		{ $$ = $2; }
	;

form_switch
        : block
	| case
	| continue
	| expression
	| flet
	| funcall
	| function
	| if
	| labels
	| let
        | stack_allocate
	| progn
        | return_from
	| setq
  	| statement
	| test
	| the_continuation
	| keyword
	| application
	;

block
     	: BLOCK { $$ = Form(); } blk body
     			{ $$ = BlockForm($2, $3, $4); }
	;

case
	: CASE { $$ = Form(); } form clauses default_clause
			{ $$ = CaseForm($2, $3, $4, $5); }
	;

continue
	: CONTINUE { $$ = Form(); } form form
			{ $$ = ContinueForm($2, $3, $4); }
	;

expression
	: EXPRESSION { $$ = Form(); } TEXT args
			{ $$ = ExpressionForm($2, $3, $4); }
	;

flet
	: FLET { $$ = Form(); } '(' defs ')' body
			{ $$ = FletForm($2, $4, $6); }
	;

funcall
	: FUNCALL { $$ = Form(); } form args
			{ $$ = FuncallForm($2, $3, $4); }
	;

function
	: FUNCTION { $$ = Form(); } fun
			{ $$ = FunctionForm($2, $3); }
	;

if
	: IF { $$ = Form(); } form form form
			{ $$ = IfForm($2, $3, $4, $5); }
	;

labels
	: LABELS { $$ = Form(); } '(' defs ')' body
			{ $$ = LabelsForm($2, $4, $6); }
	;

let
	: LET { $$ = Form(); } '(' binds ')' body
			{ $$ = LetForm($2, $4, $6); }
	;

stack_allocate
	: STACK_ALLOCATE { $$ = Form(); } '(' sizes ')' body
                                { $$ = StackAllocateForm($2, $4, $6); }
	;

progn
	: PROGN body	{ $$ = $2; }
	;

return_from
	: RETURN_FROM { $$ = Form(); } blk form
			{ $$ = Return_FromForm($2, $3, $4); }
	;

setq
	: SETQ { $$ = Form(); } var form
			{ $$ = SetqForm($2, $3, $4); }
	;

statement
	: STATEMENT { $$ = Form(); } TEXT args
			{ $$ = StatementForm($2, $3, $4); }
	;

test
	: TEST { $$ = Form(); } TEXT args
			{ $$ = TestForm($2, $3, $4); }
	;

the_continuation
	: THE_CONTINUATION
			{ $$ = The_ContinuationForm(Form()); }
	;

keyword
	: KEYWORD { $$ =  Form(); } args
			{ $$ = KeywordForm($2, $1, $3); }

application
	: { $$ = Form(); } fun args
			{ $$ = ApplicationForm($1, $2, $3); }
	;



%%


/**********************************************************************
 *
 * Declaration d'un mot-clef abreviation
 *
 **********************************************************************
 *
 * Fonction :
 *
 * 	DeclareKeyword	enregistre la declaration d'une abreviation
 *
 **********************************************************************/

static void DeclareExpression(char *id, char *text)
{
     DeclareKeyword(EXPRESSION, id, text);
}

static void DeclareTest(char *id, char *text)
{
     DeclareKeyword(TEST, id, text);
}

static void DeclareStatement(char *id, char *text)
{
     DeclareKeyword(STATEMENT, id, text);
}

static void DeclareKeyword(int token, char *id, char *text)
{
     struct hash_cell *c;
     struct keyword *k = PNEW(struct keyword);
     
     k->token = token;
     k->text = pscopy(text);
     
     c = find_create(KeywordsTable, id);
     c->contents = k;
}




/**********************************************************************
 *
 * Internement des formes et des sous-formes
 *
 **********************************************************************
 *
 * Fonctions :
 *
 *	Variable_ref	construit une reference a une variable.
 *	Function_ref	construit une reference a une fonction.
 *	Block_ref	construit une reference a un bloc lexical.
 *	Int_ref	        construit une reference a un entier.
 *
 *	Form		construit une forme en n'initialisant que
 * 			les champs concernant sa position dans le source.
 *	XXXForm		construit la forme XXX.
 *
 **********************************************************************/


/*
 * References aux variables et fonctions.
 */

static struct variable_ref *Variable_ref(struct lex_variable *lexvar)
{
     struct variable_ref *vref = NEW(struct variable_ref);
     
     vref->lexvar = lexvar;
     vref->origin.lineno = Lineno;
     vref->origin.vlineno = VirtualLineno;
     vref->origin.vname = VirtualInName;
     vref->var = NULL;
     return vref;
}

static struct function_ref *Function_ref(struct lex_variable *lexvar)
{
     struct function_ref *fref = NEW(struct function_ref);
     
     fref->lexvar = lexvar;
     fref->origin.lineno = Lineno;
     fref->origin.vlineno = VirtualLineno;
     fref->origin.vname = VirtualInName;
     fref->fun = NULL;
     return fref;
}

static struct block_ref *Block_ref(struct lex_variable *lexvar)
{
     struct block_ref *bref = NEW(struct block_ref);

     bref->lexvar = lexvar;
     bref->origin.lineno = Lineno;
     bref->origin.vlineno = VirtualLineno;
     bref->origin.vname = VirtualInName;
     return bref;
}


static int *Int_ref(struct lex_variable *lexvar)
{
     int *iref = NEW(int);

     *iref = atoi((char *)lexvar);
     return iref;
}


/*
 * Sous-formes.
 */

static struct definition *Definition(void)
{
     struct definition *def = NEW(struct definition);
     
     def->origin.lineno = Lineno;
     def->origin.vlineno = VirtualLineno;
     def->origin.vname = VirtualInName;
     
     return def;
}
     
static 
struct definition *SetDefinition(struct definition *def,
				 struct function_ref *fref,
				 list vrefs, form *body)
{
     def->fref = fref;
     def->vrefs = vrefs;
     def->body = body;

     return def;
}

static struct binding *Binding(struct variable_ref *vref, form *val)
{
     struct binding *b = NEW(struct binding);
     
     b->vref = vref;
     b->val = val;

     return b;
}

static struct sizing *Sizing(struct variable_ref *vref, int *size)
{
     struct sizing *b = NEW(struct sizing);
     
     b->vref = vref;
     b->size = *size;

     return b;
}

static struct clause *Clause(list texts, form *body)
{
     struct clause *c = NEW(struct clause);
     
     c->texts = texts;
     c->body = body;
     return c;
}


/*
 * Formes.
 */

static form *Form(void)
{
     form *f = NEW(form);
     
     f->origin.lineno = Lineno;
     f->origin.vlineno = VirtualLineno;
     f->origin.vname = VirtualInName;

     return f;
}

static form *BlockForm(form *f, struct block_ref *bref, form *body)
{
     f->tag = BLOCK;
     f->node.block_i.bref = bref;
     f->node.block_i.body = body;
     return f;
}


static form *CaseForm(form *f, form *val, list clauses, form *def)
{
     if (clauses == NIL) return def;
     else
       {
	    
	    f->tag = CASE;
	    f->node.case_i.val = val;
	    f->node.case_i.clauses = clauses;
	    f->node.case_i.def = def;
	    
	    return f;
       };
}

static form *ContinueForm(form *f, form *c, form *val)
{
     f->tag = CONTINUE;
     f->node.continue_i.cont = c;
     f->node.continue_i.val = val;
     
     return f;
}

static form *ExpressionForm(form *f, char *text, list args)
{
     f->tag = EXPRESSION;
     f->node.expression_i.text = text;
     f->node.expression_i.args = args;
     
     return f;
}

static form *FletForm(form *f, list defs, form *body)
{

     if (defs == NIL) return body;
     else
       {
	    f->tag = FLET;
	    f->node.flet_i.defs = defs;
	    f->node.flet_i.body = body;
	    
	    return f;
       };
}

static form *FuncallForm(form *f, form *fun, list args)
{
     f->tag = FUNCALL;
     f->node.funcall_i.fun = fun;
     f->node.funcall_i.args = args;
	    
     return f;
}

static form *FunctionForm(form *f, struct function_ref *fref)
{
     f->tag = FUNCTION;
     f->node.function_i = fref;
     
     return f;
}

static form *IfForm(form *f, form *test, form *iftrue, form *iffalse)
{
     f->tag = IF;
     f->node.if_i.test = test;
     f->node.if_i.iftrue = iftrue;
     f->node.if_i.iffalse = iffalse;
     
     return f;
}

static form *LabelsForm(form *f, list defs, form *body)
{
     if (defs == NIL) return body; /* Aucun interet... mais bof... */
     else
       {
	    f->tag = LABELS;
	    f->node.labels_i.defs = defs;
	    f->node.labels_i.body = body;
	    
	    return f;
       };
}

static form *LetForm(form *f, list binds, form *body)
{
     if (binds == NULL) return body; /* Idem */
     else
       {
	    f->tag = LET;
	    f->node.let_i.binds = binds;
	    f->node.let_i.body = body;
	    
	    return f;
       };
}

static form *StackAllocateForm(form *f, list sizes, form *body)
{
     if (sizes == NULL) return body; /* Idem */
     else
       {
	    f->tag = STACK_ALLOCATE;
	    f->node.stack_allocate_i.sizes = sizes;
	    f->node.stack_allocate_i.body = body;
	    
	    return f;
       };
}

static form *PrognForm(form *f, list body)
{

     if (CDR(body) == NIL) return (form *)CAR(body); /* Ibidem */
     else
       {
	    f->tag = PROGN;
	    f->node.progn_i.body = body;
	    
	    return f;
       };
}

static form *Return_FromForm(form *f, struct block_ref *bref, form *val)
{
     f->tag = RETURN_FROM;
     f->node.return_from_i.bref = bref;
     f->node.return_from_i.val = val;
     return f;
}

static form *SetqForm(form *f, struct variable_ref *vref, form *val)
{
     f->tag = SETQ;
     f->node.setq_i.vref = vref;
     f->node.setq_i.val = val;
     
     return f;
}

static form *StatementForm(form *f, char *text, list args)
{
     f->tag = STATEMENT;
     f->node.statement_i.text = text;
     f->node.statement_i.args = args;
     
     return f;
}

static form *TestForm(form *f, char *text, list args)
{
     f->tag = TEST;
     f->node.test_i.text = text;
     f->node.test_i.args = args;
     
     return f;
}

static form *The_ContinuationForm(form *f)
{
     f->tag = THE_CONTINUATION;
     f->node.the_continuation_i.contvar = NULL;
     return f;
}

static form *KeywordForm(form *f, struct keyword *key, list args)
{
     switch (key->token)
       {
       case EXPRESSION:
	    f->tag = EXPRESSION;
	    f->node.expression_i.text = key->text;
	    f->node.expression_i.args = args;
	    break;

       case TEST:
	    f->tag = TEST;
	    f->node.test_i.text = key->text;
	    f->node.test_i.args = args;
	    break;

       case STATEMENT:
	    f->tag = STATEMENT;
	    f->node.statement_i.text = key->text;
	    f->node.statement_i.args = args;
	    break;
       }
     
     return f;
}

static form *ApplicationForm(form *f, struct function_ref *fref, list args)
{
     f->tag = APPLICATION;
     f->node.application_i.fref = fref;
     f->node.application_i.args = args;
     
     return f;
}

static form *VariableForm(struct variable_ref *n)
{
     form *f = Form();
     
     f->tag = VARIABLE;
     f->node.variable_i = n;
     
     return f;
}
