/***************************************************************************
  Copyright (C) Nitsan Seniak 1989, 1990

  This file is part of the K2 compiler.
  Permission to copy this software, in whole or in part, to use this
  software for any lawful noncommercial purpose, and to redistribute
  this software is granted subject to the restriction that all copies
  made of this software must include this copyright notice in full.
  The author(s) makes no warranties or representations of any kind, either
  express or implied, including but not limited to implied warranties
  of merchantability or fitness for any particular purpose.
  All materials developed as a consequence of the use of this
  software shall duly acknowledge such use, in accordance with the usual
  standards of acknowledging credit in research.
 ***************************************************************************/ 

/**********************************************************************
 *                                                                    *
 *          COLLECTE D'INFORMATIONS SUR UNE FONCTION A COMPILER       *
 *                                                                    *
 **********************************************************************
 *
 * Ce module contient un arpenteur qui decore les formes syntaxiques
 * avec des informations collectees, et qui cree les objets "fonction",
 * "variable" et "block" du programme.
 *
 * La principale tache de l'arpenteur consiste a resoudre les references, 
 * ie a assigner aux references de variables,  fonctions et blocs locaux
 * les objets "fonction", "variable" et "block" correspondants. Les references
 * non resolues a des fonctions ou des variables sont considerees comme
 * concernant des entites globales, qui sont hors de la visibilite du
 * compilateur.
 *
 **********************************************************************/

#include <stdlib.h>
#include <stdio.h>
#include <setjmp.h>
#include "util.h"
#include "objects.h"
#include "yystype.h"
#include "k2.h"
#include "collect.h"
#include "tokens.h"
#include "print.h"
#include "atoms.h"

static void WalkForm(form *, scont);
static void WalkForms(list, scont);

static void WalkBlock(form *, scont);
static void WalkCase(form *, scont);
static void WalkContinue(form *, scont);
static void WalkExpression(form *, scont);
static void WalkFlet(form *, scont);
static void WalkFuncall(form *, scont);
static void WalkFunction(form *, scont);
static void WalkIf(form *, scont);
static void WalkLabels(form *, scont);
static void WalkLet(form *, scont);
static void WalkStack_Allocate(form *, scont);
static void WalkProgn(form *, scont);
static void WalkReturn_From(form *, scont);
static void WalkSetq(form *, scont);
static void WalkStatement(form *, scont);
static void WalkTest(form *, scont);
static void WalkThe_Continuation(form *, scont);
static void WalkVariable(form *, scont);
static void WalkApplication(form *, scont);

static void ReferencedVar(variable *);
static void CalledFunction(function *, scont);
static variable *NewVariable(struct lex_variable *);
static function *NewFunction(struct lex_variable *, struct definition *);

static void WalkDefinitions(list);
static void WalkDefinition(struct definition *);
static void CreateNewFunctions(list);
static void CreateNewFunction(struct definition *);
static void AddEnvFunctions(list);
static void AddEnvFunction(struct function_ref *);
static void AddEnvVariable(struct variable_ref *);


/**********************************************************************
 *
 *                       Variables exportees
 *
 **********************************************************************
 *
 * 	AllFunctions	la liste de toutes les fonctions rencontrees.
 *	AllVariables	la liste de toutes les variables rencontrees.
 * 	AllBlocks	la liste des blocs lexicaux rencontres.
 *
 **********************************************************************/

list AllFunctions;
list AllVariables;
list AllBlocks;


/**********************************************************************
 *
 *                    Arpentage d'une forme
 *
 **********************************************************************
 *
 * Fonctions :
 *
 *	CollectInformations
 *			la fonction d'entree.
 *	Walk		arpentage d'une forme quelconque.
 *	WalkXXX		fonction d'arpentage de la forme XXX.
 *
 *
 * Les fonctions d'arpentage se passent les parametres suivants :
 *
 *	f		la forme a arpenter.
 *	c		une continuation symbolique.
 *
 * Variables :
 *
 *	ContNumber	un entier pour generer des continuations
 *			symboliques.
 *
 * Les variables suivantes sont gerees de facon "dynamique", grace aux
 * macros SAVE et REST :
 *
 *	VarEnv		l'environnement des variables (une p-liste).
 *	FunEnv		l'environnement des fonctions (une p-liste).
 *	BlockEnv	l'environnement des blocs lexicaux (une p-liste).
 *	Level		le niveau lexical courant, incremente a chaque
 *			flet, labels, let, block. Il n'est en fait pas
 * 			utile de l'incrementer au let, mais ca ne fait
 * 			aucun mal...
 *	CurrentFun	la fonction locale,  ou globale, ou la pseudo-
 * 			fonction associee a un bloc lexical, dont on
 * 			arpente le corps.
 *	CurrentGlobal	la fonction globale dont on arpente le corps,
 *			ou NULL si on arpente une fonction locale. Cas
 * 			special pour les fonctions de blocs lexicaux.
 *
 **********************************************************************/


/*
 * Pour la generation de continuations.
 */

scont ContNumber = DUMSCONT;

#define	NEWSCONT	(ContNumber=(char *)ContNumber+1)

/*
 * Variables dynamiques
 */

static list VarEnv;
static list FunEnv;
static list BlockEnv;
static int Level;
static function *CurrentFun;
static function *CurrentGlobal;


/*
 * Macros pour la gestion de variables dynamiques.
 */

#define	SAVE(type,var)		{ type _tmp = var;
#define REST(var)		  var = _tmp; }


/*
 * Fonction d'entree, appelee sur la fonction globale a compiler.
 */

void CollectInformations(struct definition *def)
{
     struct function_ref *fref = def->fref;


     DEBUG
       {
	    fprintf(Out, "Collecting informations\n");
       }
     END

				/* Initialisation des variables dynamiques. */
     VarEnv = NIL;
     FunEnv = NIL;
     BlockEnv = NIL;
     Level = 0;
     CurrentFun = NULL;

				/* Initialisation des variables globales. */
     AllFunctions = NIL;
     AllVariables = NIL;
     AllBlocks = NIL;
     
     SAVE(list, FunEnv);	/* Pas vraiment utile, mais plus propre. */
     SAVE(int, Level);		/* Idem. */
     
     Level = Level+1;

     CreateNewFunction(def);
     fref->fun->status = ROOT;
     AddEnvFunction(fref);

     CurrentGlobal = fref->fun;
     CurrentGlobal->contvar = NewVariable(fref->lexvar);
     
     WalkDefinition(def);

     if (!CurrentGlobal->continued) CurrentGlobal->contvar = NULL;

     REST(Level);
     REST(FunEnv);

     DEBUG
       {
	    PrintFunctions(AllFunctions);
	    PrintVariables(AllVariables);
       }
     END
}


/*
 * Arpentage d'une forme.
 */

static void WalkForm(form *f, scont c)
{
     switch (f->tag)
       {
       case BLOCK:
	    WalkBlock(f, c);
	    break;
	    
       case CASE:
	    WalkCase(f, c);
	    break;
	    
       case CONTINUE:
	    WalkContinue(f, c);
	    break;
	    
       case EXPRESSION:
	    WalkExpression(f, c);
	    break;
	    
       case FLET:
	    WalkFlet(f, c);
	    break;
	    
       case FUNCALL:
	    WalkFuncall(f, c);
	    break;
	    
       case FUNCTION:
	    WalkFunction(f, c);
	    break;
	    
       case IF:
	    WalkIf(f, c);
	    break;
	    
       case LABELS:
	    WalkLabels(f, c);
	    break;
	    
       case LET:
	    WalkLet(f, c);
	    break;
	    
       case STACK_ALLOCATE:
	    WalkStack_Allocate(f, c);
	    break;
	    
       case PROGN:
	    WalkProgn(f, c);
	    break;

       case RETURN_FROM:
	    WalkReturn_From(f, c);
	    break;
	    
       case SETQ:
	    WalkSetq(f, c);
	    break;
	    
       case STATEMENT:
	    WalkStatement(f, c);
	    break;
	    
       case TEST:
	    WalkTest(f, c);
	    break;
	    
       case THE_CONTINUATION:
	    WalkThe_Continuation(f, c);
	    break;
	    
       case VARIABLE:
	    WalkVariable(f, c);
	    break;
	    
       case APPLICATION:
	    WalkApplication(f, c);
	    break;
	    
       default:
	   fprintf(Err, "%s panic: unknown tag in WalkForm: %d\n", Me, f->tag);
	   exit(3);
       }
}


/*
 * Arpentage d'une liste de formes.
 */

static void WalkForms(list lf, scont c)
{
     list l;
     
     if (lf == NIL) return;
     
     for (l = lf; CDR(l) != NIL; l = CDR(l))
	  WalkForm(CAR(l), NEWSCONT);
     
     WalkForm(CAR(l), c);
}


/* 
 * Arpentage des differentes formes.
 */



/* Le traitement du (:block ...) utilise la similitude qu'il y entre
 * la determination de la maniere dont on compile une fonction locale
 * et les appels a celle-ci, et la determination de la maniere dont
 * on compile un bloc lexical et les echappements de celui-ci.
 * Un block lexical est identifie a une fonction locale possedant une
 * variable locale destinee a contenir la continuation d'appel de cette
 * fonction. Un return-from est identifie a un appel (bidon) a cette
 * fonction et a une reference a la variable.
 */

static void WalkBlock(form *f, scont c)
{
     struct block_ref *bref = f->node.block_i.bref;
     form *body = f->node.block_i.body;
     struct lex_variable *lexvar = bref->lexvar;

     block *blk = NEW(block);
     struct definition *def = NEW(struct definition);
     struct function_ref *fref = NEW(struct function_ref);
     function *fun;

     AllBlocks = cons(blk, AllBlocks);
     
     SAVE(int, Level);
     Level = Level+1;

				/* Construction de la pseudo-fonction. */

     fref->lexvar = lexvar;	/* Pseudo reference a une fonction... */
     fref->origin = bref->origin;
     
     def->origin = f->origin;	/* Pseudo definition... */
     def->fref = fref;
     def->vrefs = NIL;
     def->body = body;

     CreateNewFunction(def);	/* Pseudo fonction... */

				/* Fin de la construction. */

     Level = Level+1;		/* Pour la variable. */

     fun = fref->fun;
     fun->contvar = NewVariable(lexvar);
     blk->fun = fun;
     blk->cont = (cont) 0;
     blk->label = 0;
     blk->used = FALSE;
     bref->block = blk;

     SAVE(list, BlockEnv);
     BlockEnv = addenv(lexvar, blk, BlockEnv);

     WalkDefinition(def);	/* NB: CurrentGlobal pas touchee. */

     REST(BlockEnv);
     REST(Level);


				/* Pour eliminer les blocs inutiles. */
				/* Ne sert en fait a rien si -O >= 3 */

     if (blk->used) CalledFunction(fun, c);
     else
       {
	    list l;
	    
	    for (l = fun->called; l != NIL; l = CDR(l))
	      {
		   struct edge *e = CAR(l);
		   
		   if (e->scont == TERMSCONT) e->scont = c;
		   CurrentFun->called = cons(CAR(l), CurrentFun->called);
	      }
	    
		 
	    for (l = fun->freevars; l != NIL; l = CDR(l))
	      {
		   variable *var = CAR(l);
		   list f = CurrentFun->freevars;
		   
		   if ((var->level < CurrentFun->level) && (!memberp(var, f)))
			CurrentFun->freevars = cons(var, f);
	      }
	    
	    fun->called = NIL;
	    fun->freevars = NIL;
       }
}

static void WalkCase(form *f, scont c)
{
     form *val = f->node.case_i.val;
     list clauses = f->node.case_i.clauses;
     form *def = f->node.case_i.def;
     list l;

     WalkForm(val, NEWSCONT);

     for (l = clauses; l != NIL; l = CDR(l))
	  WalkForm(((struct clause *)CAR(l))->body, c);

     WalkForm(def, c);
}

static void WalkContinue(form *f, scont c)
{
     form *co = f->node.continue_i.cont;
     form *val = f->node.continue_i.val;

     WalkForm(co, NEWSCONT);
     WalkForm(val, NEWSCONT);
}

static void WalkExpression(form *f, scont c)
{
     list args = f->node.expression_i.args;

     WalkForms(args, NEWSCONT);
}

static void WalkFlet(form *f, scont c)
{
     list defs = f->node.flet_i.defs;
     form *body = f->node.flet_i.body;

     SAVE(function *, CurrentGlobal);
     SAVE(int, Level);

     CurrentGlobal = NULL;
     Level = Level+1;

     CreateNewFunctions(defs);
     WalkDefinitions(defs);

     REST(Level);
     REST(CurrentGlobal);

     SAVE(list, FunEnv);
     
     AddEnvFunctions(defs);
     WalkForm(body, c);
     
     REST(FunEnv);
}

static void WalkFuncall(form *f, scont c)
{
     form *fun = f->node.funcall_i.fun;
     list args = f->node.funcall_i.args;

     WalkForm(fun, NEWSCONT);
     WalkForms(args, NEWSCONT);
}

static void WalkFunction(form *f, scont c)
{
     struct function_ref *fref = f->node.function_i;
     function *fun;

     if (((fun = readenv(fref->lexvar, FunEnv)) != NULL)
	 &&(fun->status != ROOT))
       {
	    PrintError("functional value of local function", 
		       f->origin.lineno, f->origin.vlineno, 
		       f->origin.vname);
	    longjmp(DefunExit, TRUE);
       }

     fref->fun = NULL;
}

static void WalkIf(form *f, scont c)
{
     form *test = f->node.if_i.test;
     form *iftrue = f->node.if_i.iftrue;
     form *iffalse = f->node.if_i.iffalse;

     WalkForm(test, NEWSCONT);
     WalkForm(iftrue, c);
     WalkForm(iffalse, c);
}

static void WalkLabels(form *f, scont c)
{
     list defs = f->node.labels_i.defs;
     form *body = f->node.labels_i.body;

     SAVE(list, FunEnv);
     SAVE(function *, CurrentGlobal);
     SAVE(int, Level);

     CurrentGlobal = NULL;
     Level = Level+1;
     
     CreateNewFunctions(defs);
     AddEnvFunctions(defs);
     WalkDefinitions(defs);

     REST(Level);
     REST(CurrentGlobal);
     
     WalkForm(body, c);

     REST(FunEnv);
}

static void WalkLet(form *f, scont c)
{
     list binds = f->node.let_i.binds;
     form *body = f->node.let_i.body;
     list l;
     
     for (l = binds; l != NIL; l = CDR(l))
	  WalkForm(((struct binding *)CAR(l))->val, NEWSCONT);

     SAVE(int, Level);
     SAVE(list, VarEnv);
     
     Level = Level+1;
     
     for (l = binds; l != NIL; l = CDR(l))
       {
	    struct binding *bind = CAR(l);
	    AddEnvVariable(bind->vref);
       }
     
     WalkForm(body, c);
     
     REST(VarEnv);
     REST(Level);
}

static void WalkStack_Allocate(form *f, scont c)
{
     list sizes = f->node.stack_allocate_i.sizes;
     form *body = f->node.stack_allocate_i.body;
     list l;
     
     SAVE(int, Level);
     SAVE(list, VarEnv);
     
     Level = Level+1;
     
     for (l = sizes; l != NIL; l = CDR(l))
       {
	    struct sizing *size = CAR(l);
	    AddEnvVariable(size->vref);
       }
     
     WalkForm(body, c);
     
     REST(VarEnv);
     REST(Level);
}

static void WalkProgn(form *f, scont c)
{
     list body = f->node.progn_i.body;

     WalkForms(body, c);
}

static void WalkReturn_From(form *f, scont c)
{
     struct block_ref *bref = f->node.return_from_i.bref;
     form *val = f->node.return_from_i.val;
     block *blk;
     
     blk = readenv(bref->lexvar, BlockEnv);
     if (blk == NULL)
       {
	    PrintError("return from an unknown block", 
		       f->origin.lineno, f->origin.vlineno, 
		       f->origin.vname);
	    longjmp(DefunExit, TRUE);
       }

     bref->block = blk;
     
     blk->used = TRUE;
     CalledFunction(blk->fun, DUMSCONT);
     ReferencedVar(blk->fun->contvar);
     
     WalkForm(val, NEWSCONT);
}

static void WalkSetq(form *f, scont c)
{
     struct variable_ref *vref = f->node.setq_i.vref;
     form *val = f->node.setq_i.val;
     variable *var;
     
     var = readenv(vref->lexvar, VarEnv);
     if (var != NULL)
       {
	    vref->var = var;
	    ReferencedVar(var);
	    var->mutated = TRUE;
       }

     WalkForm(val, NEWSCONT);
}

static void WalkStatement(form *f, scont c)
{
     list args = f->node.statement_i.args;

     WalkForms(args, NEWSCONT);
}

static void WalkTest(form *f, scont c)
{
     list args = f->node.test_i.args;

     WalkForms(args, NEWSCONT);
}

static void WalkThe_Continuation(form *f, scont c)
{
     if (CurrentGlobal == NULL)
       {
	    PrintError("continuation of a non global function",
		       f->origin.lineno, f->origin.vlineno, 
		       f->origin.vname);
	    longjmp(DefunExit, TRUE);
       }
     
     CurrentGlobal->continued = TRUE;
     f->node.the_continuation_i.contvar = CurrentGlobal->contvar;
     ReferencedVar(CurrentGlobal->contvar);
}

static void WalkVariable(form *f, scont c)
{
     struct variable_ref *vref = f->node.variable_i;
     variable *var;

     var = readenv(vref->lexvar, VarEnv);
     if (var != NULL)		/* Sinon, variable presumee globale */
       {
	    vref->var = var;
	    ReferencedVar(var);
       }
}

static void WalkApplication(form *f, scont c)
{
     struct function_ref *fref = f->node.application_i.fref;
     list args = f->node.application_i.args;
     function *fun;
     
     fun = readenv(fref->lexvar, FunEnv);
     if (fun != NULL)
       {
	    fref->fun = fun;
	    CalledFunction(fun, c);
       };

     WalkForms(args, NEWSCONT);
}




/**********************************************************************
 *
 *                    Variables et fonctions
 *
 **********************************************************************
 *
 * Fonctions :
 *
 *	ReferencedVar	annonce qu'une variable est referencee en
 * 			lecture ou en ecriture.
 *	CalledFunction	annonce qu'une fonction est appelee.
 *	NewVariable	cree une nouvelle variable en initialisant
 *			ses champs.
 *	NewFunction	idem, pour une fonction.
 *
 **********************************************************************/


/*
 * Annonce qu'une variable est referencee.
 */

static void ReferencedVar(variable *var)
{
     if (var->level < CurrentFun->level)
       {
	    list l = CurrentFun->freevars;
     
	    if(!memberp(var, l))
		 CurrentFun->freevars = cons(var, l);
       }
}

/*
 * Annonce qu'une fonction est appelee.
 */

static void CalledFunction(function *fun, scont c)
{
     list l = CurrentFun->called;
     struct edge *e = NEW(struct edge);
     
     e->scont = c;
     e->fun = fun;
     CurrentFun->called = cons(e, l);

     fun->callsno = fun->callsno+1;
}

/*
 * Creation d'une nouvelle variable.
 */

static variable *NewVariable(struct lex_variable *lex)
{
     variable *var = NEW(variable);
     
     var->mutated = FALSE;
     var->reference = FALSE;
     var->level = Level;
     var->lexvar = lex;
     var->suffix = (lex->count)++;
     
     AllVariables = cons(var, AllVariables);

     return var;
}


/*
 * Creation d'une nouvelle fonctions.
 */

static function *NewFunction(struct lex_variable *lex, struct definition *def)
{
     function *fun = NEW(function);
     
     fun->level = Level;
     fun->lexvar = lex;
     fun->suffix = (lex->count)++;
     fun->def = def;
     
     fun->freevars = NIL;
     fun->called = NIL;
     fun->sconts = NIL;
     fun->called_t = NIL;
     fun->called_n = NIL;
     fun->status = UNKNOWN;
     fun->continued = FALSE;
     fun->callsno = 0;
     fun->labels = NIL;
     fun->labelof = NULL;
     fun->contvar = NULL;
     fun->generated = FALSE;
     fun->visitor = NULL;
     fun->visited = FALSE;
     
     AllFunctions = cons(fun, AllFunctions);

     return fun;
}



/**********************************************************************
 * 
 *                         Utilitaires
 *
 **********************************************************************
 *
 *	WalkDefinition	arpente une definition de fonction.
 *	CreateNewFunction
 *			cree une nouvelle fonction et l'affecte a une
 * 			reference.
 *	AddEnvFunction	cree une nouvelle fonction et l'ajoute par
 * 			effet de bord a l'environnement des fonctions.
 *	AddEnvVariable	idem, pour une variable.
 *
 **********************************************************************/

static void WalkDefinitions(list defs)
{
     list l;
     
     for (l = defs; l != NIL; l = CDR(l))
	  WalkDefinition(CAR(l));
}

static void WalkDefinition(struct definition *def)
{
     list vrefs = def->vrefs;
     form *body = def->body;
     list l;
     
     SAVE(list, VarEnv);
     SAVE(function *, CurrentFun);
     SAVE(int, Level);
     
     CurrentFun = def->fref->fun;
     Level = Level+1;		/* Pour les arguments */

     for (l = vrefs; l != NIL; l = CDR(l))
       {
	    struct variable_ref *vref = CAR(l);
	    AddEnvVariable(vref);
       }
     
     WalkForm(body, TERMSCONT);

     REST(Level);
     REST(CurrentFun);
     REST(VarEnv);
}

static void CreateNewFunctions(list defs)
{
     list l;
     
     for (l = defs; l != NIL; l = CDR(l))
	  CreateNewFunction(CAR(l));
}

static void CreateNewFunction(struct definition *def)
{
     struct function_ref *fref = def->fref;
     struct lex_variable *lexvar = fref->lexvar;
     function *fun;
     
     fun = NewFunction(lexvar, def);
     fref->fun = fun;
}     

static void AddEnvFunctions(list defs)
{
     list l;
     
     for (l = defs; l != NIL; l = CDR(l))
	  AddEnvFunction(((struct definition *)CAR(l))->fref);
}

static void AddEnvFunction(struct function_ref *fref)
{
     struct lex_variable *lexvar = fref->lexvar;
     function *fun = fref->fun;

     FunEnv = addenv(lexvar, fun, FunEnv);
}

static void AddEnvVariable(struct variable_ref *vref)
{
     struct lex_variable *lexvar = vref->lexvar;
     variable *var;

     var = NewVariable(lexvar);
     vref->var = var;
     VarEnv = addenv(lexvar, var, VarEnv);
}
