%{
/***************************************************************************
  Copyright (C) Nitsan Seniak 1989, 1990

  This file is part of the K2 compiler.
  Permission to copy this software, in whole or in part, to use this
  software for any lawful noncommercial purpose, and to redistribute
  this software is granted subject to the restriction that all copies
  made of this software must include this copyright notice in full.
  The author(s) makes no warranties or representations of any kind, either
  express or implied, including but not limited to implied warranties
  of merchantability or fitness for any particular purpose.
  All materials developed as a consequence of the use of this
  software shall duly acknowledge such use, in accordance with the usual
  standards of acknowledging credit in research.
 ***************************************************************************/ 

/****************************************************************************
 *                                                                          *
 *                            ANALYSEUR LEXICAL                             *
 *                                                                          *
 ****************************************************************************
 *
 * Ce module contient des directives LEX et des actions d'internement
 * des les lexemes du langage. Ceux-ci se repartissent en lexemes non
 * internes :
 *
 *	- Les parentheses
 *	- Les mots clefs (:if, :otherwise ...)
 *
 * en en lexemes internes :
 *
 *	- Les variables (*, foo ...)
 *	- Les textes C ("$+$" ...), qui ne sont PAS des constantes.
 *
 * Une variable (ie, un nom de variable ou de fonction) est internee dans
 * une table de hash et est representee par l'adresse une paire (chaine
 * de caracteres, entier) unique ; l'entier est initialement 0 et sert
 * a l'alpha-conversion.
 *
 * Les textes C sont internes sous forme de simples chaines de caracteres.
 *
 *
 * En plus de l'internement des lexemes, l'analyseur lexical gere les
 * directives #line (avec la syntaxe ANSI), avec des affectations
 * sur des variables globales ; il contient de plus les fonctions
 * d'affichage d'erreurs avec localisation (pourquoi dans ce module ? Et
 * pourquoi pas ?).
 *
 *
 * Ce fichier est compatible lex/flex (ce qui a demande certains efforts !).
 *
 *
 * Important : tous les objets crees sont alloues dans le tas temporaire.
 * 
 ****************************************************************************/



/*****************************************************************************
 *
 *                              Declarations
 *
 *****************************************************************************
 *
 * D			un Digit.
 * L			une Lettre, constituante de symbole.
 * B			un separateur marquant la fin d'un lexeme.
 *
 *****************************************************************************/
%}



D			[0-9]
L			[a-zA-Z0-9!@$%^&*_+|=[{}'`~?/.,<>\]\\-]
B			[ \n\t\v\f)]


%{

#include <stdlib.h>
#include <setjmp.h>
#include <ctype.h>
#include <string.h>
#include "util.h"
#include "objects.h"
#include "yystype.h"
#include "ident.h"
#include "atoms.h"
#include "forms.h"
#include "tokens.h"
#include "k2.h"


char *True_lit = "KTRUE";
char *False_lit = "KFALSE";

static void count(void);

/*
 * Gestion des mots-clefs
 */

struct hash_table *KeywordsTable;

static struct {
     char *name;
     int token;
} keywords[] =
{
  {":defun", DEFUN}, {":defvar", DEFVAR}, {":declaration", DECLARATION},
  {":declare", DECLARE}, {":static", STATIC}, {":extern", EXTERN},
  {":inline", INLINE}, {":block", BLOCK}, {":case", CASE},
  {":continue", CONTINUE}, {":expression", EXPRESSION}, {":flet", FLET},
  {":funcall", FUNCALL}, {":function", FUNCTION}, {":if", IF},
  {":labels", LABELS}, {":let", LET}, {":otherwise", OTHERWISE},
  {":progn", PROGN}, {":return-from", RETURN_FROM}, {":setq", SETQ},
  {":stack-allocate", STACK_ALLOCATE}, {":statement", STATEMENT}, 
  {":test", TEST}, {":the-continuation", THE_CONTINUATION}
};

#define KEYWORDS (sizeof(keywords)/sizeof(*keywords))

static int Keyword(char[]);
static int Text(char[], int);
static int Var(char[]);
static int Num(char[]);

static void ReadSharpLine(boolean);

static boolean Wrongp = FALSE;

				/* Pour lex */
#ifdef YYLMAX
#undef YYLMAX
#define YYLMAX		10000
#endif

#undef YY_READ_BUF_SIZE		/* Pour flex */
#define YY_READ_BUF_SIZE 16384
#undef YY_BUF_SIZE
#define YY_BUF_SIZE (YY_READ_BUF_SIZE * 2) /* size of default input buffer */

#ifdef yywrap	
#undef yywrap
#define yywrap()	(Eof = TRUE, 1)
#endif

%}



%%

%{
/****************************************************************************
 *
 *                                Regles
 *
 ****************************************************************************
 *
 * Les particularites de l'analyseur lexical sont les suivantes :
 *
 * - Case sensitive.
 *
 * - Les mots-clefs doivent commencent tous par ":", ce qui permet a
 *   une couche superieure d'expliciter une forme parenthesee comme
 *   un appel de fonction ou une forme speciale.
 *
 * - Tout lexeme different de "(" et de ")" doit etre suivi par un separateur
 *   (B), ce qui interdit par exemple d'ecrire "foo(x)". Ce n'est pas une
 *   bug mais un choix.
 *
 * - Une variable ne peut plus commencer par un digit.
 *
 ****************************************************************************/
%}


;.*$			;

"("			{ count(); return '('; }
")"			{ count(); return ')'; }

":"{L}+/{B}		{ count(); return Keyword(yytext); }

"#line"[ \t]*{D}+[ \t]*$	         { count(); ReadSharpLine(FALSE); }
"#line"[ \t]*{D}+[ \t]*\"[^"]+\"[ \t]*$  { count(); ReadSharpLine(TRUE); }

\"([^"\\]|\\.)*\"/{B}	{ count(); return Text(yytext, yyleng); }

{D}+/{B}                { count(); return Num(yytext); }

({L}+:)?{L}+/{B} 	{ count(); return Var(yytext); }

[ \n\t\v\f]		{ count(); }

.			{ count(); Wrongp = TRUE; return WRONG; }



%%

/*****************************************************************************
 *
 *                          Gestion des erreurs
 *
 *****************************************************************************
 *
 * L'utilisation d'une fonction "cont" est largement inspiree d'une
 * grammaire C parue dans les News.
 *
 * Variables representant la "position" courante du lexer :
 *
 *	column		colonne courante de l'"ecran".
 *	input_str	la ligne en cours de lexture.
 *	input_col	compteur de caracteres lus.
 *	VirtualInName	nom du dernier fichier specifie par #line,  ou NULL.
 *	VirtualLineno	numero de ligne tenant compte du dernier #line, ou -1.
 *	Lineno		numero reel de ligne de l'entree.
 *	Eof		indique si on a rencontre, ou pas, la fin de fichier.
 *
 * Autre variable :
 *
 *	Wrongp		un flag leve lors d'une erreur d'analyse lexicale,
 * 			et qui sert a l'edition d'un message d'erreur
 * 			approprie.
 *
 * Fonctions :
 *
 *	count		met a jour column, input_str et input_col en
 *			fonction du token lu.
 *	ReadSharpLine	initialise les variables VirtualInName et VirtualLineno
 * 			en fonction d'une directive #line lue.
 *	yyerror		fonction appelee par lex et yacc pour afficher
 *			un message d'erreur.
 *	PrintError	affiche un message d'erreur et une position.
 *
 *****************************************************************************/


char *VirtualInName = NULL;
int VirtualLineno = -1;
int Lineno = 1;
boolean Eof = FALSE;

#define LINE_LENGTH 79		/* Taille d'une ligne ecran, simplement */
				/* pour l'affichage. */


int column = 0;
char input_str[LINE_LENGTH];
int input_col = 0;

static void count(void)
{
     int i;
     
     for (i = 0; yytext[i] != '\0'; i++)
       {
	    
	    if ((input_col == LINE_LENGTH) || (column >= LINE_LENGTH))
	      {
		   strcpy(input_str, "... ");
		   input_col = 4;
		   column = 4;
	      };
	    input_str[input_col++]=yytext[i];
	    
	    if (yytext[i] == '\n')
	      {
		   column = 0;
		   input_col = 0;
		   Lineno++;
		   if (VirtualLineno >= 0) VirtualLineno++;
	      }
	    else if (yytext[i] == '\t')
		 column += 8 - (column % 8);
	    else
		 column++;
	    
       }
}

static void ReadSharpLine(boolean filep)
{
     char *s = yytext;
     char *q;
     unsigned l;

     while (!isdigit(*s)) s++;
     VirtualLineno = (int)strtol(s, &s, 10)-1;
     if (!filep) return;
     
     while (*s != '"') s++;
     q = (++s);
     
     while (*s != '"') s++;
     *s = '\0';
     
     l = strlen(q);
     VirtualInName = strncpy(allocp(l+1), q, l+1);
}

void yyerror(char *s)
{

     Error = TRUE;
     
     fflush(Out);

     if (strcmp(s, "syntax error") == 0) /* Beurk (mais oblige) */
	  if (Wrongp)
	    {
		 s = "illegal token";
		 Wrongp = FALSE;
	    }

     PrintError(s, Lineno, VirtualLineno, VirtualInName);

     if (Localize)
       {
	    int here = input_col;
	    int real_col = column;
	    int inchar;

	    if (!Eof)		/* Pray God, it works ! */
		 do
		   {
			inchar = input();
			if (Eof || (inchar == '\0') || (inchar == EOF)) break;
			input_str[input_col++]=inchar;

			if (inchar == '\t') real_col += 8-(real_col%8);
			else real_col++;
		   }	
	         while ((inchar != '\n')&&(real_col < LINE_LENGTH));
	    
	    fprintf(Err, "%.*s", input_col, input_str);
	    if (input_str[input_col-1] != '\n')
		 fprintf(Err, "\n");
	    fprintf(Err, "%*s\n", column, "^");
	    
	    while (input_col!=here)
		 unput(input_str[--input_col]);
       }     
     
}

void PrintError(char *msg, int lineno, int vlineno, char *vname)
{
     fflush(Out);
	   
     if (vname != NULL)
	  fprintf(Err, "\"%s\", line %d", vname, vlineno);
     else 
       {
	    if (In != stdin) fprintf(Err, "\"%s\"", InName);
	    else fprintf(Err, "stdin");
	    
	    fprintf(Err, ", line %d", ((vlineno>=0)?vlineno:lineno));
       };
     
     if ((vlineno >=0) && Localize)
       {
	    fprintf(Err, " (actually ");

	    if (In != stdin)
		 fprintf(Err, "\"%s\"", InName);
	    else fprintf(Err, "stdin");
	    
	    fprintf(Err, " line %d)", lineno);
       }

     fprintf(Err, ": %s\n", msg);
}

#ifndef yywrap			/* Compatibilite lex/flex */
int yywrap(void)
{
     Eof = TRUE;
     return 1;
}
#endif



/***********************************************************************
 *
 *	            Internement des autres lexemes
 *
 ***********************************************************************
 *
 * Fonctions :
 *
 *	Keyword		interne un mot cle
 *	Text		interne une chaine de texte C.
 * 	Var		interne une variable.
 *      Num             interne un entier.
 *
 ***********************************************************************/


static int Keyword(char str[])
{
     struct hash_cell *c = find(KeywordsTable, str);
     struct keyword *k;
     
     if (c == NULL)		/* Mot-clef non interne */
       {
	    yylval = scopy(str);
	    return UNIN_KEYWORD;
       }
     
     k = c->contents;

     if (k->text == NULL)	/* Mot-clef predefini */
	  return k->token;
     else
       {
	    yylval = k;		/* Abreviation */
	    return KEYWORD;
       }
}


/*
 * Toute occurence de \x, ou x est un caractere quelconque, est
 * convertie en x,  sauf \$ qui est converti en 1 (vilain kludge)
 */

static int Text(char s[], int t)
{
     char *r = sncopy(s+1, t-2);
     char *a = r, *b = r;

     r[t-2] = '\0';

     while (*b != '\0')
       {
	    if (*b == '\\')
	      {
		   b++;
		   *(a++) = *(b++);
	      }
	    else if (*b == '$') { *(a++) = '\01'; b++; }
	    else *(a++) = *(b++);
       }
     *a = '\0';

     yylval = r;
     return TEXT;
}

static int Num(char s[])
{
     yylval = s;

     return NUM;
}

static int Var(char str[])
{
     struct hash_cell *s;

     if (SymbolsTable != NULL)	/* Sinon, c'est qu'on est en dehors d'une */
				/* forme de toplevel */
       {
	    s = find_create(SymbolsTable, str);
	    if (s->contents == NULL)
	      {
		   struct lex_variable *l = NEW(struct lex_variable);

		   l->name = s->str; /* Car deja copie dans le tas */
		   l->cname = cname(str);
		   l->count = 0; /* was 1 */

		   s->contents = l;
	      };
     
	    yylval = s->contents;
       }
     
     return VAR;
}

void InitLex(void)
{
     int i;
     struct hash_cell *c;

     KeywordsTable = create_hash_table(H_GLOBAL, 211);
     
     for (i = 0; i<KEYWORDS; i++)
       {
	    struct keyword *k = PNEW(struct keyword);

				/* Le mot-clef est recopie dans le */
				/* tas, mais tant pis. */
	    c = find_create(KeywordsTable, keywords[i].name);
	    c->contents = k;
	    k->token = keywords[i].token;
	    k->text = NULL;
       }
}
