/* $Header: /Nfs/radyr/usr11/rc136/Projet/GC/RCS/public.h,v 1.1 1992/09/03 21:14:21 rc136 Exp rc136 $ */
/* Auteur: Vincent Delacour (delacour@poly.polytechnique.fr) */

#include "config.h"

typedef void **val_t;
typedef ulint nbytes_t;


#define OneWord		((nwords_t)1)
#define OnePage 	((1 << PagePower)*OneWord)
#define OneBitMap	(OnePage/sizeof(val_t))

#define WordSize        (sizeof(val_t))
#define PageSize        ((nbytes_t)(OnePage*WordSize))

/**********************************************************************
 * Types publics
 **********************************************************************
 * Ces types forment,  avec les quelques macros d'acces qui leur sont
 * associees,  le modele memoire.
 *
 * En signant les types 'nombre d'octets' et 'nombre de mots' on se permet
 * une ou deux verifications dynamiques...
 **********************************************************************/

                                   
                      /* - Types scalaires - */

typedef slint  nwords_t;                /* nombre de mots             */

typedef unsigned int  npage_t;                 /* numero ou nombre de pages  */
#define MAXPAGES   USHRT_MAX            /* cf le type npage_t         */

typedef unsigned int  nregion_t;               /* numero de region           */

                                   
                      /* - types construits - */

/**********************************************************************
 * Les structures de donnees du gestionnaire memoire. Quand seulement
 * certains champs sont 'publics', ces champs publics sont marques X
 * (X comme eXport).
 **********************************************************************
 * pagetag_t: tags de pages
 *
 * Les tags de page regroupent en un seul mot memoire les informations
 * les plus utiles a connaitre sur la page,  assurant de ce fait une
 * bonne superficialite d'acces de ces informations: zone,  mode et
 * region. Ces trois attributs sont souvent utilises ensemble.
 *
 * pagedesc_t: descripteurs de pages
 *
 * region_t: regions
 *
 **********************************************************************/ 


typedef struct                 /* - tags de pages - */
{
     unsigned  char zone;      /* GC                          */
     unsigned  char mode;      /* property                    */
     nregion_t nregion;        /*Xnumero de region            */
} pagetag_t;


typedef struct                  /* - descripteurs de pages - */
{
     pagetag_t tag;             /*Xsee above                           */
     npage_t   bsize;           /* bloc size                           */
     npage_t   bhead;           /* bloc head                           */
     npage_t   next;            /* next bloc or page                   */
     npage_t   prev;            /* previous                            */
     void     *p;               /* scavenging pointer, bit-map...      */
     val_t     top;             /* first address unused (possibly just */
                                /* after the page limit).              */
} pagedesc_t;


/**********************************************************************
 * Regions : les champs marques @ sont initialises statiquement
 * par la chaine de compilation K2+. Il ne faut pas les reordonner
 * sans faire de modification dans la chaine de compilation k2+.
 **********************************************************************/


typedef struct                  /*  - Regions - */
{
     pagetag_t  pagetag;        /* @Initial page tag for region        */
     pagetag_t  bigtag;         /* @page tag for big objects           */
     char*      name;           /* @                                   */
     nwords_t   cost;           /* @cost of 1 page for small objets    */
     nwords_t   waste;          /* @residu */
     nwords_t   size;           /*X@size of objects (homogeneous reg.) */
     nwords_t   minimum_size;   /*X@                                   */
     nwords_t   maximum_size;   /*X@                                   */
     val_t      ap;             /*X allocation pointer                 */
     val_t      alimit;         /*X allocation limit                   */
     val_t      scp;            /*  scavenging pointer                 */
     val_t      popbreak;       /*  (gc internals) stack regions       */
     pagetag_t  copytag;	/*  tag for copy pages                 */
     nregion_t  next;           /*  next "copying" region  */
     npage_t    apg;            /*  allocation page  */
     nwords_t   remainder;      /*  residu (cf compte memoire libre)    */

     val_t 	freehead; 	/* liste objets libres */
     val_t 	freetail;	   /* queue liste libre */
} region_t;



/**********************************************************************
 * Variables externes
 **********************************************************************
 * Ces variables implementent l'etat "public" du gestionnaire memoire.
 * Elles sont accedees par des macros exportees par le gestionnaire
 * memoire: macros d'allocation,  tests de types,  etc.
 **********************************************************************/


/* 
 * Tableau des pages
 */

extern npage_t     DescBloc;            /* bloc des descripteurs de pages */
extern npage_t     DescSize;            /* taille de ce bloc              */
extern pagedesc_t *PageDescs;           /* adresse du tableau des pages   */
extern int         DescCapacity;        /* taille en descripteurs         */

/* 
 * POOL
 */

extern npage_t PoolFirst;               /* point d'entree sur le POOL */
extern npage_t PoolSize;                /* nombre de pages dans le pool */

/* 
 * Regions
 */

extern nregion_t NRegions;              /* nombre de regions */
extern region_t **PRegions;             /* tableau de pointeurs sur les */
                                        /* regions  */
extern region_t StaticRegions[];        /* Regions statiques */

                       /* Acces aux regions */
#define Region(nreg)		(PRegions[nreg])

#define RegionFreeHead(nreg)	(PRegions[(nreg)]->freehead)
#define RegionFreeTail(nreg)	(PRegions[(nreg)]->freetail)

#define RegionSize(nreg)        (PRegions[(nreg)]->size)
#define RegionAP(nreg)          (PRegions[(nreg)]->ap)
#define RegionAPG(nreg)         (PRegions[(nreg)]->apg)
#define RegionLimit(nreg)       (PRegions[(nreg)]->alimit)
#define RegionRemainder(nreg)   (PRegions[(nreg)]->remainder)
#define RegionWaste(nreg)       (PRegions[(nreg)]->waste)

#define StaticRegionAP(nreg)    (StaticRegions[(nreg)].ap)
#define StaticRegionLimit(nreg) (StaticRegions[(nreg)].alimit)


               /*-------------------------------------
               | adressages, constantes, conversions |
               -------------------------------------*/


#define SmallSizeLimit  (OnePage/2)

#define PageNum(add)    ((ulint)(add) / PageSize)


#define RegionOf(x)             ((ulint)((PageTag(PageNum((x)))).nregion))



                 /*---------------------------------
                 | Acces aux descripteurs de pages |
                 ---------------------------------*/


/* #define Pages                   (gm_state.pages) */
#define NullPage                ((npage_t)0)

#define PageTag(np)             (PageDescs[(np)].tag)


/**********************************************************************
 * Regions standard
 **********************************************************************/

#define STD_HIDDEN_REG  0
#define STD_FREE_REG    1
#define STD_GM_REG      2


/**********************************************************************
 * Zones
 **********************************************************************/


#define HIDDEN_ZONE                     0
#define FREE_ZONE                       1

/* - Zone ``du gc'' - */

#define GC_ZONE                         2

/* - Zones d'allocation - */

#define FROM_FIX_ZONE                   3       /* zone d'allocation */
#define FROM_FIX_OPAQUE_ZONE            4       /* zone d'allocation */
#define FROM_COPY_ZONE                  5       /* zone d'allocation */
#define FROM_COPY_OPAQUE_ZONE           6       /* zone d'allocation */
#define FROM_BIG_ZONE                   7       /* zone d'allocation */
#define FROM_BIG_OPAQUE_ZONE            8       /* zone d'allocation */
#define FOLLOW_ZONE                     9       /* pseudo zone */

/* - zones de promotion - */

#define TO_FIX_ZONE                     10
#define TO_FIX_OPAQUE_ZONE              11
#define TO_FIXED_ZONE                   12
#define TO_FIXED_OPAQUE_ZONE            13
#define TO_COPY_ZONE                    14
#define TO_COPY_OPAQUE_ZONE             15
#define TO_BIG_ZONE                     16
#define TO_BIG_OPAQUE_ZONE              17



/**********************************************************************
 * Modes
 **********************************************************************/

/* - Modes - (po == petits objets; go == grands objets) */

#define NO_MODE                 0
#define W1_MODE                 1       /* po composes de 1 mot     */
#define W2_MODE                 2       /* po composes de 2 mots    */
#define W3_MODE                 3       /* po composes de 3 mots    */
#define W4_MODE                 4       /* po composes de 4 mots    */
#define W8_MODE                 5       /* po composes de 8 mots    */
#define SMALLSIZE_MODE          6       /* po,  taille dans region  */
#define HEADER_MODE             7       /* po,  avec entete         */
#define CSTRING_MODE            8       /* chaine C                 */
#define FOLLOW_MODE             9       /* pseudo-mode              */
#define BIG_MODE               10       /* gros objets              */
#define STRING_MODE            11       /* chaine, taille sur 1 mot */

#define W2_MASK                 (2*WordSize-1)
#define W4_MASK                 (4*WordSize-1)
#define W8_MASK                 (8*WordSize-1)

